/* eslint-disable camelcase */
const INGESTION_CONFIGURATION_PROPERTY = 'ingest';

export default class Client {

  constructor(basePath, httpClient, federateResolver) {
    this.BASE_PATH = basePath;
    this.API_ROOT = `${basePath}/connector_elasticsearch`;
    this.SIM_API = `${basePath}/elasticsearch/_ingest/pipeline/_simulate`;
    this.INDEX_ROOT = `${this.API_ROOT}/_siren/connector/index`;
    this.INGESTION_ROOT = `${this.API_ROOT}/_siren/connector/ingestion`;
    this.DATASOURCE_ROOT = `${this.API_ROOT}/_siren/connector/datasource`;
    this.JOBS_ROOT = `${this.API_ROOT}/_siren/connector/jobs`;
    if (federateResolver.checkApiAvailability('INGESTION_LIST_API_V2')) {
      this.FETCH_INGESTION_CONFIG = `${this.INGESTION_ROOT}/_all?detailed=false&status=true`;
    } else {
      this.FETCH_INGESTION_CONFIG = `${this.INGESTION_ROOT}/_search?status=true`;
    }
    this.httpClient = httpClient;
    this._ingestApiV1 = `${basePath}/ingest/api/v1/`;
    this._neo4jAPI = `${this._ingestApiV1}neo4j`;
  }

  async fetch(url, method, data) {
    if (!method) {
      method = 'GET';
    }
    const response = await this.httpClient({
      method,
      url,
      data
    });
    return response.data;
  }

  async simulatePipeline(data, pipeline) {
    if (data && typeof data !== 'object') {
      try {
        data = JSON.parse(data);
      } catch (e) {
        return Promise.resolve(e.toString());
      }
    }
    let response;
    try {
      pipeline = typeof pipeline === 'object' ? pipeline : JSON.parse(pipeline);
      const docs = [
        {
          _index: 'index',
          _type: '_doc',
          _id: 'id',
          _source: data
        }
      ];
      response = await this.fetch(`${this.SIM_API}`, 'POST',
        {
          pipeline,
          docs
        });
    } catch (e) {
      return e.data ? e.data.error.reason : e.toString();
    }
    return (response.docs[0].doc && response.docs[0].doc._source) || response.docs[0].error;
  }

  _getReasonForError(e) {
    let currentException = e;
    let reason = '';
    while (currentException.caused_by) {
      reason += `${currentException.caused_by.reason} | `;
      currentException = currentException.caused_by;
    }
    return reason.slice(0, -3) || e.reason;
  }

  /**
   * Compute nested fields.
   * Note: Since nested fields do not receive any type, they are not
   * included in fields array
   * @param  {Object} options.result
   * @param  {Array.<String>} options.fields
   * @return {Array.<String>} nestedFields
   */
  _getNestedFields({ result, fields }) {
    const fieldSet = new Set(fields.map(({ field }) => field));
    const nestedFields = [];
    Object.keys(result).forEach(field => {
      if (!fieldSet.has(field)) {
        nestedFields.push(field);
      }
    });
    return nestedFields;
  }

  async fetchSample(datasource, query) {
    let response;
    try {
      response =  await this.fetch(`${this.DATASOURCE_ROOT}/${datasource}/_sample`,
        'POST', { query });
    } catch (e) {
      if (e.status === 404) {
        return `Datasource (${datasource}) Not Found`;
      }
      return this._getReasonForError(e.data.error);
    }
    if (!response.found) {
      return {
        fields: [],
        results: []
      };
    }
    const { types, results } = response;
    const fields = [];
    for (const field of Object.keys(types)) {
      if (types.hasOwnProperty(field)) {
        fields.push({
          field,
          type: types[field]
        });
      }
    }
    const computedResult = {
      results,
      fields
    };
    // Only considering first row of results, as in JDBC: the column count remains constant for the entire ResultSet
    if (results.length > 0 && fields.length < Object.keys(results[0]).length) {
      computedResult.nestedFields = this._getNestedFields({ fields, result: results[0] });
    }
    return computedResult;
  }

  async fetchDefaultQuery(virtualIndexName) {
    const response = await this.fetch(`${this.INDEX_ROOT}/${virtualIndexName}/_default_query`);
    if (response.found) {
      return response.query;
    }
    return '';
  }

  ensureFields(configuration) {
    if (!configuration.transforms) {
      configuration.transforms = [];
    }
    if (!configuration.removed_fields) {
      configuration.removed_fields = [];
    }
    return configuration;
  }

  async fetchConfigurations() {
    const response = await this.fetch(this.FETCH_INGESTION_CONFIG);
    if (response.hits && response.hits.total > 0) {
      return response.hits.hits
        .filter(hit => hit._source.hasOwnProperty(INGESTION_CONFIGURATION_PROPERTY))
        .map(hit => {
          const configuration = ({ ...{ id: hit._id },
            ...(hit._last && { _last: hit._last }),
            ...(hit._run && { _run: hit._run }) });
          for (const key of Object.keys(hit._source.ingest)) {
            if (hit._source.ingest.hasOwnProperty(key)) {
              configuration[key] = hit._source.ingest[key];
            }
          }
          return this.ensureFields(configuration);
        });
    } else {
      return [];
    }
  }

  async fetchConfiguration(id) {
    const response = await this.fetch(`${this.INGESTION_ROOT}/${id}`);
    response._source.ingest.id = response._id;
    return this.ensureFields(response._source.ingest);
  }

  async configurationExist(id) {
    let response;
    try {
      response = await this.fetch(`${this.INGESTION_ROOT}/${id}`);
    } catch (e) {
      response = e.data;
    }
    return response.found;
  }

  /**
   * Cheap ass hack because federate can't take care of it's job logs.
   * @param  {String} jobId
   * @return {Object}
   */
  async _deleteJobLogs(jobId) {
    return await this.fetch(`${this._ingestApiV1}clearJobLogs/${jobId}`, 'DELETE');
  }

  async deleteConfiguration(id) {
    const resp = await this.fetch(`${this.INGESTION_ROOT}/${id}`, 'DELETE');
    this._deleteJobLogs(id);
    return resp;
  }

  filterConfiguration(config) {
    const {
      batch_size,
      datasource,
      description,
      enable_scheduler,
      mapping,
      pipeline,
      pk_field,
      query,
      removed_fields,
      schedule,
      staging_prefix,
      strategy,
      target,
      transforms,
      virtual_index,
      es_credentials,
      ds_credentials
    } = config;
    return {
      ...(batch_size && { batch_size }),
      ...(datasource && { datasource }),
      ...(description && { description }),
      ...{ enable_scheduler },
      ...(mapping && { mapping }),
      ...(pipeline && { pipeline }),
      ...(pk_field && { pk_field }),
      ...(query && { query }),
      ...(removed_fields && { removed_fields }),
      ...(schedule && { schedule }),
      ...(staging_prefix && { staging_prefix }),
      ...(strategy && { strategy }),
      ...(target && { target }),
      ...(transforms && { transforms }),
      ...(virtual_index && { virtual_index }),
      ...(es_credentials && { es_credentials }),
      ...(ds_credentials && { ds_credentials })
    };
  }

  async saveConfiguration(id, configuration) {
    const body = {};
    body[INGESTION_CONFIGURATION_PROPERTY] = this.filterConfiguration(configuration);
    const response = await this.fetch(`${this.INGESTION_ROOT}/${id}`, 'PUT', body);
    return response;
  }

  /**
   * Fetches jobs created in the last 60 minutes.
   */
  async fetchJobs() {
    const response = await this.fetch(`${this.JOBS_ROOT}/_search`);
    if (response.hits.total === 0) {
      return [];
    }

    const jobs = response.hits.hits.map(hit => {
      const job = {
        id: hit._id
      };
      for (const key of Object.keys(hit._source.ingestion)) {
        if (key === 'id') {
          job.configuration_id = hit._source.ingestion.id;
          continue;
        }
        if (hit._source.ingestion.hasOwnProperty(key)) {
          job[key] = hit._source.ingestion[key];
        }
      }
      return job;
    });

    return jobs;
  }

  async abortJob(id, type = 'ingestion') {
    const response = await this.fetch(
      `${this.BASE_PATH}/api/console/proxy?path=_siren%2Fconnector%2Fjobs%2F${type}%2F${id}%2F_abort&method=POST`,
      'POST');
    return response; // TODO: response not checked at this time as there are pending changes in the format.
  }

  /**
   * Validates an ingestion configuration.
   *
   * @param id - The id of the ingestion configuration to run.
   * @returns true if the configuration is valid.
   */
  async validateConfiguration(id) {
    const response = await this.fetch(`${this.INGESTION_ROOT}/${id}/_validate`, 'POST');
    if (!response.valid) {
      // TODO: verify that an exception is always thrown by the plugin if the configuration is not valid.
      throw new Error('Invalid configuration.');
    }
    return true;
  }

  /**
   * Runs an ingestion configuration.
   *
   * @param id - The id of the ingestion configuration to run.
   * @returns the identifier of the job.
   */
  async runConfiguration(id) {
    const response = await this.fetch(`${this.INGESTION_ROOT}/${id}/_run`, 'POST');
    return response.jobId;
  }

  async getPendingJobs() {
    return await this.fetch(`${this._neo4jAPI}/pendingJobs`);
  }

  async fetchJobInfo(jobId) {
    const jobData = await this.fetch(`${this._neo4jAPI}/job/${jobId}`);
    return {
      jobId,
      jobData,
      jobs_created: Object.keys(jobData).map(key => jobData[key].jobId)
    };
  }
}