import cloneDeep from 'lodash.clonedeep';
import ConfigurationsHeader from './header';
import Section from '../section';
import React, { Fragment } from 'react';
import { Prompt } from 'react-router';
import Beforeunload from 'react-beforeunload';
import NavigationPrompt from 'react-router-navigation-prompt';
import TransformsEditor from './transforms_editor/transforms_editor';
import MappingTable from '../xlsx/components/mapping_table';
import Scheduler from './scheduler';
import {
  getDelayedTrigger
} from '../xlsx/services/helper.js';
import {
  getUpdatedMappingTableState
} from '../xlsx/services/services';
import {
  buildTransformationFromMapping,
  getCustomTransformations
} from './transformation_helper';
import { resolve } from '../../utils';
import 'brace/theme/github';
import 'brace/mode/json';
import 'brace/mode/sql';
import 'brace/ext/language_tools';
import { PipelineDefinition } from '../../shared_components';
import {
  EuiAccordion,
  EuiComboBox,
  EuiStepsHorizontal,
  EuiFieldNumber,
  EuiFieldText,
  EuiFieldPassword,
  EuiCodeEditor,
  EuiCallOut,
  EuiTitle,
  EuiForm,
  EuiFormRow,
  EuiPanel,
  EuiSpacer,
  EuiSelect,
  EuiSuperSelect,
  EuiConfirmModal,
  EuiOverlayMask,
  EUI_MODAL_CONFIRM_BUTTON,
  EuiTabbedContent,
  EuiText,
  EuiTextArea,
  EuiFlexGroup,
  EuiFlexItem,
  EuiButton,
  EuiBasicTable
} from '@elastic/eui';
import { styles } from './styles.js';


const STEP_INPUT = 'step-input';
const STEP_TARGET = 'step-target';
const STEP_TRANSFORM = 'step-transform';
const STEP_PIPELINE = 'step-pipeline';
const STEP_SCHEDULE = 'step-schedule';
const ID_SPLITTER = ':';
const SOURCE_TYPE = {
  VI: 'virtual-index',
  DS: 'datasource'
};
const DEFAULT_QUERY = 'SELECT * FROM events';
const pipelineMethodList = [
  {
    value: 'skip',
    inputDisplay: 'Skip Transform'
  },
  {
    value: 'elasticsearch',
    inputDisplay: 'Elasticsearch ingestion pipeline method'
  },
  {
    value: 'openrefine',
    disabled: true,
    inputDisplay: 'Openrefine pipeline (Not yet available)'
  }
];
const INDEXING_STRATEGY = {
  incremental: 'INCREMENTAL',
  replace: 'REPLACE'
};
const INDEXING_OPTIONS =  [
  {
    value: INDEXING_STRATEGY.incremental,
    inputDisplay: 'Incremental',
    dropdownDisplay: (
      <Fragment>
        <strong>Incremental</strong>
        <EuiSpacer size="xs" />
        <EuiText size="s" color="subdued">
          <p className="euiTextColor--subdued">
            The index is created if it does not exists. The reflected records are inserted or updated in place.<br />
            <b>Note</b>: You must define a unique primary key, otherwise you will get duplicate records.
          </p>
        </EuiText>
      </Fragment>
    ),
  },
  {
    value: INDEXING_STRATEGY.replace,
    inputDisplay: 'Replace',
    dropdownDisplay: (
      <Fragment>
        <strong>Replace</strong>
        <EuiSpacer size="xs" />
        <EuiText size="s" color="subdued">
          <p className="euiTextColor--subdued">
            Data is reflected to a staging index and hot swapped upon completion.<br />
            If anything goes wrong the target index is untouched.
          </p>
        </EuiText>
      </Fragment>
    ),
  }
];


export default class ConfigurationForm extends Section {

  constructor(props) {
    document.title = 'Ingestion Configuration - Siren';
    super(props);
    this.notifier = props.createNotifier('Ingestion configurations');
    const id = this.props.match.params.id;
    this.state = this.initialState(id);
    this.virtualIndexPrimaryKeys = {};

    this.updateSource = this.updateSource.bind(this);
    this.updatePipelineDef = this.updatePipelineDef.bind(this);
    this.onChangeRemovedFields = this.onChangeRemovedFields.bind(this);
    this.onFieldRemovalFromMappingTable = this.onFieldRemovalFromMappingTable.bind(this);
    this.testPipeline = this.testPipeline.bind(this);
    this.pipelineDefDelayedTrigger = getDelayedTrigger();
  }

  initialState(id) {
    const isNew = !!!id;
    const steps = [
      {
        id: STEP_INPUT,
        name: 'Input data',
        disabled: false
      },
      {
        id: STEP_PIPELINE,
        name: '(opt) Transform Pipeline',
        disabled: isNew
      },
      {
        id: STEP_TRANSFORM,
        name: 'Mapping',
        disabled: isNew
      },
      {
        id: STEP_TARGET,
        name: 'Target index',
        disabled: isNew
      },
      {
        id: STEP_SCHEDULE,
        name: 'Schedule',
        disabled: isNew
      },
    ];
    return cloneDeep({
      id: this.props.match.params.id,
      isNew: isNew,
      dirty: false,
      valid: false,
      steps,
      hasNextStep: true,
      mappingData: {
        items: [],
        examples: [],
        removedItems: []
      },
      customTransforms: [],
      tested: false,
      configuration: {
        batch_size: null,
        description: '',
        datasource: null,
        virtual_index: null,
        pipelineMethod: pipelineMethodList[0].value,
        pipeline: '',
        pipelineInput: 0,
        pipelineResult: '',
        query: DEFAULT_QUERY,
        target: '',
        strategy: INDEXING_STRATEGY.replace,
        staging_prefix: '',
        schedule: '',
        enable_scheduler: false,
        mapping: {},
        pk_field: '',
        removed_fields: [],
        transforms: [],
        es_credentials: {
          username: '',
          password: ''
        },
        ds_credentials: {
          username: '',
          password: ''
        }
      },
      datasourceDrivers: {},
      selectedSource: '',
      displayNew: true,
      currentStep: STEP_INPUT
    });
  }

  autoFill(newState, oldState) {
    if (oldState.isNew) {
      const getAutoFillName = state => {
        return `${state.configuration.virtual_index || state.configuration.datasource}.reflection`;
      };
      const getAutoFillDescription = state => {
        return `Reflection from ${state.configuration.virtual_index || state.configuration.datasource}.`;
      };
      if (!oldState.id || oldState.id === getAutoFillName(oldState)) {
        newState.id = getAutoFillName(newState);
      }
      if (!oldState.configuration.description || oldState.configuration.description === getAutoFillDescription(oldState)) {
        newState.configuration.description = getAutoFillDescription(newState);
      }
      if (!oldState.configuration.target || oldState.configuration.target === getAutoFillName(oldState)) {
        newState.configuration.target = getAutoFillName(newState);
      }
      if (newState.configuration.virtual_index) {
        newState.configuration.pk_field = this.virtualIndexPrimaryKeys[newState.configuration.virtual_index];
      }
      newState.dirty = false;
    }
    return newState;
  }

  updateSource(selectedSource) {
    const [id, type, optional] = selectedSource.split(ID_SPLITTER);
    const tested = false;
    if (type === SOURCE_TYPE.DS) {
      this.setState(state => {
        const configuration = cloneDeep(state.configuration);
        configuration.virtual_index = null;
        configuration.datasource = id;
        return this.autoFill({
          configuration,
          selectedSource,
          tested
        }, state);
      });
    } else if (type === SOURCE_TYPE.VI) {
      this.props.client.fetchDefaultQuery(id).then((query) => {
        if (query) {
          this.setState(state => {
            const configuration = state.configuration;
            configuration.query = query;
            return {
              configuration,
              tested
            };
          });
        }
      });
      this.setState(state => {
        const configuration = cloneDeep(state.configuration);
        configuration.datasource = optional;
        configuration.virtual_index = id;
        return this.autoFill({
          configuration,
          selectedSource,
          tested
        }, state);
      });
    }
  }

  initDatasourceList(initializeDataSource) {
    let dataSourceList = [];
    const dropdownDisplay = function (option, type) {
      return (
        <Fragment>
          <strong>{option._id}</strong>
          <EuiSpacer size="xs" />
          <EuiText size="s" color="subdued">
            <p className="euiTextColor--subdued">
              {type} {type === SOURCE_TYPE.VI ? <b>(source: {option._source.datasource})</b> : ''}
            </p>
          </EuiText>
        </Fragment>
      );
    };
    const buildSourceId = function (source, type) {
      return `${source._id}${ID_SPLITTER}${type}${ID_SPLITTER}${source._source.datasource}`;
    };
    const buildList = function (resp, dataSourceList, type, datasourceDrivers) {
      return dataSourceList.concat(resp.map((e) => {
        if (type === SOURCE_TYPE.DS) {
          datasourceDrivers[e._id] = e._source.jdbc.driver;
        }
        return {
          value: buildSourceId(e, type),
          inputDisplay: e._id,
          dropdownDisplay: dropdownDisplay(e, type)
        };
      }));
    };
    const buildSourceFromConfig = function (config) {
      const id = config.virtual_index || config.datasource;
      const optional = (config.virtual_index || undefined) && config.datasource;
      return `${id}${ID_SPLITTER}${config.virtual_index ? SOURCE_TYPE.VI : SOURCE_TYPE.DS}${ID_SPLITTER}${optional}`;
    };
    const datasourceDrivers = {};
    this.props.jdbcDatasources.list().then((resp) => {
      dataSourceList = buildList(resp, dataSourceList, SOURCE_TYPE.DS, datasourceDrivers);
      this.props.jdbcDatasources.listVirtualIndices().then((resp) => {
        dataSourceList = buildList(resp, dataSourceList, SOURCE_TYPE.VI);
        this.virtualIndexPrimaryKeys = resp.reduce((acc, ele) => {
          acc[ele._id] = ele._source.key;
          return acc;
        }, {});
        const sourceId = buildSourceFromConfig(this.state.configuration);
        const found = sourceId && dataSourceList.find((e) => sourceId === e.value);
        !initializeDataSource && !found &&
          this.notifier.error(`Datasource ${this.state.configuration.datasource} Not Found!`);
        if (initializeDataSource || !found) {
          if (found && found.value || dataSourceList[0]) {
            this.updateSource(found ? found.value : dataSourceList[0].value);
          }
          this.setState({
            dataSourceList,
            datasourceDrivers
          });
        }
        else {
          this.setState({
            dataSourceList,
            datasourceDrivers,
            selectedSource: sourceId
          });
        }
      });
    });

  }

  async onMount() {
    if (this.state.id) {
      try {
        this.initDatasourceList(false);
        const serverConfiguration = await this.props.client.fetchConfiguration(this.state.id);
        this.setState(state => {
          const configuration = state.configuration;
          const localMapping = {};
          if (!!serverConfiguration.mapping) {
            for (const key of Object.keys(serverConfiguration.mapping)) {
              if (serverConfiguration.mapping.hasOwnProperty(key)) {
                localMapping[key] = serverConfiguration.mapping[key];
              }
            }
          }
          configuration.mapping = localMapping;
          if (serverConfiguration.pipeline) {
            serverConfiguration.pipeline = typeof serverConfiguration.pipeline !== 'string' ?
              JSON.stringify(serverConfiguration.pipeline, null, 2) : serverConfiguration.pipeline;
            if (!serverConfiguration.pipelineMethod) {
              serverConfiguration.pipelineMethod = pipelineMethodList[1].value;
            }
          } else {
            serverConfiguration.pipelineMethod = pipelineMethodList[0].value;
          }
          for (const key of Object.keys(configuration)) {
            if (!configuration.hasOwnProperty(key)) {
              continue;
            }
            if (key === 'es_credentials' || key === 'ds_credentials') {
              const serverCredentials = serverConfiguration[key];
              const localCredentials = {};
              if (serverCredentials) {
                localCredentials.username = serverCredentials.username;
                localCredentials.password = '';
              } else {
                localCredentials.username = '';
                localCredentials.password = '';
              }
              configuration[key] = localCredentials;
            }
            else if (key === 'mapping') {
              continue;
            }
            else if (key === 'transforms') {
              const serverTransforms = serverConfiguration.transforms.map(transform => {
                // TODO: for simplicity the input is expected to be a sequence of comma separated source fields,
                // but for each transform type there should probably be a different UI (e.g. two text inputs for a geo_point transform).
                const localTransform = {};
                if (transform.input) {
                  localTransform.input = transform.input.map(def => def.source).join(',');
                }
                if (transform.transform) {
                  localTransform.transform = transform.transform;
                } else {
                  localTransform.transform = '';
                }
                if (transform.mapping) {
                  localTransform.mapping = JSON.stringify(transform.mapping);
                  localTransform.mappingObject = transform.mapping;
                } else {
                  localTransform.mapping = '{}';
                  localTransform.mappingObject = {};
                }
                localTransform.mappingOption = '';
                if (localTransform.transform === '') {
                  localTransform.transform = 'copy';
                }
                localTransform.output = transform.output;
                return localTransform;
              });
              configuration.transforms = serverTransforms;
            } else {
              let stateValue = serverConfiguration[key];
              if (typeof stateValue === 'undefined' || stateValue === null) {
                stateValue = '';
              }
              configuration[key] = stateValue;
            }
          }
          this.testQuery(configuration, true);
          return {
            dirty: false,
            steps: this.updateSteps(0),
            configuration
          };
        });
      } catch (e) {
        this.notifier.error(e);
      }
    } else {
      this.initDatasourceList(true);
    }
    return;
  }

  async componentDidMount() {
    await this.onMount();
  }

  renderTitle() {
    if (this.state.id) {
      return (
        <EuiTitle size="m">
          <h1>{this.state.id}</h1>
        </EuiTitle>
      );
    } else {
      return (
        <EuiTitle size="m">
          <h1>Create configuration</h1>
        </EuiTitle>
      );
    }
  }

  skipPipeline() {
    return this.state.configuration.pipelineMethod === pipelineMethodList[0].value;
  }

  updatePipelineDef(param) {
    if (!param.trigger) {
      param.trigger = true;
      this.pipelineDefDelayedTrigger(this.updatePipelineDef, 200, param);
      return;
    }
    this.changeFieldValue(param.value, 'pipeline');
  }

  allowNext() {
    if (this.state.currentStep === STEP_INPUT) {
      return this.state.id && (this.state.tested || !this.state.dirty);
    } else if (this.state.currentStep === STEP_PIPELINE) {
      return this.skipPipeline() || (this.state.tested || !this.state.dirty);
    }
    return true;
  }

  changeFieldValue = (value, fieldName) => {
    if (fieldName === 'id') {
      this.setState({ id: value, dirty: true });
      return;
    } else if (fieldName === 'query') {
      this.setState({ tested: false });
    }
    this.setState(state => {
      const configuration = state.configuration;
      configuration[fieldName] = value;
      return {
        dirty: true,
        configuration
      };
    });
  }

  onChangeMapping(items) {
    this.setState(state => {
      const mappingData = state.mappingData;
      mappingData[items] = items;
      return {
        dirty: true,
        mappingData
      };
    });
  }

  onCredentialsChange(credentials, fieldName, value) {
    this.setState(state => {
      const configuration = state.configuration;
      configuration[credentials][fieldName] = value;
      return {
        dirty: true,
        configuration
      };
    });
  }

  onTextChange(event, fieldName) {
    this.changeFieldValue(event.target.value, fieldName);
  }

  onTransformsChange(newTransforms) {
    this.setState(state => {
      const customTransforms = state.customTransforms;

      for (let i = 0; i < newTransforms.length; i++) {
        if (customTransforms[i]) {
          customTransforms[i].input = newTransforms[i].input;
          customTransforms[i].transform = newTransforms[i].transform;
          customTransforms[i].output = newTransforms[i].output;
          customTransforms[i].mapping = newTransforms[i].mapping;
        } else {
          customTransforms.push(newTransforms[i]);
        }
      }
      customTransforms.splice(newTransforms.length);

      return {
        dirty: true,
        customTransforms
      };
    });
  }

  async testQuery(configuration = this.state.configuration, fetchCustomTransforms = false) {
    if (!configuration.datasource) {
      this.notifier.warning('Please specify a datasource or create a new one [Management->Datasources]');
      return;
    }
    this.setState({ loadingPreview: true });
    try {
      const response = await this.props.client.fetchSample(configuration.datasource,
        configuration.query);
      if (!response.fields) {
        throw new Error(response.toString());
      }
      const mappingData = this.buildMappingData(response, configuration.transforms);
      let customTransforms = this.state.customTransforms;
      if (fetchCustomTransforms) {
        customTransforms = getCustomTransformations(configuration.transforms, mappingData.items);
      }
      this.setState({ customTransforms, preview: response, loadingPreview: false,
        mappingData, tested: true });
    } catch (e) {
      this.setState({ loadingPreview: false });
      this.notifier.error(e);
    }
  }

  buildMappingData(data, existingTransformations = []) {
    const transformationMap = existingTransformations.reduce((acc, ele) => {
      let advJson = {};
      let mapping = {};
      try {
        mapping = JSON.parse(ele.mapping);
        advJson = cloneDeep(mapping);
      } catch (e) {
        this.notifier.error(e);
      }
      delete advJson.type;
      advJson = Object.keys(advJson).length !== 0 ? advJson : null;
      acc[ele.output] = {
        type: mapping && mapping.type,
        advJson,
        useAdv: advJson ? true : false,
        name: ele.output,
        sanitizedFieldName: ele.output
      };
      return acc;
    }, {});
    const items = data.fields.map((ele) => {
      if (transformationMap[ele.field]) {
        return transformationMap[ele.field];
      }
      return {
        name: ele.field,
        sanitizedFieldName: ele.field,
        type: ele.type
      };
    });
    const examples = data.results;
    return {
      items,
      examples,
      removedItems: this.state.configuration.removed_fields.reduce((acc, ele) => {
        acc.push({
          label: ele
        });
        return acc;
      }, [])
    };
  }

  onFieldRemovalFromMappingTable(removedItem) {
    const newRemovalArray = this.state.mappingData.removedItems.concat({
      label: removedItem.sanitizedFieldName
    });
    this.onChangeRemovedFields(newRemovalArray);
  }

  onChangeRemovedFields(newRemovedFieldsArray) {
    this.setState((state) => {
      const mappingData = state.mappingData;
      mappingData.removedItems = newRemovedFieldsArray;
      return {
        dirty: true,
        mappingData
      };
    });
  }

  async testPipeline(paramData) {
    const response = await this.props.client.simulatePipeline(
      paramData || this.state.preview.results[this.state.configuration.pipelineInput], this.state.configuration.pipeline);
    this.setState({ tested: true });
    return response;
  }

  renderStepInput() {
    let dataSourceSelector;
    if (this.state.dataSourceList) {
      dataSourceSelector = (
        <EuiSuperSelect
          valueOfSelected={this.state.selectedSource}
          options={this.state.dataSourceList}
          onChange={this.updateSource}
        />
      );
    } else {
      dataSourceSelector = <b>Loading...</b>;
    }
    const queryTabs = [
      {
        id: 'query',
        name: 'Datasource Query',
        content: (
          <div>
            <EuiCodeEditor
              mode="sql"
              theme="github"
              width="100%"
              height="180px"
              value={this.state.configuration.query}
              onChange={value => this.changeFieldValue(value, 'query')}
              setOptions={{
                fontSize: '12px',
                enableBasicAutocompletion: true
              }}
            />
          </div>
        )
      },
      {
        id: 'help',
        name: 'Help',
        content: (
          <EuiPanel paddingSize="l" style={styles.helpPanel}>
            <p>
              The query defined in the reflection configuration is written in the
              datasource language.
            </p>
            <p style={styles.helpTextListItem}>
              The query can be written using mustache and the following variables are
              provided, if applicable, when converting the query to a string:
            </p>
            <ul style={styles.marginLeft14}>
              <li>
                <p style={styles.helpTextListItem}>
                  <font color="#353744"><font face="Consolas, serif"><b>max_primary_key</b></font></font>
                    :
                      the maximum value of the primary key in Elasticsearch, available if
                      the primary key is numeric.
                </p>
              </li>
              <li>
                <p style={styles.helpTextListItem}>
                  <font color="#353744"><font face="Consolas, serif"><b>last_record_timestamp</b></font></font>
                  :
                    the UTC timestamp at which the last record was successfully
                    processed by a reflection job.
                </p>
              </li>
              <li>
                <p style={styles.helpTextListItem}>
                  <font color="#353744"><font face="Consolas, serif"><b>last_record</b></font></font>
                  :
                    an array with the scalar values in the last record that was
                    successfully processed by the reflection job.
                </p>
              </li>
            </ul>
            <p style={styles.helpTextListItem}>
              <b>Example:</b>
            </p>
            <p align="LEFT" style={styles.marginLeft11}><a name="_GoBack" />
              <font color="#5a5a5a">
                SELECT * FROM &quot;Player&quot;&nbsp;{'{{#max_primary_key}}'}
              </font>
            </p>
            <p align="LEFT" style={styles.marginLeft22}>
              <font color="#5a5a5a">
              WHERE &quot;NAME&quot;&gt;{'\'{{max_primary_key}}\''}{'{{/max_primary_key}}'}
              </font>
            </p>
          </EuiPanel>
        )
      }
    ];
    const isNeo4jSelected = this.state.datasourceDrivers[this.state.configuration.datasource] &&
      this.state.datasourceDrivers[this.state.configuration.datasource].toLowerCase().includes('neo4j');
    return (
      <EuiForm>
        <EuiFlexGroup>
          <EuiFlexItem>
            <EuiFormRow
              className={this.state.isNew ? '' : 'hidden'}
              label="Name"
              fullWidth
              helpText="The name of this reflection configuration"
            >
              <EuiFieldText
                data-test-subj="ingest-job-name"
                value={this.state.id}
                onChange={e => this.onTextChange(e, 'id')}
              />
            </EuiFormRow>
            <EuiFormRow
              label="Description"
              fullWidth
              helpText="The description of this reflection configuration"
            >
              <EuiTextArea
                rows={4}
                value={this.state.configuration.description}
                onChange={e => this.onTextChange(e, 'description')}
              />
            </EuiFormRow>
          </EuiFlexItem>
          <EuiFlexItem>
            <EuiFormRow
              label="Datasource"
              fullWidth
              helpText="The datasource against which the query will be executed."
            >
              {dataSourceSelector}
            </EuiFormRow>
            {
              this.state.isNew && isNeo4jSelected ?
                <EuiFormRow>
                  <EuiButton
                    key="neo4jWizard"
                    iconType="indexMapping"
                    href={resolve(`/neo4j-wizard/${this.state.configuration.datasource}`)}
                  >
                    Use Neo4j Importer
                  </EuiButton>
                </EuiFormRow>
                : ''
            }
          </EuiFlexItem>
        </EuiFlexGroup>
        <EuiFormRow
          label="Query"
          fullWidth
          helpText="The query that will fetch the data."
        >
          <EuiTabbedContent
            tabs={queryTabs}
          />
        </EuiFormRow>
        <EuiFormRow fullWidth>
          <EuiFlexGroup justifyContent="spaceBetween">
            <EuiFlexItem grow={false}>
              <EuiButton
                data-test-subj="test-connection"
                color="primary"
                fill
                isDisabled={this.state.loadingPreview}
                iconType="check"
                onClick={() => this.testQuery()}
              >
                Test
              </EuiButton>
            </EuiFlexItem>
            <EuiFlexItem
              grow={false}
            >
              {this.renderNext()}
            </EuiFlexItem>
          </EuiFlexGroup>
        </EuiFormRow>
        {this.renderPreview()}
      </EuiForm>
    );
  }

  renderPreview() {
    const preview = this.state.preview;
    if (typeof preview === 'object') {
      const width = `${(preview.fields.length * 100)}px`;
      let warning = '';
      let  columns = preview.fields.map((ele) => {
        return {
          field: ele.field,
          name: ele.field,
          truncateText: true
        };
      });
      if (preview.nestedFields && preview.nestedFields.length > 0) {
        columns = columns.concat(preview.nestedFields.map(field => {
          return {
            field: field,
            name: field,
            render: value => {
              return value && JSON.stringify(value);
            },
            truncateText: true
          };
        }));
        warning = (
          <EuiCallOut
            size="s"
            title="Nested fields detected!"
            color="warning"
            style={{ maxWidth: '100em' }}
            iconType="alert"
          >
            <p>
              Nested fields: <b>{preview.nestedFields.toString()}</b> would not be auto-mapped by Federate.
              You must add mapping for these fields manually if you do not wish to rely on Elasticsearch to detect mapping
              or a mapping conflict occurs.
            </p>
          </EuiCallOut>
        );
      }
      return (
        <Fragment>
          {warning}
          <EuiSpacer size="xs" />
          <EuiFormRow fullWidth>
            <div style={{ overflowX: 'scroll', maxWidth: '100em', border: 'groove rgba(192, 192, 192, 0.52)' }}>
              <EuiBasicTable
                data-test-subj="test-sample-results"
                items={preview.results}
                columns={columns}
                style={{ minWidth: width }}
              />
            </div>
          </EuiFormRow>
        </Fragment>
      );
    } else if (typeof preview === 'string') {
      return (
        <EuiFormRow fullWidth>
          <EuiCallOut
            style={{ width: '100em' }}
            title="Sorry, there was an error"
            color="danger"
            iconType="cross"
          >
            <p>
              {preview}
            </p>
          </EuiCallOut>
        </EuiFormRow>
      );
    }
    return '';
  }

  renderHeader() {
    return (
      <ConfigurationsHeader
        sections={this.props.sections}
        currentSection={this.props.currentSection}
        displaySave={true}
        displayNew={false}
        onSaveClick={(e) => this.onSaveClick(e)}
      />
    );
  }
  //Without FormRow
  renderNext() {
    return (
      <EuiButton
        data-test-subj="next-btn"
        color="primary"
        fill
        isDisabled={!this.allowNext()}
        iconType="arrowRight"
        onClick={() => this.advance()}
      >
        Next
      </EuiButton>
    );
  }

  renderNextButton() {
    return (
      <EuiFormRow hasEmptyLabelSpace fullWidth className={this.state.hasNextStep ? '' : 'hidden'}>
        <EuiFlexGroup justifyContent="flexEnd">
          <EuiFlexItem grow={false}>
            {this.renderNext()}
          </EuiFlexItem>
        </EuiFlexGroup>
      </EuiFormRow>
    );
  }

  validateAndCleanseBeforeSave(configuration) {
    // Check primary key exists, if not then reset to ''
    if (configuration.pk_field && this.state.mappingData.items.findIndex((ele) => ele.sanitizedFieldName === configuration.pk_field) < 0) {
      if (!this.state.steps[this.state.steps.findIndex(ele => ele.id === STEP_TARGET)].disabled) {
        this.notifier.warning(`Primary key: ${configuration.pk_field} cleared as it does not exist`);
      }
      configuration.pk_field = '';
    }
  }

  async saveAndStartNow() {
    await this.onSaveClick();
    await this.props.client.runConfiguration(this.state.id);
    this.notifier.info(`Job: ${this.state.id} triggered successfully.`);
  }

  async onSaveClick() {
    const isDirty = this.state.dirty;
    try {
      if (!this.state.id) {
        throw 'Please define a name for the configuration';
      }
      isDirty && this.setState({ dirty: false });
      const body = cloneDeep(this.state.configuration);
      this.validateAndCleanseBeforeSave(body);
      if (this.skipPipeline()) {
        delete body.pipeline;
      } else {
        try {
          body.pipeline = typeof body.pipeline === 'string' ? JSON.parse(body.pipeline) : body.pipeline;
        } catch (e) {
          throw `Please check your pipeline definition. ${e}`;
        }
      }
      delete body.pipelineMethod;
      delete body.pipelineResult;
      delete body.pipelineInput;
      body.enable_scheduler = body.enable_scheduler && !!body.schedule;
      let mapping;
      if (this.state.currentStep === STEP_TRANSFORM) {
        mapping = {
          items: getUpdatedMappingTableState(this.state.mappingData.items, this.state.mappingData.removedItems),
          removedItems: this.state.mappingData.removedItems
        };
      } else {
        mapping = this.state.mappingData;
      }
      body.removed_fields = this.state.mappingData.removedItems.reduce((acc, ele) => {
        acc.push(ele.label);
        return acc;
      }, []);
      body.transforms = buildTransformationFromMapping(mapping, cloneDeep(this.state.customTransforms));
      // Currently being handled by buildTransformationFromMapping, commented here for bugfixes and reference
      // for (const transform of body.transforms) {
      //   if (transform.transform === 'copy') {
      //     delete transform.transform;
      //   }
      //   delete transform.mappingOption;
      //   delete transform.mappingObject;
      //   delete transform.valid;

      //   transform.mapping = JSON.parse(transform.mapping);
      //   // TODO: will have to be adapted to the UI to select multiple fields.
      //   // Input is forced on `source` (the resultset) at this time
      //   transform.input = transform.input.split(',').map(i => ({ source: i }));
      // }
      for (const credentials of ['es_credentials', 'ds_credentials']) {
        if (body[credentials].password === '') {
          delete body[credentials].password;
        }
        if (body[credentials].username === '') {
          delete body[credentials];
        }
      }
      if (this.state.isNew) {
        const alreadyExists = await this.props.client.configurationExist(this.state.id);
        if (alreadyExists) {
          throw 'Duplicate Configuration! Configuration with the same ID already exists,' +
            ' please change the ID or edit existing Configuration';
        }
        await this.props.client.saveConfiguration(this.state.id, body);
      } else {
        await this.props.client.saveConfiguration(this.state.id, body);
      }
      this.notifier.info('Configuration saved successfully.');
      this.props.history.push('/configurations');
    } catch (e) {
      !isDirty && this.setState({ dirty: true });
      this.notifier.error(e);
    }
  }

  onBeforeUnload() {
    return 'Are you sure? Your changes in current configuration will be lost if you leave the page.';
  }

  renderModal() {
    // TODO: disabled for the moment as it seems affected by https://github.com/ReactTraining/react-router/issues/5405;
    // probably best to have the backend provide support for the BrowserRouter as HashRouter is pretty flakey.
    return [];
    return (
      <NavigationPrompt when={this.state.dirty}>
        {({ onConfirm, onCancel }) => {
          return (
            <EuiOverlayMask>
              <EuiConfirmModal
                title="Are you sure?"
                onCancel={() => onCancel()}
                onConfirm={() => onConfirm()}
                cancelButtonText="Cancel"
                confirmButtonText="Yes"
                defaultFocusedButton={EUI_MODAL_CONFIRM_BUTTON}
              >
                <p>Your changes will be lost.</p>
              </EuiConfirmModal>
            </EuiOverlayMask>
          );
        }}
      </NavigationPrompt>
    );
  }

  renderSteps() {
    const tabs = this.state.steps.map((step) => {
      return {
        onClick: () => this.goToStep(step.id),
        isSelected: step.id === this.state.currentStep,
        isComplete: step.isComplete,
        title: step.name,
        disabled: step.disabled
      };
    });
    return (
      <EuiStepsHorizontal
        steps={tabs}
      />
    );
  }

  updateSteps(index) {
    return this.state.steps.map((ele, curr) => {
      if (curr < index) {
        ele.isComplete = true;
      } else {
        ele.isComplete = !ele.disabled || false;
      }
      return ele;
    });
  }

  goToStep(stepId) {
    // TODO: validate the current step and put an error icon if needed but do not prevent navigation.
    if (this.state.loadingPreview) {
      setTimeout(() => this.goToStep(stepId), 200);
      return;
    }
    this.stepChangeHook(this.state.currentStep);
    const index = this.state.steps.findIndex(s => s.id === stepId);
    const hasNextStep = !(index === this.state.steps.length - 1);
    this.setState({ currentStep: stepId, hasNextStep: hasNextStep, steps: this.updateSteps(index) });
  }

  stepChangeHook(currentStep) {
    if (currentStep === STEP_TRANSFORM) {
      const items = getUpdatedMappingTableState(this.state.mappingData.items, this.state.mappingData.removedItems);
      this.setState({
        mappingData: {
          items,
          examples: this.state.mappingData.examples,
          removedItems: this.state.mappingData.removedItems
        }
      });
    }
  }

  advance() {
    // TODO: validate the current step and disable others if needed
    this.stepChangeHook(this.state.currentStep);
    this.setState(state => {
      const steps = state.steps;
      const nextStep = state.steps.findIndex(s => s.id === state.currentStep) + 1;
      steps[nextStep - 1].isComplete = true;
      steps[nextStep].disabled = false;
      let hasNextStep = true;
      if (nextStep === state.steps.length - 1) {
        hasNextStep = false;
      }
      return {
        currentStep: state.steps[nextStep].id,
        tested: false,
        steps,
        hasNextStep
      };
    });
  }

  renderPipeline() {
    if (!this.state.preview) {
      return (
        <EuiCallOut
          size="s"
          title="Your Datasource/Query doesn't seem to be working, please correct it and run 'Test' to use this section"
          color="warning"
          iconType="alert"
        />
      );
    }
    return (
      <EuiFormRow
        label="Transform (Optional)"
        fullWidth
      >
        <PipelineDefinition
          documents={this.state.preview.results}
          configuration={this.state.configuration}
          updatePipelineDef={this.updatePipelineDef}
          onConfigChange={this.changeFieldValue}
          testPipeline={this.testPipeline}
        />
      </EuiFormRow>
    );
  }

  renderStepTarget() {
    const primaryKeyOptions = [{ value: '', text: 'Auto-Generate Primary Key.' }].concat(
      this.state.mappingData.items.map((item) => {
        return {
          value: item.sanitizedFieldName,
          text: item.name
        };
      })
    );
    return (
      <EuiForm>

        <EuiSpacer size="m"/>

        <EuiFormRow
          label="Target index"
          fullWidth
          helpText="The target Elasticsearch index."
        >
          <EuiFieldText
            data-test-subj="ingest-target-index"
            value={this.state.configuration.target}
            onChange={e => this.onTextChange(e, 'target')}
          />
        </EuiFormRow>

        <EuiFormRow
          label="Primary key"
          fullWidth
          helpText="(Optional) The field that will be used to populate the document _id in Elasticsearch."
        >
          <EuiSelect
            data-test-subj="ingest-primary-key"
            value={this.state.configuration.pk_field}
            options={primaryKeyOptions}
            onChange={e => this.onTextChange(e, 'pk_field')}
          />
        </EuiFormRow>

        <EuiFormRow
          label="Target indexing options"
          fullWidth
        >
          <EuiSuperSelect
            data-test-subj="ingest-index-options"
            options={INDEXING_OPTIONS}
            valueOfSelected={this.state.configuration.strategy}
            onChange={e => this.changeFieldValue(e, 'strategy')}
            itemLayoutAlign="top"
            hasDividers
          />
        </EuiFormRow>

        <EuiFormRow
          className={this.state.configuration.strategy === INDEXING_STRATEGY.replace ? '' : 'hidden'}
          label="Staging Index Prefix"
          fullWidth
          helpText="(Optional) Staging index prefix for Elasticsearch."
        >
          <EuiFieldText
            data-test-subj="ingest-stg-index-prefix"
            value={this.state.configuration.staging_prefix}
            onChange={e => this.onTextChange(e, 'staging_prefix')}
          />
        </EuiFormRow>

        <EuiFormRow
          label="Batch Size"
          fullWidth
          helpText="(Optional) Batch size (overriding the default global value)"
        >
          <EuiFieldNumber
            data-test-subj="ingest-batch-size"
            value={this.state.configuration.batch_size}
            placeholder={100}
            onChange={e => this.changeFieldValue(Number(e.target.value), 'batch_size')}
          />
        </EuiFormRow>

        <EuiAccordion
          id="custom_credentials"
          buttonContent="Custom Credentials"
        >
          <EuiFormRow
            label="Elasticsearch username"
            fullWidth
            helpText="(Optional) The username of an Elasticsearch user with full access to the target index."
          >
            <EuiFieldText
              data-test-subj="ingest-es-credentials-username"
              value={this.state.configuration.es_credentials.username}
              onChange={e => this.onCredentialsChange('es_credentials', 'username', e.target.value)}
            />
          </EuiFormRow>

          <EuiFormRow
            label="Elasticsearch password"
            fullWidth
            helpText="(Optional) The password of the user specified above; if left empty, the currently set password will not change."
          >
            <EuiFieldPassword
              data-test-subj="ingest-es-credentials-password"
              value={this.state.configuration.es_credentials.password}
              onChange={e => this.onCredentialsChange('es_credentials', 'password', e.target.value)}
            />
          </EuiFormRow>

          <EuiFormRow
            label="Datasource username"
            fullWidth
            helpText="(Optional) The username set in the datasource connection (will override the default datasource username)."
          >
            <EuiFieldText
              data-test-subj="ingest-ds-credentials-username"
              value={this.state.configuration.ds_credentials.username}
              onChange={e => this.onCredentialsChange('ds_credentials', 'username', e.target.value)}
            />
          </EuiFormRow>

          <EuiFormRow
            label="Datasource password"
            fullWidth
            helpText="(Optional) The password set in the datasource connection (will override the default datasource password)."
          >
            <EuiFieldPassword
              data-test-subj="ingest-ds-credentials-password"
              value={this.state.configuration.ds_credentials.password}
              onChange={e => this.onCredentialsChange('ds_credentials', 'password', e.target.value)}
            />
          </EuiFormRow>
        </EuiAccordion>
        {this.renderNextButton()}
      </EuiForm>
    );
  }

  renderStepPipeline() {
    return (
      <EuiForm>
        {this.renderPipeline()}
        {this.renderNextButton()}
      </EuiForm>
    );
  }

  renderStepTransform() {
    return (
      <EuiForm>
        <EuiCallOut
          size="s"
          title="Add tranformations (Only supports 'Copy' operation on new fields introduced by pipelines):-"
          iconType="help"
        />
        <TransformsEditor
          transforms={this.state.customTransforms}
          onChange={transforms => this.onTransformsChange(transforms)}
        />
        <EuiSpacer size="s"/>
        <EuiCallOut
          size="s"
          color="success"
          title="Auto-Detected Field Mappings :-"
          iconType="indexMapping"
        />
        <EuiFormRow
          label="Remove Mappings"
          helpText="Note: This does NOT remove fields from final documents. Use Query to do so."
        >
          <EuiComboBox
            options={(() => this.state.mappingData.items.reduce((acc, ele) => {
              acc.push({ label: ele.sanitizedFieldName });
              return acc;
            }, []))()}
            selectedOptions={this.state.mappingData.removedItems}
            onChange={this.onChangeRemovedFields}
          />
        </EuiFormRow>
        <EuiSpacer size="s" />
        <MappingTable
          items={this.state.mappingData.items}
          hideMultiValued={true}
          removedItems={this.state.mappingData.removedItems}
          examples={this.state.mappingData.examples}
          onFieldRemoval={this.onFieldRemovalFromMappingTable}
          onChangeMapping={this.onChangeMapping}
        />
        {this.renderNextButton()}
      </EuiForm>
    );
  }

  renderStepSchedule() {
    return (
      <EuiForm>
        <EuiCallOut
          title={
            <p>
              (Optional) Schedule using the&nbsp;
              <a
                target="_blank"
                href="https://docs.support.siren.io/latest/platform/en/siren-investigate/data-reflection/scheduler-cron-syntax.html"
              >
                cron syntax.
              </a>
            </p>
          }
          iconType="help"
        />

        <EuiFormRow
          label="Schedule"
          fullWidth
          helpText="Set if the reflection should be recurring. (Scheduling disabled if empty)"
        >
          <Scheduler
            configuration={this.state.configuration}
            onChange={this.changeFieldValue}
          />
        </EuiFormRow>
        {this.renderSaveButtons()}
      </EuiForm>
    );
  }

  renderSaveButtons() {
    return (
      <EuiFormRow fullWidth>
        <EuiFlexGroup justifyContent="flexEnd">
          <EuiFlexItem grow={false}>
            <EuiButton
              data-test-subj="save-configuration"
              color="primary"
              fill
              iconType="save"
              onClick={() => this.onSaveClick()}
            >
              Save
            </EuiButton>
          </EuiFlexItem>
          <EuiFlexItem
            grow={false}
          >
            <EuiButton
              data-test-subj="save-and-run-configuration"
              color="secondary"
              fill
              iconType="play"
              onClick={() => this.saveAndStartNow()}
            >
              Save & Start Now
            </EuiButton>
          </EuiFlexItem>
        </EuiFlexGroup>
      </EuiFormRow>
    );
  }

  renderCurrentStep() {
    if (this.state.currentStep === STEP_INPUT) {
      return this.renderStepInput();
    } else if (this.state.currentStep === STEP_PIPELINE) {
      return this.renderStepPipeline();
    } else if (this.state.currentStep === STEP_TRANSFORM) {
      return this.renderStepTransform();
    } else if (this.state.currentStep === STEP_TARGET) {
      return this.renderStepTarget();
    } else if (this.state.currentStep === STEP_SCHEDULE) {
      return this.renderStepSchedule();
    }
  }

  renderSection() {
    return (
      <Beforeunload onBeforeunload={()=>this.onBeforeUnload()}>
        <Prompt
          when={this.state.dirty}
          message={this.onBeforeUnload()}
        />
        {this.renderModal()}

        {this.renderTitle()}

        <EuiSpacer size="l" />

        {this.renderSteps()}


        <EuiSpacer size="l" />

        {this.renderCurrentStep()}

      </Beforeunload>
    );
  }
}
