import { resolve } from '../../utils';
import React, { Component } from 'react';
import PropTypes from 'prop-types';
import {
  confirmDestructModal,
  modalWithForm,
  tooltipWrapComponent
} from '../../shared_components';
import {
  secondsToHms
} from '../xlsx/services/helper.js';
import cronstrue from 'cronstrue';
// eslint-disable-next-line import/no-unresolved
import chrome from 'ui/chrome';
import {
  JobStatus
} from './neo4j_wizard/constants';
import {
  EuiTitle,
  EuiButton,
  EuiButtonIcon,
  EuiButtonEmpty,
  EuiForm,
  EuiFormRow,
  EuiFieldText,
  EuiNotificationBadge,
  EuiHealth,
  EuiCard,
  EuiIcon,
  EuiInMemoryTable,
  EuiSpacer,
  EuiText,
  EuiToolTip,
  EuiFlexGroup,
  EuiFlexItem,
  EuiPopover
} from '@elastic/eui';

const noValueFound = '-';
// Possible job statuses as per Federate
const status = {
  completed: 'successful',
  running: 'running',
  aborting: 'aborting',
  user_terminated: 'aborted',
  failed: 'error'
};

const ROW_PROPS = {
  [JobStatus.FATAL]: {
    style: {
      backgroundColor: 'rgb(255,0,0,0.1)'
    }
  },
  [JobStatus.WAITING]: {
    style: {
      backgroundColor: 'rgb(255,211,0,0.1)'
    }
  },
  [JobStatus.SUCCESS]: {
    style: {
      backgroundColor: 'rgb(50,205,0,0.1)'
    }
  }
};

class JobsTable extends Component {
  constructor(props) {
    super(props);

    this.notifier = props.createNotifier('Reflection configurations');
    this.state = {
      pendingJobs: [],
      isPopoverOpen: false,
      configurations: [],
      cloning: false,
      deleteConfigId: '',
      cloningId: '',
      saveCloneJobAs: ''
    };
    this.refreshAfterModification = 1000;
    this.actions = [
      {
        render: (config) => {
          const ingestRunning = config._run && config._run.state === status.running;
          if (ingestRunning) {
            return (
              <EuiFlexGroup gutterSize="s" justifyContent="spaceAround" style={{ width: '128px' }}>
                <EuiFlexItem grow={false}>
                  <EuiButtonIcon
                    onClick={() => this.abortJob(config.id)}
                    iconType="stopFilled"
                    color="danger"
                    aria-label="Abort Job"
                    data-test-subj={`${config.id}-abort`}
                  />
                </EuiFlexItem>
              </EuiFlexGroup>
            );
          }
          return (
            <EuiFlexGroup gutterSize="s" alignItems="center">
              <EuiFlexItem grow={false}>
                <a href={resolve(`/configuration/${config.id}`)}>
                  <EuiButtonIcon
                    title="Edit"
                    iconType="pencil"
                    aria-label="Edit"
                    data-test-subj={`${config.id}-edit`}
                  />
                </a>
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiButtonIcon
                  title="Run"
                  onClick={() => this.runConfiguration(config.id)}
                  color="success"
                  iconType="play"
                  aria-label="Run"
                  data-test-subj={`${config.id}-run`}
                />
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiButtonIcon
                  title="Delete"
                  onClick={() => this.setState({ deleteConfigId: config.id })}
                  color="danger"
                  iconType="trash"
                  aria-label="Delete"
                  data-test-subj={`${config.id}-delete`}
                />
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiButtonIcon
                  title="Clone"
                  onClick={() => this.setState({ cloning: true, cloningId: config.id, saveCloneJobAs: `${config.id}_clone` })}
                  color="text"
                  iconType="copy"
                  aria-label="Clone"
                  data-test-subj={`${config.id}-copy`}
                />
              </EuiFlexItem>
            </EuiFlexGroup>
          );
        }
      }
    ];
    const formatValue = function (value) {
      return value || noValueFound;
    };

    const formatTime = function (value) {
      return (value && (new Date(value)).toUTCString()) || noValueFound;
    };

    const tooltipText = function (item) {
      const firstLine = (item._run && item._run.infos) || (item._last && item._last.infos) || 'Not Available';
      let text;
      if (item._run) {
        const timeElapsed = ((item._run.end_time || new Date()) - item._run.start_time) / 1000;
        text = (
          <div>
            <p>{firstLine}</p>
            <p><b>Document Count: </b>{item._run.count}</p>
            <p><b>Time Elapsed: </b>{secondsToHms(timeElapsed)}</p>
            <p><b>Reflection Rate: </b>
              {(item._run.count / timeElapsed).toFixed(2)} Docs/Sec
            </p>
          </div>
        );
      } else {
        text = (<p>{firstLine}</p>);
      }
      const errorDetails = item._run || item._last;
      if (errorDetails && errorDetails.error) {
        return (
          <div>
            {text}
            <p><b>Error Log(s):</b></p>
            <ul>
              {errorDetails.error.map(err => (<li>&nbsp;-&nbsp;{err}</li>))}
            </ul>
          </div>
        );
      }
      return text;
    };
    const displayStats = (info) => {
      const timeTaken = (info.end_time - info.start_time) / 1000;
      const docsPerSec = (info.count / timeTaken).toFixed(2);
      return (
        <div>
          <p><b>Documents Added:</b> {info.count}</p>
          <p><b>Reflection Rate:</b> {docsPerSec} Docs/Sec</p>
          <p><b>Time Taken:</b> {secondsToHms(timeTaken)}</p>
        </div>
      );
    };
    this.columns = [
      {
        field: 'id',
        name: 'ID'
      },
      {
        field: '_run.state',
        name: 'Status',
        render: (status, item) => tooltipWrapComponent(
          this.renderStatus(formatValue(status || (item._last && item._last.state)), item.id),
          tooltipText(item)
        )
      },
      {
        field: 'description',
        name: 'Description',
        render: formatValue
      },
      {
        field: '_last.start_time',
        name: 'Last Run',
        render: (startTime, { _run }) =>
          formatTime(_run ? _run.start_time : startTime)
      },
      {
        field: '_last.end_time',
        name: 'Last Completed',
        render: (endTime, { _last }) => {
          if (endTime) {
            return tooltipWrapComponent(
              <span className="helptext">{formatTime(endTime)}</span>,
              displayStats(_last)
            );
          }
          return formatTime(endTime);
        }
      },
      {
        field: 'schedule',
        name: 'Scheduling',
        render: (schedule, item) => {
          let scheduleTranslation;
          if (schedule) {
            try {
              scheduleTranslation = cronstrue.toString(schedule);
            } catch (e) {
              scheduleTranslation = schedule;
            }
          }
          const badge = schedule ? (
            <EuiToolTip
              title={item.enable_scheduler ? 'Enabled' : 'Disabled'}
              content={scheduleTranslation}
              position="right"
            >
              <EuiButtonEmpty
                size="s"
                color={item.enable_scheduler ? 'primary' : 'danger'}
                iconType={item.enable_scheduler ? 'check' : 'cross'}
                onClick={() => this.toggleScheduler(item)}
              >
                {formatValue(schedule)}
              </EuiButtonEmpty>
            </EuiToolTip>
          ) : formatValue(schedule);
          return (
            <div>
              {badge}
            </div>
          );
        }
      },
      {
        field: 'target',
        name: 'Target Index',
        render: formatValue
      }
    ];
    if (this.props.hasActions) {
      this.columns.push(
        {
          name: 'Actions',
          actions: this.actions
        }
      );
    }

    this.pagination = {
      initialPageSize: 10,
      pageSizeOptions: [5, 10, 20]
    };
  }

  getSearchBar() {
    return {
      toolsRight: this.renderToolsRight(),
      box: {
        incremental: true
      }
    };
  }

  _getRowProps({ _status }) {
    if (_status !== undefined && _status !== null) {
      return ROW_PROPS[_status];
    }
    return {};
  }

  async validateConfiguration(id) {
    try {
      await this.props.client.validateConfiguration(id);
      this.notifier.info('Configuration validated successfully.');
    } catch (e) {
      this.notifier.error(e);
    }
  }

  switchCloning(cloning) {
    this.setState({ cloning, cloningId: '', saveCloneJobAs: '' });
  }

  async toggleScheduler(jobConfiguration) {
    const jobId = jobConfiguration.id;
    jobConfiguration.enable_scheduler = !jobConfiguration.enable_scheduler;
    await this.props.client.saveConfiguration(jobId, jobConfiguration);
    this.notifier.info(`Scheduler ${jobConfiguration.enable_scheduler ? 'Enabled' : 'Disabled'} successfully!`);
    setTimeout(() => this.refreshJobs(), this.refreshAfterModification);
  }

  async cloneConfig(configId, cloneId) {
    const alreadyExists = await this.props.client.configurationExist(cloneId);
    if (alreadyExists) {
      this.notifier.warning('Unable to Clone! Configuration with the same ID already exists,' +
        ' please change the ID.');
      this.switchCloning(false);
      return;
    }
    const sourceConfig = await this.props.client.fetchConfiguration(configId);
    delete sourceConfig.id;
    await this.props.client.saveConfiguration(cloneId, sourceConfig);
    this.switchCloning(false);
    this.notifier.info('Configuration Cloned successfully!');
    setTimeout(() => this.refreshJobs(), this.refreshAfterModification);
  }

  deleteConfirmation() {
    if (this.state.deleteConfigId) {
      const title = 'Delete Ingest Configuration';
      const body = (
        <div>
          <p>You&rsquo;re about to delete <b>{this.state.deleteConfigId}</b>.</p>
          <p>Are you sure you want to do this?</p>
        </div>
      );
      const onCancel = () => {
        this.setState({ deleteConfigId: '' });
      };
      const onConfirm = () => {
        this.props.client.deleteConfiguration(this.state.deleteConfigId).then(
          res => {
            if (res.result === 'deleted') {
              this.notifier.info('Configuration Deleted successfully!');
            } else {
              this.notifier.error(`Unable to delete Ingestion: ${res.result}`);
            }
            setTimeout(() => this.refreshJobs(), this.refreshAfterModification);
          })
          .catch((e) => {
            this.notifier.error(`Unable to delete Ingestion: ${e.statusText}`);
          });
        this.setState({ deleteConfigId: '' });
      };
      return confirmDestructModal(title, body, onCancel, onConfirm);
    }
  }

  cloneModal() {
    if (this.state.cloning && this.state.cloningId) {
      const cloneNameElement = 'cloneName';
      const title = 'Clone Configuration';
      const form = (
        <EuiForm>
          <EuiFormRow
            label="Give a name to the clone:"
          >
            <EuiFieldText
              name={cloneNameElement}
              value={this.state.saveCloneJobAs}
              onChange={(e) => this.setState({ 'saveCloneJobAs': e.target.value })}
            />
          </EuiFormRow>
        </EuiForm>
      );
      const footer = (
        <div>
          <EuiButtonEmpty
            onClick={() => this.switchCloning(false)}
          >
            Cancel
          </EuiButtonEmpty>

          <EuiButton
            onClick={() => this.cloneConfig(this.state.cloningId, this.state.saveCloneJobAs)}
            fill
          >
            Save
          </EuiButton>
        </div>
      );
      const onClose = () => this.switchCloning(false);
      return modalWithForm(title, form, footer, onClose);

    }
  }

  async runConfiguration(id) {
    try {
      await this.props.client.runConfiguration(id);
      // TODO: put link to job in info message
      this.notifier.info('Configuration scheduled successfully.');
      setTimeout(() => this.refreshJobs(), this.refreshAfterModification);
    } catch (e) {
      this.notifier.error(e);
    }
  }

  async abortJob(id) {
    try {
      await this.props.client.abortJob(id);
      this.notifier.info('Job Aborted.');
      setTimeout(() => this.refreshJobs(), this.refreshAfterModification);
    } catch (e) {
      this.notifier.error(e);
    }
  }


  async refreshJobs() {
    try {
      const result = await this.props.fetchJobs();
      this.setState({
        configurations: result.jobs
      });
    } catch (e) {
      this.notifier.error(e);
      this.setState({ configurations: [] });
    }
  }

  async onMount() {
    this.interval = setInterval(() => this.refreshJobs(), chrome.getInjected('jobPageRefresh'));
    this.props.client.getPendingJobs().then(jobs => {
      this.setState({ pendingJobs: Object.keys(jobs).map(key => ({ jobId: key, jobData: jobs[key] })) });
    });
    this.refreshJobs();
  }

  async componentDidMount() {
    await this.onMount();
  }

  componentWillUnmount() {
    clearInterval(this.interval);
  }

  renderStatus(currentStatus, id) {
    let color;
    switch (currentStatus) {
      case noValueFound:
        color = 'subdued';
        break;
      case status.completed:
      case status.running:
        color = 'success';
        break;
      case status.aborting:
      case status.user_terminated:
        color = 'warning';
        break;
      case status.failed:
        color = 'danger';
        break;
      default:
        color = 'subdued';
        break;
    }
    return <EuiHealth data-test-subj={`${id}-status`} color={color}><span className="helptext">{currentStatus}</span></EuiHealth>;
  }

  togglePopover = () => {
    this.setState(state => {
      state.isPopoverOpen = !state.isPopoverOpen;
      return state;
    });
  }

  closePopover = () => {
    this.setState({ isPopoverOpen: false });
  }

  renderToolsRight() {
    const button = (
      <EuiButton
        iconType="arrowDown"
        iconSide="right"
        onClick={this.togglePopover}
      >
        <EuiNotificationBadge>{this.state.pendingJobs.length}</EuiNotificationBadge> Pending Jobs
      </EuiButton>
    );
    return [
      (
        this.state.pendingJobs.length > 0 ?
          <EuiPopover
            key="neo4jPopover"
            button={button}
            isOpen={this.state.isPopoverOpen}
            closePopover={this.closePopover}
          >
            {
              this.state.pendingJobs.map(job => {
                return (
                  <EuiCard
                    data-test-subj={job.jobId}
                    key={job.jobId}
                    icon={<EuiIcon style={{ color: '#476b9a' }} size="xxl" type="indexMapping" />}
                    title={`Continue Neo4J Job`}
                    description={`For Nodes: ${Object.keys(job.jobData)}`}
                    onClick={() => window.location.href = resolve(`/neo4j-wizard/pendingJob/${job.jobId}`)}
                  />
                );
              })
            }
          </EuiPopover>
          : <div key="neo4jPopover"/>
      ),
      (
        <EuiButton
          key="newReflectionJob"
          iconType="plusInCircle"
          href={resolve('/configuration')}
        >
          Add New
        </EuiButton>
      )
    ];
  }

  render() {
    if (this.state.configurations.length === 0) {
      return (
        <EuiText>
          <p>No configurations found, click on <strong>New</strong> to create a new one.</p>
        </EuiText>
      );
    }
    return (
      <div>
        <EuiInMemoryTable
          items={this.state.configurations}
          itemId="id"
          columns={this.columns}
          search={this.props.showToolbar ? this.getSearchBar() : null}
          hasActions={this.props.hasActions}
          rowProps={this._getRowProps}
          //pagination={this.pagination} Temporary remove pagination due to kibi-internal #8666
        />
        {this.cloneModal()}
        {this.deleteConfirmation()}
      </div>
    );
  }
}

JobsTable.propTypes = {
  fetchJobs: PropTypes.func.isRequired,
  hasActions: PropTypes.bool.isRequired,
  showToolbar: PropTypes.bool.isRequired,
  createNotifier: PropTypes.func.isRequired,
  client: PropTypes.object
};

export {
  JobsTable,
  status
};