/* eslint-disable import/no-unresolved */
import {
  setESIndexName
} from '../../xlsx/services/sheetServices';

const neo4jDocumentField = 'neo4jDocument';
const NEO4J_GENERATED_ID = 'Neo4j Generated';

/**
 * Get all unique values of an array.
 * @param  {Array.<Primitive>} values
 * @return {Array}
 */
function getUniqueValues(values) {
  const unique = new Set(values);
  const result = [];
  unique.forEach(value => result.push(value)); // Note: not using `Array.from()` because of IE 10
  return result;
}

/**
 * Facilitates mapping a Neo4j Datasource to Reflection jobs.
 */
class Neo4jMapperService {

  constructor(datasourceId, client, chrome) {
    this._datasourceId = datasourceId;
    this._client = client;
    this._apiBasePath = chrome.addBasePath(chrome.getInjected('apiBasePath'));
    this._neo4jAPI = `${this._apiBasePath}v1/neo4j/`;
    this._nodes = [];
    this._relations = [];
    this._nodeProperties = {};
    // Max queries allowed against the datasource.
    // To prevent overloading Federate or datasource.
    this._parallelDatasourceQueries = 5;
    this._isInitialized = this._initialize();
  }

  /**
   * Returns a promise which resolves when this service is initialized.
   * @return {Promise}
   */
  waitForInitialization() {
    return this._isInitialized;
  }

  get datasourceId() {
    return this._datasourceId;
  }

  /**
   * @return {Array.<Object>} Array of { label: 'nodeName' }
   */
  getNodes() {
    return this._nodes.map(node => ({ label: node }));
  }

  /**
   * @param  {Array.<Object>} nodes
   * @return {Array.<Object>}
   */
  getRelations(nodes) {
    return this._calculateRelations(new Set(nodes.map(ele => ele.label)));
  }

  /**
   * Prepare reflection jobs for passed nodes
   * @param  {Array.<Object>} nodes
   * @return {Array.<Object>} reflection jobs
   */
  prepareReflectionJobs(nodes) {
    const reflectionJobs = nodes.map(({ label }) => ({
      nodeName: label,
      targetIndex: this._getIndexName(label),
      jobName: this._getJobName(label),
      primaryKey: NEO4J_GENERATED_ID,
      properties: this._getNodeProperties(label)
    }));
    const relations = this.getRelations(nodes);
    return reflectionJobs.concat(
      relations.map(({ label, relations }) => ({
        nodeName: label,
        targetIndex: this._getIndexName(label),
        jobName: this._getJobName(label),
        primaryKey: NEO4J_GENERATED_ID,
        properties: this._getNodeProperties(label),
        relations
      }))
    );
  }

  /**
   * Create ingestion jobs
   * @param  {Array.<Object>} jobs
   * @param  {Boolean} triggerImmediately
   * @param  {Boolean} overwrite
   * @return {Object} Containing jobId and jobData if successful
   */
  async createIngestionJobs(jobs, triggerImmediately, overwrite) {
    const jobInfo = await this._createJobs(this._getJobsPayload(jobs));
    await this._deletePreviousJobLogs(jobs);
    if (triggerImmediately) {
      jobInfo.jobs_created.forEach(jobId => this._client.runConfiguration(jobId));
    }
    return jobInfo;
  }

  /**
   * Temporay hack until we implement JobLog curation in Federate
   * @param  {Array.<Object>} jobs
   */
  async _deletePreviousJobLogs(jobs) {
    try {
      const promises = [];
      jobs.forEach(({ jobName }) => promises.push(this._client._deleteJobLogs(jobName)));
      await Promise.all(promises);
    } catch (e) {
      // Do Nothing
    }
  }

  /**
   * Build jobs payload from jobs array, filtering unwanted fields,
   * generating required fields
   * @param  {Array.<Object>} jobs
   * @param  {Array.<Object>}
   */
  _getJobsPayload(jobs) {
    this._fillJobCypherQueries(jobs);
    const jobsPayload = jobs.map(({ nodeName, targetIndex, jobName, cypherQuery, relations }) => {
      const job = { nodeName, targetIndex, jobName, cypherQuery };
      if (relations) {
        job.relations = relations;
      }
      return job;
    });
    return jobsPayload;
  }

  /**
   * Fill passed jobs with their cypher queries
   * @param  {Array.<Object>} jobs
   */
  _fillJobCypherQueries(jobs) {
    const selectedPrimaryKeys = jobs.reduce((acc, { nodeName, primaryKey }) => {
      acc[nodeName] = primaryKey;
      return acc;
    }, {});
    jobs.forEach(job => {
      const { nodeName, primaryKey, relations } = job;
      job.cypherQuery = this._isRelationJob(job) ?
        this._getRelationCypherQuery(nodeName, primaryKey, relations, selectedPrimaryKeys) :
        this._getNodeCypherQuery(nodeName, primaryKey);
    });
  }

  /**
   * Check if the current job config belongs to a relation node
   * @param  {Object}  job
   * @return {Boolean}
   */
  _isRelationJob(job) {
    return job.relations && job.relations.length > 0;
  }

  async _queryDatasource(query) {
    if (this._parallelDatasourceQueries > 0) {
      try {
        this._parallelDatasourceQueries--;
        const resp = await this._client.fetchSample(this.datasourceId, query);
        if (!resp.fields) {
          throw new Error(resp.toString());
        }
        this._parallelDatasourceQueries++;
        return resp;
      } catch (e) {
        this._parallelDatasourceQueries++;
        throw e;
      }
    } else {
      return new Promise(resolve => setTimeout(() => resolve(this._queryDatasource(query)), 50));
    }
  }

  /**
   * Fetch all the nodes present in the Neo4J Datasource
   * @return {Array.<String>} Nodes
   */
  async _fetchNodes() {
    const { results } = await this._queryDatasource('call db.schema()');
    if (results.length > 0) {
      return results[0].nodes.map(node => node.name);
    }
    return results;
  }

  /**
   * Fetch all relation types present in the Neo4J Datasource
   * @return {Object} Relations [{label, relations}]
   */
  async _fetchRelations() {
    const { results } = await this._queryDatasource('call db.relationshipTypes()');
    const relations = {};
    if (results.length > 0) {
      const relationTypes = results.map(res => res.relationshipType);
      const promises = [];
      relationTypes.forEach(relationType => {
        const relationArray = [];
        promises.push(
          this._queryDatasource(`match (s)-[r:${relationType}]->(e) return distinct labels(s) as start, labels(e) as end`)
            .then(({ results }) => {
              const pushedRelations = new Set();
              results.forEach(res => {
                for (let i = 0, len = res.start.length; i < len; i++) {
                  const startNode = res.start[i];
                  for (let j = 0, len = res.end.length; j < len; j++) {
                    const endNode = res.end[j];
                    const relationKey = `${startNode}-${endNode}`;
                    if (!pushedRelations.has(relationKey)) {
                      pushedRelations.add(relationKey);
                      relationArray.push({ startNode, endNode });
                    }
                  }
                }
              });
              relations[relationType] = {
                label: relationType,
                relations: relationArray
              };
            })
        );
      });
      await Promise.all(promises);
    }
    return relations;
  }

  /**
   * Fetchs all possible properties of the nodeType
   * Warning: Not all properties would be present in each document of the Node!
   * @param  {String}  nodeName
   * @param  {Boolean} isRelation
   * @return {Set}
   */
  async _getAllProperties(nodeName, isRelation) {
    const properties = 'keys(n)';
    let query;
    if (isRelation) {
      query = `match ()-[n:${nodeName}]-() return distinct ${properties}`;
    } else {
      query = `match (n:${nodeName}) return distinct ${properties}`;
    }
    const { results } = await this._queryDatasource(query);
    const setOfProperties = new Set([NEO4J_GENERATED_ID]);
    results.forEach(row =>
      row[properties].forEach(prop => setOfProperties.add(prop))
    );
    return setOfProperties;
  }

  /**
   * Returns all possible properties of the nodeName
   * Warning: Not all properties would be present in each document of the Node!
   * @param  {String} nodeName
   * @return {Array.<Object>}
   */
  _getNodeProperties(nodeName) {
    return this._nodeProperties[nodeName];
  }

  /**
   * @param {String} nodeName
   * @param  {Boolean} isRelation
   */
  async _initNodeProperties(nodeName, isRelation) {
    const properties = await this._getAllProperties(nodeName, isRelation);
    const propertiesArray = [];
    properties.forEach(prop => {
      propertiesArray.push({ value: prop, text: prop });
    });
    this._nodeProperties[nodeName] = propertiesArray;
  }

  /**
   * Create reflection jobs
   * @param  {Array.<Object>} jobsData; An array of reflection job data
   * @return {Object}
   */
  async _createJobs(jobsData) {
    const data = await this._client.fetch(`${this._neo4jAPI}createJobs`, 'POST', {
      datasourceId: this.datasourceId,
      neo4jDocumentField,
      jobs: jobsData
    });
    if (!data.acknowledged && data.error) {
      throw new Error(data.error);
    }
    return data;
  }

  /**
   * Sets up the Neo4J mapping service, should be called in the constructor
   */
  async _initialize() {
    const promises = [];
    promises.push(this._fetchNodes().then(nodes => this._nodes = nodes));
    promises.push(this._fetchRelations().then(relations => this._relations = relations));
    await Promise.all(promises);
    const fetchProperties = [];
    this._nodes.forEach(node => fetchProperties.push(this._initNodeProperties(node, false)));
    Object.keys(this._relations).forEach(node => fetchProperties.push(this._initNodeProperties(node, true)));
    await Promise.all(fetchProperties);
    return true;
  }

  /**
   * Returns a valid ES Index name for the label
   * @param  {String} label
   * @return {String}
   */
  _getIndexName(label) {
    return setESIndexName(`${this.datasourceId}-${label}`);
  }

  /**
   * Returns a reflection job name for the label
   * @param  {String} label
   * @return {String}
   */
  _getJobName(label) {
    return `${this.datasourceId}:${label}`;
  }

  /**
   * Calculates applicable relations for the given set of nodes
   * @param  {Set.<String>} nodes; Selected nodes
   * @return {Array.<Object>} applicable relations
   */
  _calculateRelations(nodes) {
    const applicableRelations = [];
    for (const relation in this._relations) {
      if (this._relations.hasOwnProperty(relation)) {
        const relationArray = this._relations[relation].relations.filter(relationEdge =>
          nodes.has(relationEdge.startNode) && nodes.has(relationEdge.endNode)
        );
        if (relationArray.length > 0) {
          applicableRelations.push({
            label: relation,
            relations: relationArray
          });
        }
      }
    }
    return applicableRelations;
  }

  /**
   * @param  {String} nodeName
   * @param  {String} primaryKey
   * @param  {String} alias
   * @return {String} query to get primary key
   */
  _getPrimaryKeyQuery(nodeName, primaryKey, alias) {
    if (primaryKey === NEO4J_GENERATED_ID) {
      return `id(${alias})`;
    }
    return `${alias}.${primaryKey}`;
  }

  /**
   * Creates a cypher query for a data node
   * @param  {String} nodeName
   * @return {String} Cypher query for the node
   */
  _getNodeCypherQuery(nodeName, primaryKey) {
    return `MATCH (n:${nodeName}) RETURN ${this._getPrimaryKeyQuery(nodeName, primaryKey, 'n')} as node_id, ` +
      `labels(n) as node_labels, n as ${neo4jDocumentField}`;
  }

  /**
   * Builds section of query to fetch primary key of each node in relations
   * @param  {Array.<Object>}  relations
   * @param  {Boolean} isStartId
   * @param  {Object}  selectedPrimaryKeys; Object containing selected primay keys for every node
   * @return {Sting}
   */
  _getConnectedNodeIdQuery(relations, isStartId, selectedPrimaryKeys) {
    let nodeAlias;
    let nodes;
    if (isStartId) {
      nodeAlias = 's';
      nodes = getUniqueValues(relations.map(({ startNode }) => startNode));
    } else {
      nodeAlias = 'e';
      nodes = getUniqueValues(relations.map(({ endNode }) => endNode));
    }
    const nodesWithCustomKey = nodes.filter(node => selectedPrimaryKeys[node] !== NEO4J_GENERATED_ID);
    if (nodesWithCustomKey.length === 0) {
      return `id(${nodeAlias})`;
    } else {
      let fetchPrimaryKeyQuery = 'Case\n';
      nodesWithCustomKey.forEach(node => {
        let clause = `when Any(label in labels(${nodeAlias}) where label = '${node}')\n`;
        clause += `then ${nodeAlias}.${selectedPrimaryKeys[node]}\n`;
        fetchPrimaryKeyQuery += clause;
      });
      fetchPrimaryKeyQuery += `else id(${nodeAlias}) end`;
      return fetchPrimaryKeyQuery;
    }
  }

  /**
   * Creates a cypher query for a relation node
   * @param  {String} relationType
   * @param  {String} primaryKey
   * @param  {Array.<Object>} relations
   * @param  {Object} selectedPrimaryKeys; Object containing selected primay keys for every node
   * @return {String} Cypher query for the relation node
   */
  _getRelationCypherQuery(relationType, primaryKey, relations, selectedPrimaryKeys) {
    const matchQuery = `Match (s)-[r:${relationType}]->(e)`;
    let whereClause = 'where ';
    relations.forEach(({ startNode, endNode }) => {
      whereClause += `s:${startNode} AND e:${endNode} OR `;
    });
    whereClause = whereClause.slice(0, -3);
    const returnQuery = 'return\n' +
      `${this._getConnectedNodeIdQuery(relations, true, selectedPrimaryKeys)} as start_node_id,\n` +
      `${this._getConnectedNodeIdQuery(relations, false, selectedPrimaryKeys)} as end_node_id, ` +
      `${this._getPrimaryKeyQuery(relationType, primaryKey, 'r')} as node_id, ` +
      `TYPE(r) as relation_type, r as ${neo4jDocumentField}`;
    return  `${matchQuery}\n${whereClause}\n${returnQuery}`;
  }
}

export {
  Neo4jMapperService
};