import React from 'react';
import PropTypes from 'prop-types';
import {
  EuiButton,
  EuiButtonEmpty,
  EuiCallOut,
  EuiComboBox,
  EuiFlexGroup,
  EuiFlexItem,
  EuiPanel,
  EuiPageContent,
  EuiPageContentBody,
  EuiPageContentHeaderSection,
  EuiSpacer
} from '@elastic/eui';

import {
  ConfirmReflectionJobs
} from './confirm_reflection_jobs';
import {
  Neo4jMapperService
} from './neo4j_mapper_service';
import {
  ConfigureRelations
} from './configure_relations';
import {
  ConfigureSavedSearches
} from './configure_saved_searches';
import {
  JobStatus
} from './constants';
import {
  OntologyService
} from './ontology_service';
import { Wizard } from '../wizard';
import {
  JobsTable
} from '../jobs_table';

// The id also happens to be the location of step in this.state.step array
const STEPS = Object.freeze({
  NODE_SELECTION: 0,
  CONFIRM_REFLECTION_JOBS: 1,
  NOTIFICATION_SCREEN: 2,
  SAVED_SEARCH_OPTIONS: 3,
  RELATION_OPTIONS: 4,
  COMPLETION_SCREEN: 5
});

const LOG = {
  error: console.error.bind(null, 'Neo4j Wizard:')
};

class Neo4jWizard extends Wizard {

  /**
   * @param  {number} startingStep should either be STEPS.NODE_SELECTION or STEPS.NOTIFICATION_SCREEN
   * @return {Object}
   */
  _getInitialStepsState(startingStep) {
    return [
      {
        id: STEPS.NODE_SELECTION,
        name: 'Select Nodes',
        isComplete: false,
        disabled: true,
        nextButtonText: 'Next',
        nextButtonColor: 'primary',
        allowBack: true,
        allowNext: false,
        isLoading: false,
        nextFunction: this.resolveReflectionJobs
      },
      {
        id: STEPS.CONFIRM_REFLECTION_JOBS,
        name: 'Confirm Reflection Jobs',
        isComplete: false,
        disabled: true,
        nextButtonText: 'Confirm',
        nextButtonColor: 'primary',
        allowBack: true,
        allowNext: false,
        isLoading: false,
        nextFunction: this.createReflectionJobs
      },
      {
        id: STEPS.NOTIFICATION_SCREEN,
        name: 'Waiting for data',
        isComplete: false,
        disabled: true,
        nextButtonText: 'Continue',
        nextButtonColor: 'primary',
        allowBack: false,
        allowNext: false,
        isLoading: false,
        nextFunction: this.setupOntologyData
      },
      {
        id: STEPS.SAVED_SEARCH_OPTIONS,
        name: 'Configure Saved Searches',
        isComplete: false,
        disabled: true,
        nextButtonText: 'Next',
        nextButtonColor: 'primary',
        allowBack: true,
        allowNext: true,
        isLoading: false,
        nextFunction: this.goToNextStep
      },
      {
        id: STEPS.RELATION_OPTIONS,
        name: 'Configure Relations',
        isComplete: false,
        disabled: true,
        nextButtonText: 'Create Ontology',
        nextButtonColor: 'primary',
        allowBack: true,
        allowNext: true,
        isLoading: false,
        nextFunction: this.createOntology
      },
      {
        id: STEPS.COMPLETION_SCREEN,
        name: 'Done!',
        isComplete: false,
        disabled: true,
        nextButtonText: 'Go to Data Model',
        savedSearchId: '',
        nextButtonColor: 'secondary',
        allowBack: false,
        allowNext: true,
        isLoading: false,
        nextFunction: this.toToDataModelPage
      },
    ].reduce((acc, ele) => {
      if (ele.id === startingStep) {
        ele.disabled = false;
      }
      acc[ele.id] = ele;
      return acc;
    }, {});
  }

  /**
   * Mutates steps and disable all previous steps
   * @param  {Object} steps
   * @param  {Number} currentStep
   */
  static _disablePreviousSteps(steps, currentStep) {
    Object.keys(steps).forEach(step => {
      if (step < currentStep) {
        steps[step].disabled = true;
      }
    });
  }

  /**
   * Mutates state object according to defined interceptors
   * @param  {Object} state
   */
  _nextStepInterceptor(state) {
    const nextStep = state.currentStep + 1;
    switch (nextStep) {
      case STEPS.NOTIFICATION_SCREEN:
      case STEPS.COMPLETION_SCREEN:
        Neo4jWizard._disablePreviousSteps(state.steps, nextStep);
    }
  }

  constructor(props) {
    super(props);
    this.neo4jDatasource = this.props.match.params.neo4jDatasource;
    const jobId = this.props.match.params.jobId;
    if (jobId) {
      this.state = {
        currentStep: STEPS.NOTIFICATION_SCREEN,
        steps: this._getInitialStepsState(STEPS.NOTIFICATION_SCREEN),
        jobId,
        initializing: true
      };
    } else {
      this.state = {
        currentStep: STEPS.NODE_SELECTION,
        steps: this._getInitialStepsState(STEPS.NODE_SELECTION),
        initializing: true,
        reflectionJobs: {}
      };
    }
    this.title = 'Neo4j Importer';
    this.notifier = this.props.createNotifier(this.title);
    document.title = `${this.title} - Siren`;
  }

  /**
   * Fetches jobInfo of the passes jobId
   * And iniializes ontologyService
   * @param  {String} jobId
   */
  async _fetchJobInfo(jobId) {
    const jobInfo = await this.props.client.fetchJobInfo(jobId);
    this._initializeOntologyService(jobInfo);
  }

  _initializeOntologyService(jobInfo) {
    this.ontologyService = new OntologyService(jobInfo, this.props.client, this.props.chrome);
  }

  async onMount() {
    try {
      if (!this.state.jobId) {
        this._neo4jMapperService = new Neo4jMapperService(this.neo4jDatasource, this.props.client, this.props.chrome);
        await this.neo4jMapperService.waitForInitialization();
        const availableNodes = this.neo4jMapperService.getNodes();
        this.setState(state => {
          state.availableNodes = availableNodes;
          state.selectedNodes = availableNodes;
          state.initializing = false;
          state.steps[STEPS.NODE_SELECTION].allowNext = availableNodes.length > 0;
          return state;
        });
      } else {
        await this._fetchJobInfo(this.state.jobId);
        this.setState({ initializing: false });
      }
    } catch (e) {
      this.setState({
        mountError: e.data || {
          error: e.name,
          message: e.message
        }
      });
    }
  }

  /**
   * This service is initialized if starting the wizard from scratch.
   * Only to be used till creation of Reflection jobs.
   * @Nullable
   * @return {Neo4jMapperService}
   */
  get neo4jMapperService() {
    return this._neo4jMapperService;
  }

  async componentDidMount() {
    await this.onMount();
  }

  _isAtFirstStep() {
    return this.state.currentStep === STEPS.NODE_SELECTION;
  }

  _isAtLastStep() {
    return this.state.currentStep >= STEPS.COMPLETION_SCREEN;
  }

  goBack() {
    window.history.back();
  }

  getSteps() {
    return this.state.steps;
  }

  goToStep(stepId) {
    this.setState({ currentStep: stepId });
  }

  getTitle() {
    return this.title;
  }

  /**
   * Resolve and generate the reflection jobs required for selected nodes.
   * And move onto next step
   */
  resolveReflectionJobs = () => {
    this.setState(state => {
      state.reflectionJobs = this.neo4jMapperService.prepareReflectionJobs(state.selectedNodes);
      state.steps[STEPS.CONFIRM_REFLECTION_JOBS].allowNext = true;
      return state;
    });
    this.goToNextStep();
  }

  createReflectionJobs = async () => {
    if (!this.goingNext) {
      this.goingNext = true;
      this.setState(state => {
        state.steps[STEPS.CONFIRM_REFLECTION_JOBS].isLoading = true;
      });
      try {
        const jobInfo = await this.neo4jMapperService.createIngestionJobs(this.state.reflectionJobs, true, true);
        this._initializeOntologyService(jobInfo);
        this.goToNextStep();
      } catch (e) {
        this.notifier.error(e);
      }
      this.goingNext = false;
    }
  }

  setupOntologyData = () => {
    const indexEntitiesList = this.ontologyService.getJobDataList();
    this.setState({ indexEntitiesList });
    this.goToNextStep();
  }

  createOntology = async () => {
    if (!this.goingNext) {
      this.goingNext = true;
      this.setState(state => {
        state.steps[STEPS.RELATION_OPTIONS].isLoading = true;
      });
      try {
        await this.ontologyService.publishOntology();
        this.goToNextStep();
      } catch (e) {
        this.notifier.error(e);
      }
      this.goingNext = false;
    }
  }

  toToDataModelPage = () => {
    window.location.href = `./kibana#/management`;
  }

  goToNextStep = () => {
    if (!this._isAtLastStep()) {
      this.setState(state => {
        this._nextStepInterceptor(state);
        const currentStep = state.currentStep;
        state.steps[currentStep].isComplete = true;
        state.steps[currentStep].isLoading = false;
        state.steps[currentStep + 1].disabled = false;
        state.currentStep++;
        return state;
      });
    } else {
      window.location.href = `./kibana#/management/siren/datamodel/SAVED_SEARCH/${this.state.steps[this.state.currentStep].savedSearchId}`;
    }
  }

  goToPreviousStep = () => {
    if (!this._isAtFirstStep()) {
      this.setState(state => {
        state.currentStep--;
        return state;
      });
    } else {
      this.goBack();
    }
  }

  changeSelectedDataNodes = selectedNodes => {
    this.setState(state => {
      state.selectedNodes = selectedNodes;
      state.steps[STEPS.NODE_SELECTION].allowNext = selectedNodes.length > 0;
      return state;
    });
  }

  onIndexEntityChange = changedEntity => {
    this.setState(state => {
      const entityIndex = state.indexEntitiesList.findIndex(entity => changedEntity.nodeName === entity.nodeName);
      state.indexEntitiesList[entityIndex] = changedEntity;
      return state;
    });
  }

  fetchJobs = async () => {
    const result = await this.ontologyService.getReflectionJobs();
    this.setState(state => {
      state.steps[STEPS.NOTIFICATION_SCREEN].allowNext = result.status === JobStatus.SUCCESS;
      return state;
    });
    return result;
  }

  changeJobData = (nodeName, key, value) => {
    this.setState(state => {
      const index = state.reflectionJobs.findIndex(job => job.nodeName === nodeName);
      state.reflectionJobs[index][key] = value;
      return state;
    });
  }

  renderNodeSelectionStep() {
    return (
      <React.Fragment>
        <h3>Select Nodes:</h3>
        <EuiComboBox
          placeholder="Select nodes to reflect"
          options={this.state.availableNodes}
          selectedOptions={this.state.selectedNodes}
          onChange={this.changeSelectedDataNodes}
        />
      </React.Fragment>
    );
  }

  renderConfirmReflectionJobsStep() {
    return (
      <ConfirmReflectionJobs
        reflectionJobs={this.state.reflectionJobs}
        changeJobData={this.changeJobData}
        toggleConfirmButton={() => {}} // TODO: Implement mutation
      />
    );
  }

  renderNotificationScreenStep() {
    return (
      <React.Fragment>
        <EuiCallOut
          size="s"
          title="Waiting for the required jobs to start reflecting data."
          iconType="clock"
        />
        <JobsTable
          fetchJobs={this.fetchJobs}
          hasActions={false}
          showToolbar={false}
          createNotifier={this.props.createNotifier}
          client={this.props.client}
        />
      </React.Fragment>
    );
  }

  renderSavedSearchOptionsStep() {
    return (
      <ConfigureSavedSearches
        indexEntitiesList={this.state.indexEntitiesList}
        onItemChange={this.onIndexEntityChange}
      />
    );
  }

  renderRelationOptionsStep() {
    return (
      <ConfigureRelations
        relationEntitiesList={this.state.indexEntitiesList.filter(entity => !!entity.relations)}
        onItemChange={this.onIndexEntityChange}
      />
    );
  }

  renderCompletionScreen() {
    return 'Ontology created. You may go to Data Model page to review/make changes.';
  }

  renderLoadingScreen() {
    if (this.state.mountError) {
      return (
        <EuiCallOut
          title={this.state.mountError.error}
          color="danger"
          iconType="alert"
        >
          <p>
            {this.state.mountError.message}
          </p>
        </EuiCallOut>
      );
    }
    return <h3>Loading...</h3>;
  }

  _renderCurrentStep() {
    if (this.state.initializing) {
      return this.renderLoadingScreen();
    }
    switch (this.state.currentStep) {
      case STEPS.NODE_SELECTION:
        return this.renderNodeSelectionStep();
      case STEPS.CONFIRM_REFLECTION_JOBS:
        return this.renderConfirmReflectionJobsStep();
      case STEPS.NOTIFICATION_SCREEN:
        return this.renderNotificationScreenStep();
      case STEPS.SAVED_SEARCH_OPTIONS:
        return this.renderSavedSearchOptionsStep();
      case STEPS.RELATION_OPTIONS:
        return this.renderRelationOptionsStep();
      case STEPS.COMPLETION_SCREEN:
        return this.renderCompletionScreen();
      default:
        LOG.error(`Step: ${this.state.currentStep}, Not Found!`);
        return this.renderNodeSelectionStep();
    }
  }

  renderCurrentStep() {
    return (
      <EuiPageContent verticalPosition="center" horizontalPosition="center">
        <EuiPageContentBody>
          {this._renderCurrentStep()}
          <EuiSpacer size="l" />
          {this.renderFooter()}
        </EuiPageContentBody>
      </EuiPageContent>
    );
  }

  renderFooter() {
    return (
      <EuiFlexGroup justifyContent="flexEnd">
        {
          this.state.steps[this.state.currentStep].allowBack ?
            <EuiFlexItem grow={false}>
              <EuiButtonEmpty
                onClick={this.goToPreviousStep}
              >
                {this._isAtFirstStep() ? 'Cancel' : 'Back'}
              </EuiButtonEmpty>
            </EuiFlexItem>
            : ''
        }
        <EuiFlexItem grow={false}>
          <EuiButton
            color={this.state.steps[this.state.currentStep].nextButtonColor}
            onClick={this.state.steps[this.state.currentStep].nextFunction}
            isDisabled={!this.state.steps[this.state.currentStep].allowNext ||
              this.state.steps[this.state.currentStep].isLoading}
            isLoading={this.state.steps[this.state.currentStep].isLoading}
            fill
          >
            {this.state.steps[this.state.currentStep].nextButtonText}
          </EuiButton>
        </EuiFlexItem>
      </EuiFlexGroup>
    );
  }

}

Neo4jWizard.propTypes = {
  createNotifier: PropTypes.func.isRequired,
  client: PropTypes.object.isRequired,
  sections: PropTypes.arrayOf(
    PropTypes.shape({
      href: PropTypes.string.isRequired,
      id: PropTypes.string.isRequired,
      name: PropTypes.oneOfType([
        PropTypes.element,
        PropTypes.string
      ]).isRequired
    })
  ).isRequired,
  currentSection: PropTypes.string.isRequired
};

export {
  Neo4jWizard
};
