import React from 'react';

import {
  EuiFormRow,
  EuiTextArea,
  EuiFieldText
} from '@elastic/eui';


export default class MappingEditor extends React.Component {

  static getDerivedStateFromProps(nextProps, prevState) {
    const derived = {};

    if (prevState === null || prevState.prevProps.mappingOption !== nextProps.mappingOption) {
      derived.mappingOption = nextProps.mappingOption;
      const validation = {
        mapping: {
          valid: true,
          errors: []
        },
        format: {
          valid: true,
          errors: []
        }
      };
      derived.validation = validation;
    }

    if (prevState === null || prevState.prevProps.mapping !== nextProps.mapping) {
      derived.mapping = nextProps.mapping;
      derived.mappingObject = JSON.parse(nextProps.mapping);
    }

    derived.prevProps = nextProps;

    return derived;
  }

  constructor(props) {
    super(props);
    this.state = {
      validation: null,
      mappingObject: {},
      mapping: props.mapping,
      prevProps: {
        mappingOption: props.mappingOption,
        mapping: props.mapping
      }
    };
  }

  getFieldMappingValue(field) {
    if (this.state.mappingObject[field]) {
      return this.state.mappingObject[field];
    }
    return '';
  }

  onMappingChange(value) {
    const errors = [];
    try {
      JSON.parse(value);
    } catch (e) {
      errors.push('Could not parse mapping JSON.');
    }
    this.setState(state => {
      const validation = state.validation;
      validation.mapping.valid = errors.length === 0;
      validation.mapping.errors = errors;
      return {
        mapping: value,
        validation
      };
    });
    if (errors.length === 0) {
      this.props.onChange(value);
    }
  }

  onMappingFieldChange(field, value) {
    const mappingObject = JSON.parse(this.props.mapping);
    if (value === '') {
      delete mappingObject[field];
    } else {
      mappingObject[field] = value;
    }
    this.onMappingChange(JSON.stringify(mappingObject));
  }

  render() {

    switch (this.props.mappingOption) {
      case 'custom':
        return (
          <EuiFormRow
            label="JSON mapping definition"
            isInvalid={this.state.validation.mapping.valid !== true}
            errors={this.state.validation.mapping.errors}
          >
            <EuiTextArea
              value={this.state.mapping}
              onChange={e => this.onMappingChange(e.target.value)}
            />
          </EuiFormRow>
        );
      case 'date':
        return (
          <EuiFormRow
            label="Date formats"
            helpText={
              'Dates must be formatted according to one of the syntaxes allowed by the ' +
              'strict_date_optional_time ISO format e.g. YYYY-MM-DD [\'T\' HH:MM:SS [offset]]'
            }
          >
            <EuiFieldText
              value={this.getFieldMappingValue('format')}
              onChange={e => this.onMappingFieldChange('format', e.target.value)}
            />
          </EuiFormRow>
        );
      default:
        return null;
    }
  }

}
