import isEmpty from 'lodash.isempty';

const MAPPING_OPTIONS = [
  {
    value: 'default',
    text: 'Not set'
  },
  {
    value: 'text_keyword',
    text: 'Text and keyword'
  },
  {
    value: 'text',
    text: 'Text'
  },
  {
    value: 'keyword',
    text: 'Keyword'
  },
  {
    value: 'geo_point',
    text: 'Geo location'
  },
  {
    value: 'long',
    text: 'Long'
  },
  {
    value: 'short',
    text: 'Short'
  },
  {
    value: 'integer',
    text: 'Integer'
  },
  {
    value: 'double',
    text: 'Double'
  },
  {
    value: 'float',
    text: 'Float'
  },
  {
    value: 'byte',
    text: 'Byte'
  },
  {
    value: 'binary',
    text: 'Binary'
  },
  {
    value: 'ip',
    text: 'IP'
  },
  {
    value: 'geo_shape',
    text: 'Geo Shape'
  },
  {
    value: 'date',
    text: 'Date'
  },
  {
    value: 'boolean',
    text: 'Boolean'
  },
  {
    value: 'object',
    text: 'Object'
  },
  {
    value: 'custom',
    text: 'Custom'
  }
];

const OPTION_VALUES = MAPPING_OPTIONS.map(o => o.value);


export default class MappingHandler {

  /**
   * Returns a list of mapping options.
   */
  static getMappingOptions() {
    return MAPPING_OPTIONS;
  }

  /**
   * Returns the option value for a mapping.
   *
   * @param {Object} mapping A JSON mapping in string format.
   */
  static getMappingOption(mapping) {
    let parsedMapping;
    try {
      parsedMapping = JSON.parse(mapping);
    } catch (e) {
      return 'custom';
    }
    if (isEmpty(parsedMapping)) {
      return 'default';
    }
    if (OPTION_VALUES.indexOf(parsedMapping.type) !== -1) {
      switch (parsedMapping.type) {
        case 'text':
          if (parsedMapping.fields && parsedMapping.fields.keyword) {
            if (Object.keys(parsedMapping.fields.keyword).length === 1 &&
                parsedMapping.fields.keyword.type === 'keyword') {
              return 'text_keyword';
            }
          }
          if (Object.keys(parsedMapping).length === 1) {
            return 'text';
          }
          break;
        default:
          if (Object.keys(parsedMapping).length === 1) {
            return parsedMapping.type;
          } else {
            return 'custom';
          }
      }
    }
    return 'custom';
  }

  /**
   * Returns the default mapping corresponding to an option.
   */
  static getMapping(option) {
    const mapping = {};
    switch (option) {
      case 'text_keyword':
        mapping.type = 'text';
        mapping.fields = {
          keyword: {
            type: 'keyword'
          }
        };
        break;
      case 'custom':
        break;
      default:
        mapping.type = option;
        break;
    }
    return mapping;
  }

}