import PropTypes from 'prop-types';
import React, {
  Component,
  Fragment
} from 'react';
import {
  EuiButton,
  EuiButtonEmpty,
  EuiFlexGroup,
  EuiFlexItem,
  EuiFormRow,
  EuiProgress,
  EuiSpacer
} from '@elastic/eui';
import {
  secondsToHms
} from '../services/helper.js';
import { displayStats } from './ui_helpers';
import ErrorTable from './errorTable';
import {
  configService,
  confirmModal,
} from '../../../shared_components';

class ImportControls extends Component {

  constructor(props) {
    super(props);
    this.state = {
      progress: {
        current: 0,
        color: 'secondary'
      },
      newJob: true
    };
    this.jobStats = {};
    this.configService = configService();
    this.progressBarThrottle = 0;
    this.timeout = 200;
    this.updateProgressBar = true;
    document.addEventListener('visibilitychange', () => {
      this.updateProgressBar = !document.hidden;
    });
  }

  componentWillReceiveProps(nextProps) {
    if (!!nextProps.progress) {
      this.setState(state => {
        return {
          progress: {
            ...state.progress,
            ...nextProps.progress
          }
        };
      });
    }
  }

  progressTracker = (status, startingTime) => {
    if (status.progress) {
      if (status.progress !== 100) {
        if (this.progressBarThrottle > 0) {
          this.progressBarThrottle--;
          return this.jobStats;
        } else if (!this.updateProgressBar) {
          return this.jobStats;
        }
      }
      this.progressBarThrottle = this.configService.progressUpdationFrequency;
      const ingestionStats = status.stats;
      const secondsElapsed = ((new Date() - startingTime) / 1000);
      const stats = {
        documentsIngested: ingestionStats.documentsIngested,
        speed: (ingestionStats.documentsIngested / secondsElapsed).toFixed(2)
      };
      if (ingestionStats.totalDocuments) {
        stats.totalDocuments = ingestionStats.totalDocuments;
        stats.eta = secondsToHms((ingestionStats.totalDocuments - ingestionStats.documentsIngested) / stats.speed);
      } else {
        stats.totalFileSize = ingestionStats.fileParsed.totalSize;
        stats.cursor = ingestionStats.fileParsed.cursor;
        stats.eta = secondsToHms((stats.totalFileSize - stats.cursor) / (stats.cursor / secondsElapsed));
      }
      stats.secondsElapsed = secondsElapsed;
      this.jobStats = stats;
      this.setState(state => {
        return {
          progress: {
            current: status.progress,
            stats
          },
          errorsDuringIndexing: status.errorsDuringIndexing,
          newJob: status.progress <= state.current
        };
      });
    } else if (status.networkError) {
      this.props.addErrorToast('Encountered a network error!');
    }
    return this.jobStats;
  };

  handleNextStep = () => {
    this.props.handleNextStep(this.progressTracker);
  }

  importPrompt() {
    if (this.props.showImportPrompt) {
      const title = 'Quick Import';
      const body = (
        <div>
          <p>Import spreadsheet without tweaking configuration?</p>
        </div>
      );
      const onCancel = () => {
        this.props.setImportPrompt(false);
      };
      const onConfirm = () => {
        onCancel();
        this.handleNextStep();
        setTimeout(() => document.getElementById('saveConfig').scrollIntoView(), this.timeout);
      };
      return confirmModal(title, body, onCancel, onConfirm, 'Edit Before Import', 'Quick Import!');
    }
  }

  renderImportControls() {
    let abortButton = '';
    if (this.props.importButton.loading) {
      abortButton = (
        <EuiFlexItem grow={false}>
          <EuiButton
            fill
            color="danger"
            onClick={this.props.abortIngestion}
            iconType="stopFilled"
            data-test-subj="abort-import"
          >
            Abort
          </EuiButton>
        </EuiFlexItem>
      );
    }
    return (
      <EuiFormRow fullWidth={true}>
        <EuiFlexGroup gutterSize="s" alignItems="center" justifyContent="spaceBetween">
          <EuiFlexItem grow={false}>
            <EuiButtonEmpty
              onClick={this.props.previousStep}
              size="s"
              iconType="arrowLeft"
              data-test-subj="previous-step"
            >
              back
            </EuiButtonEmpty>
          </EuiFlexItem>
          <EuiFlexItem grow={false}>
            <EuiFlexGroup gutterSize="s" alignItems="center" justifyContent="flexEnd">
              {abortButton}
              <EuiFlexItem grow={false}>
                <EuiButton
                  fill
                  isDisabled={this.props.blockIngestion}
                  onClick={this.handleNextStep}
                  iconType="importAction"
                  data-test-subj="import"
                  isLoading={this.props.importButton.loading}
                >
                  {this.props.importButton.text}
                </EuiButton>
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiButton
                  fill
                  color="secondary"
                  id="saveConfig"
                  data-test-subj="save-config"
                  isDisabled={this.props.blockIngestion}
                  onClick={() => this.props.switchSaveModal(true)}
                  iconType="save"
                >
                  Save Config
                </EuiButton>
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiFlexItem>
        </EuiFlexGroup>
      </EuiFormRow>
    );
  }

  renderProgress() {
    let statsDisplay;
    if (this.state.progress.stats) {
      const { stats } = this.state.progress;
      statsDisplay = displayStats(stats);
    }
    return (
      <Fragment>
        {statsDisplay}
        <EuiProgress
          value={this.state.progress.current}
          max={100}
          color={this.state.progress.color}
          size="s"
        />
      </Fragment>
    );
  }

  renderErrorTable() {
    if (this.state.errorsDuringIndexing) {
      return <ErrorTable errors={this.state.errorsDuringIndexing} initialIsOpen={true} newJob={this.state.newJob} />;
    }
    return '';
  }

  render() {
    return (
      <Fragment>
        {this.importPrompt()}
        {this.renderImportControls()}
        <EuiSpacer size="m" />
        {this.renderProgress()}
        <EuiSpacer size="m" />
        {this.renderErrorTable()}
      </Fragment>
    );
  }
}

ImportControls.propTypes = {
  blockIngestion: PropTypes.bool.isRequired,
  handleNextStep: PropTypes.func.isRequired,
  importButton: PropTypes.exact({
    text: PropTypes.string.isRequired,
    loading: PropTypes.bool.isRequired
  }),
  previousStep: PropTypes.func.isRequired,
  switchSaveModal: PropTypes.func.isRequired,
  abortIngestion: PropTypes.func.isRequired,
  addErrorToast: PropTypes.func.isRequired,
  progress: PropTypes.exact({
    color: PropTypes.string,
    current: PropTypes.number,
    stats: PropTypes.object
  }),
  showImportPrompt: PropTypes.bool,
  setImportPrompt: PropTypes.func.isRequired
};

export default ImportControls;