import React, { Component } from 'react';
import {
  EuiFormRow,
  EuiSwitch,
  EuiSpacer,
  EuiTextArea
} from '@elastic/eui';
import {
  jsonObjectToString,
  jsonStringToObject
} from '../../../../shared_components';

class AdvJson extends Component {
  constructor(props) {
    super(props);
    this.state = {
      item: props.item,
      isInvalid: false,
      advPlaceholder: this.getPlaceholder(props.item.type)
    };
    this.state.item.advJson = jsonObjectToString(this.state.item.advJson);
    this.errors = [];
  }

  componentWillReceiveProps(props) {
    const item = props.item;
    //Parse to String if JSON
    item.advJson = jsonObjectToString(item.advJson);
    if (this.errors.length > 0) {
      this.errors = ['Erroneous change removed!'];
    }
    this.setState({ item, advPlaceholder: this.getPlaceholder(item.type) });
  }

  getPlaceholder(type) {
    switch (type) {
      case 'date':
        return 'JSON formatted properties to be merged with the ' +
        'elasticsearch mapping definition for this field, Example:- {"format": "DD/MM/YYYY HH:mm"}';
      default:
        return 'JSON formatted properties to be merged with the ' +
        'elasticsearch mapping definition for this field, Example:- {"fielddata": true}';
    };
  }

  onChange = e => {
    const { name, value } = e.target;
    this.setState({
      item: ({ ...this.state.item,
        ...{
          [name]: value
        }
      })
    });
  }

  onBlur = e => {
    let value =  e.target.value;
    let isInvalid = false;
    if (value) {
      try {
        value = JSON.parse(value);
      } catch (e) {
        this.errors = [e.message];
        isInvalid = true;
      }
      if (!isInvalid) {
        this.errors = [];
        if (value.type) {
          delete value.type;
          this.helpText = 'Field type should be selected from the Dropdown.' +
            'You may want to define \'fields\' setting in order to use multi-fields.';
        }
        value = jsonObjectToString(value);
        this.props.triggerUpdate(value);
      } else {
        this.setState((state) => {
          return {
            isInvalid,
            item: { ...state.item,
              ...{
                advJson: value
              }
            }
          };
        });
      }
    }
  }

  onToggle = e => {
    const { name, checked } = e.target;
    const advJson = checked ? this.state.advJson : '';
    this.props.triggerUpdate(advJson, checked);
  }

  render() {
    let jsonTextField = null;
    if (this.state.item.useAdv) {
      jsonTextField = (
        <div>
          <EuiSpacer size="s" />
          <EuiFormRow
            isInvalid={this.state.isInvalid}
            error={this.errors}
            helpText={this.helpText}
          >
            <EuiTextArea
              className="advjsontext"
              name="advJson"
              rows={4}
              value={this.state.item.advJson}
              onChange={this.onChange}
              onBlur={this.onBlur}
              error={this.errors}
              isInvalid={this.state.isInvalid}
              placeholder={this.state.advPlaceholder}
            />
          </EuiFormRow>
        </div>
      );
    }
    return (
      <div style={{ width: '400px' }}>
        <EuiSwitch
          name="useAdv"
          sanitizedfieldname={this.state.item.sanitizedFieldName}
          label="Advanced Mapping?"
          checked={!!this.state.item.useAdv}
          onChange={this.onToggle}
        />
        {jsonTextField}
      </div>
    );
  }
}

export default AdvJson;

