import React from 'react';
import { mount } from 'enzyme';
import AdvJson from './advJson';
import {
  jsonObjectToString
} from '../../../../shared_components';

describe('AdvJson', () => {
  let emptyMappingProps;
  let emptyProps;
  let fullProps;
  const triggerUpdateMock = jest.fn();

  const getMount = props => {
    return mount(<AdvJson {...props} />);
  };
  const findMappingSwitch = component => {
    return component.find('.euiSwitch__input');
  };

  const findTextArea = component => {
    return component.find('textarea');
  };

  beforeEach(() => {
    triggerUpdateMock.mockReset();
    emptyProps = { item: {}, triggerUpdate: triggerUpdateMock };
    fullProps = {
      item: {
        advJson: '{ "format": "DD/MM/YYYY HH:mm" }',
        useAdv: true,
        type: 'date',
        sanitizedFieldName: 'date'
      },
      triggerUpdate: triggerUpdateMock
    };
  });

  describe('renders', () => {
    it('(emptyProps)', () => {
      const componentMount = getMount(emptyProps);
      const mappingSwitch = findMappingSwitch(componentMount);
      expect(mappingSwitch.length).toBe(1);
      expect(mappingSwitch.getDOMNode().checked).toBe(false);
      expect(triggerUpdateMock.mock.calls.length).toBe(0);
    });

    it('(fullProps)', () => {
      const componentMount = getMount(fullProps);
      const mappingSwitch = findMappingSwitch(componentMount);
      expect(mappingSwitch.length).toBe(1);
      expect(mappingSwitch.getDOMNode().checked).toBe(true);

      const textArea = findTextArea(componentMount);
      expect(textArea.length).toBe(1);
      expect(textArea.getDOMNode().value).toBe(fullProps.item.advJson);
      expect(triggerUpdateMock.mock.calls.length).toBe(0);
    });
  });

  describe('toggle', () => {
    it('open works', () => {
      const componentMount = getMount(emptyProps);
      const mappingSwitch = findMappingSwitch(componentMount);
      const props = mappingSwitch.props();

      //confirm onChange
      props.onChange({ target: { name: props.name, checked: !props.checked } });
      expect(triggerUpdateMock.mock.calls.length).toBe(1);
      expect(triggerUpdateMock).toBeCalledWith(emptyProps.item.advJson, true);

      //Trigger re-render with new props
      componentMount.setProps({ ...emptyProps, ...{ item: { useAdv: true } } });
      const textArea = findTextArea(componentMount);
      expect(textArea.length).toBe(1);
      expect(textArea.getDOMNode().value).toBe('');
    });

    it('close works', () => {
      const componentMount = getMount(fullProps);
      const mappingSwitch = findMappingSwitch(componentMount);
      const props = mappingSwitch.props();

      //confirm onChange
      props.onChange({ target: { name: props.name, checked: !props.checked } });
      expect(triggerUpdateMock.mock.calls.length).toBe(1);
      expect(triggerUpdateMock).toBeCalledWith('', false);

      //Trigger re-render with new props
      componentMount.setProps({ ...fullProps, ...{ item: { useAdv: false } } });
      const textArea = findTextArea(componentMount);
      expect(textArea.length).toBe(0);
    });
  });

  it('type field is cleared if set', () => {
    const componentMount = getMount(fullProps);
    const textArea = findTextArea(componentMount);

    const json = JSON.parse(fullProps.item.advJson);
    json.type = "should-be-cleared";
    textArea.props().onBlur({ target: { value: JSON.stringify(json) } });
    delete json.type;

    expect(triggerUpdateMock.mock.calls.length).toBe(1);
    expect(triggerUpdateMock).toBeCalledWith(jsonObjectToString(json));

  });

});