import SpreadsheetDao from './spreadsheet_dao';
import PapaParse from 'papaparse';
import {
  formatJSON
} from '../services/sheetServices.js';
import { sanitizeFieldName } from '../services/services';

export default class PapaParseDao extends SpreadsheetDao {
  constructor(file) {
    super(file);
    this.streamFile = this.file.size > this.largeFile;
    // Estimating due to bug :- https://github.com/mholt/PapaParse/issues/321
    this.sizePerDoc = 0; // Used to approximate ETA
    this.estimatedCursor = 0;
    // Estimation - End
  }

  _setMetaFields(meta) {
    this.headerRow = meta.fields;
    this.headerRowSanitized = meta.fields.map((field) => sanitizeFieldName(field));
  }
  _instantiationDone(done, result) {
    if (!this._instantiated) {
      this._instantiated = true;
      const {
        data,
        errors,
        meta
      } = result;
      if (!this.streamFile) {
        this.sheetJson = formatJSON(data);
        this._setMetaFields(meta);
      } else {
        this.sheetJson = formatJSON(this.sheetJson);
      }
      done(errors);
    }
  };

  _previewStepper = (result, parser) => {
    if (result.data) {
      this.sheetJson.push(result.data[0]);
      if (this.sheetJson.length >= this.previewSize) {
        this._setMetaFields(result.meta);
        parser.abort();
      }
    }
  };

  instantiate(done, config) {
    PapaParse.SCRIPT_PATH = this.xlsxClient.getPapaParseDistPath();
    const parserConfig = {
      skipEmptyLines: true,
      dynamicTyping: true,
      header: true,
      skipEmptyLines: true,
      complete: (result) => this._instantiationDone(done, result),
      worker: config && config.worker === false ? false : true
    };
    if (this.streamFile) {
      this.sheetJson = [];
      this.sheetJsonBatch = [];
      parserConfig.preview = this.previewSize;
      parserConfig.step = this._previewStepper;
    }
    PapaParse.parse(this.file, parserConfig);
  }

  getSheetNames() {
    return [{
      value: this.fileName,
      text: this.fileName
    }];
  }

  sheetExists(sheetName) {
    return this.fileName === sheetName;
  }

  getHeaderRow(sheetName, mapped = true) {
    if (mapped) {
      return this.headerRowSanitized.map((s) => ({
        field: s,
        name: s,
        truncateText: true
      }));
    }
    return this.headerRowSanitized;
  }

  _resolveFieldType(fieldName) {
    let currentRow = 0;
    while (currentRow < this.sheetJson.length && currentRow < this.maxRowsToParse) {
      const cellValue = this.sheetJson[currentRow][fieldName];
      switch (typeof cellValue) {
        case 'string':
          return 'keyword';
        case 'number':
          return 'float';
        case 'boolean':
          return 'boolean';
        case 'object':
          if (cellValue && cellValue.constructor.name === 'Date') {
            return 'date';
          }
      }
      currentRow++;
    }
    return 'keyword';
  }

  getHeaderWithType(sheetName) {
    const headersWithType = [];
    for (let i = 0; i < this.headerRow.length; i++) {
      const header = {
        name: this.headerRow[i]
      };
      header.type = this._resolveFieldType(this.headerRowSanitized[i]);
      headersWithType.push(header);
    }
    return headersWithType;
  }

  sheetToJSON(sheetName, previewSize) {
    if (previewSize) {
      return this.sheetJson.slice(0, Math.min(this.sheetJson.length, previewSize));
    }
    return this.sheetJson;
  }

  _stepperFunction = async (indexingConfig, progressCallback, promise, ingestionResult, parserResult, parser) => {
    const {
      bulkSize,
      columnFilter
    } = indexingConfig;
    this.sheetJsonBatch.push(parserResult.data[0]);
    ingestionResult.totalDocuments++;
    if (this.sheetJsonBatch.length < bulkSize) {
      return;
    }
    if (this.simultaneousBulkRequests <= 0) {
      if (!this.sizePerDoc) {
        this.sizePerDoc = parseInt(parserResult.meta.cursor / ingestionResult.totalDocuments);
      }
      parser.pause();
    }
    this.simultaneousBulkRequests--;
    const filteredSheetJsonBatch = this._getFilteredSheetJson(formatJSON(this.sheetJsonBatch), columnFilter);
    this.sheetJsonBatch = [];
    const bulkRequestBatch = this._createBulkRequest(filteredSheetJsonBatch, indexingConfig);
    const reject = (error) => {
      ingestionResult.jobFailed = true;
      parser.abort();
      promise.reject(error);
    };
    const resolve = (ingestionResult) => {
      parser.abort();
      promise.resolve(ingestionResult);
    };
    this.estimatedCursor += this.sizePerDoc * bulkSize; // Do Not care about the last iteration
    !ingestionResult.jobFailed && progressCallback({
      progress: (this.estimatedCursor / this.file.size) * 100,
      stats: {
        documentsIngested: ingestionResult.totalDocuments,
        fileParsed: {
          totalSize: this.file.size,
          cursor: this.estimatedCursor || parserResult.meta.cursor
        }
      },
      errorsDuringIndexing: ingestionResult.errorsDuringIndexing
    });
    await this._ingestBatchToES(indexingConfig, bulkRequestBatch[0], ingestionResult, progressCallback, resolve, reject);
    this.simultaneousBulkRequests++;
    if (parser.paused()) {
      parser.resume();
    }
  };

  _streamCompleted = async (indexingConfig, progressCallback, promise, ingestionResult) => {
    const {
      bulkSize,
      columnFilter
    } = indexingConfig;
    if (ingestionResult.jobFailed) {
      return;
    }
    if (this.sheetJsonBatch.length > 0 && this.sheetJsonBatch.length < bulkSize) {
      const filteredSheetJsonBatch = this._getFilteredSheetJson(formatJSON(this.sheetJsonBatch), columnFilter);
      this.sheetJsonBatch = [];
      const bulkRequestBatch = this._createBulkRequest(filteredSheetJsonBatch, indexingConfig);
      await this._ingestBatchToES(indexingConfig, bulkRequestBatch[0], ingestionResult, progressCallback, promise.resolve,
        promise.reject);
      progressCallback({
        progress: 100,
        stats: {
          documentsIngested: ingestionResult.totalDocuments,
          fileParsed: {
            totalSize: this.file.size,
            cursor: this.estimatedCursor
          }
        },
        errorsDuringIndexing: ingestionResult.errorsDuringIndexing
      });
    }
    promise.resolve(ingestionResult);
  };

  _ingestStreamToES(indexingConfig, progressCallback) {
    this.sizePerDoc = 0;
    this.estimatedCursor = 0;
    const ingestionResult = {
      totalDocuments: 0,
      errorsDuringIndexing: []
    };
    return new Promise((resolve, reject) => {
      const parserConfig = {
        skipEmptyLines: true,
        dynamicTyping: true,
        header: true,
        step: (parserResult, parser) =>
          this._stepperFunction(indexingConfig, progressCallback, { resolve, reject },
            ingestionResult, parserResult, parser),
        complete: () => this._streamCompleted(indexingConfig, progressCallback, { resolve, reject },
          ingestionResult),
        worker: false
      };
      PapaParse.parse(this.file, parserConfig);
    });
  };

  _ingestSheetToES(sheetName, indexingConfig, progressCallback) {
    try {
      if (this.streamFile) {
        return this._ingestStreamToES(indexingConfig, progressCallback);
      }
      const filteredSheetJson = this._getFilteredSheetJson(this.sheetToJSON(sheetName), indexingConfig.columnFilter);
      const bulkRequestObj = this._createBulkRequest(filteredSheetJson, indexingConfig);
      return this._ingestToES(filteredSheetJson, bulkRequestObj, indexingConfig, progressCallback);
    } catch (error) {
      return Promise.reject(error);
    }
  }

};