import {
  createBulk
} from '../services/services.js';
import XLSXClient from '../services/xlsx_client.js';
import { configService } from '../../../shared_components';
import _ from 'lodash';
const CLUSTER_BLOCK_EXCEPTION = 'cluster_block_exception';
const BLOCKING_EXCEPTION_STATUS = 403;

/**
 * The base class for spreadsheet parsers.
 */
export default class SpreadsheetDao {
  constructor(file) {
    if (!file) {
      throw new Error('No file/undefined passed to the constructor');
    }
    this._methodNotDefined = 'This method should be implemented by subclass.';
    this.file = file;
    this.fileName = file.name.substr(0, file.name.lastIndexOf('.'));
    this.fileExtension = file.name.split('.').pop();
    this.xlsxClient = XLSXClient.getInstance();
    this._resetFlags();
    this._initializeConfig();
  }

  _initializeConfig() {
    this.previewSize = configService().previewSize;
    this.largeFile = configService().largeFile;
    this.maxRowsToParse = configService().maxRowsToParse; // To Deetermine cell type
  }

  _resetFlags() {
    this._resetBulkRequestCounter();
    this._resetHalt();
  }

  _resetHalt() {
    this.continueIngestion = true;
  }

  _shouldContinueIngestion() {
    return this.continueIngestion === true;
  }

  _resetBulkRequestCounter() {
    this.simultaneousBulkRequests = configService().simultaneousBulkRequests;
  }

  _getFilteredSheetJson(sheetJson, columnFilter) {
    if (columnFilter && columnFilter.length > 0) {
      sheetJson.map((tuple) =>
        columnFilter.forEach((fieldToFilter) => delete tuple[fieldToFilter])
      );
    }
    return sheetJson;
  }

  _createBulkRequest(sheetJson, indexingConfig) {
    const {
      bulkSize,
      username,
      customIdModel,
      delimiters,
      mappingInfo
    } = indexingConfig;
    return createBulk(sheetJson, customIdModel, bulkSize, username, (delimiters && delimiters.length > 0), mappingInfo);
  }

  async _ingestBatchToES(indexingConfig, bulkRequestBatch, result, progressCallback, resolve, reject) {
    if (this._shouldContinueIngestion()) {
      try {
        const response = await this.xlsxClient.bulkIndex(indexingConfig.indexName, bulkRequestBatch,
          indexingConfig.pipelineId);
        if (response.errors) {
          result.errorsDuringIndexing = result.errorsDuringIndexing.concat(response.items.filter((ele) => ele.index.error));
          const indexResponse = response.items[0].index;
          if (indexResponse.status === BLOCKING_EXCEPTION_STATUS ||// CBE is always 403, but just in case the
            indexResponse.error && indexResponse.error.type === CLUSTER_BLOCK_EXCEPTION) {// behaviour changes
            resolve(result);
          }
        }
        else if (response.error !== undefined) {
          progressCallback({ networkError: true });
          reject(response.error.message);
          return;
        }
      } catch (error) {
        reject('An error occurred: ' + error + '\n' + _.get(error, 'response.data.message'));
      }
    } else {
      reject('Aborted by user!');
    }
  }

  _ingestToES(filteredSheetJson, bulkRequestObj, indexingConfig, progressCallback) {
    return new Promise(async (resolve, reject) => {
      const result = {
        totalDocuments: filteredSheetJson.length,
        errorsDuringIndexing: []
      };
      let documentsIngested = 0;
      let promises = [];
      let stopIngestion = false;
      const resolveFn = (result) => {
        stopIngestion = true;
        resolve(result);
      };
      const rejectFn = (reason) => {
        stopIngestion = true;
        reject(reason);
      };
      for (let i = 0; i < bulkRequestObj.length; i++) {
        promises.push(this._ingestBatchToES(indexingConfig, bulkRequestObj[i], result, progressCallback, resolveFn, rejectFn));
        documentsIngested += bulkRequestObj[i].length / 2;
        progressCallback({
          progress: (i / (bulkRequestObj.length - 1)) * 100,
          stats: {
            documentsIngested,
            totalDocuments: result.totalDocuments
          },
          errorsDuringIndexing: result.errorsDuringIndexing
        });
        this.simultaneousBulkRequests--;
        if (this.simultaneousBulkRequests <= 0 || i >= bulkRequestObj.length - 1) {
          await Promise.all(promises);
          promises = [];
          this._resetBulkRequestCounter();
          if (stopIngestion) {
            return;
          }
        }
      }
      resolve(result);
    });
  }

  instantiate(callback, config) {
    throw new Error(this._methodNotDefined);
  }

  getSheetNames() {
    throw new Error(this._methodNotDefined);
  }

  sheetExists(sheetName) {
    throw new Error(this._methodNotDefined);
  }

  getHeaderRow() {
    throw new Error(this._methodNotDefined);
  }

  getHeaderWithType(sheetName) {
    throw new Error(this._methodNotDefined);
  }

  sheetToJSON(sheet, previewSize) {
    throw new Error(this._methodNotDefined);
  }

  _ingestSheetToES(sheetName, indexingConfig, progressCallback) {
    throw new Error(this._methodNotDefined);
  }

  ingestSheetToES(sheetName, indexingConfig, progressCallback) {
    this._resetHalt();
    return this._ingestSheetToES(sheetName, indexingConfig, progressCallback);
  }

  haltIngestion() {
    this.continueIngestion = false;
  }
};