import React from 'react';
import { Prompt } from 'react-router';
import {
  EuiStepsHorizontal,
  EuiPageBody,
  EuiPageContent,
  EuiPageContentHeader,
  EuiPageContentBody
} from '@elastic/eui';
import StepOne from './components/stepOne.js';
import StepTwo from './components/stepTwo.js';
import StepThree from './components/stepThree.js';
import { getUpdatedMappingTableState } from './services/services';
import { getHeaderRow, getHeaderWithType, setESIndexName } from './services/sheetServices';
import Header from '../header';
import {
  confirmDestructModal
} from '../../shared_components';

import './less/main.less';
import '@elastic/eui/dist/eui_theme_k6_light.css';

//const supportedFileType = ['xlsx', 'csv'];    // kibi: Not using it right now
const STEP_FILE = 'choose-file';
const STEP_CONFIGURE = 'configure-indexing';
const STEP_DONE = 'done';

class Excel extends React.Component {
  constructor(props) {
    document.title = 'Excel/CSV Import - Siren';
    super(props);
    const steps = [
      {
        id: STEP_FILE,
        name: 'Choose a file',
        disabled: false
      },
      {
        id: STEP_CONFIGURE,
        name: 'Configure Indexing',
        disabled: true,
      },
      {
        id: STEP_DONE,
        name: 'Done!',
        disabled: true,
      }
    ];
    this.state = {
      currentStep: STEP_FILE,
      steps,
      abortConfirmFunction: null,
      hasNextStep: true,
      //Step File
      [STEP_FILE]: this.getInitialFileStepState(),
      //Step Configure
      [STEP_CONFIGURE]: this.getInitialConfigureStepState()
    };
    this.onStepValueChange = this.onStepValueChange.bind(this);
    this.onStepStateUpdate = this.onStepStateUpdate.bind(this);
  }

  onBeforeUnload() {
    return 'Are you sure? Your changes to current import configuration will be lost if you leave the page';
  }

  //Shameless polling mechanism for mapping table. Maybe fix it without performance implications?
  updateMappingInfo() {
    if (this.state[STEP_CONFIGURE].switchMap.value) {
      this.onStepValueChange(STEP_CONFIGURE, 'mappingInfo',
        getUpdatedMappingTableState(this.state[STEP_CONFIGURE].mappingInfo,
          this.state[STEP_CONFIGURE].selectedAnonOptions)
      );
    }
  }

  getInitialFileStepState() {
    return {
      workbook: {},
      filename: '',
      sheetname: '',
      configOptions: [],
      selectedConfig: '',
      selectItem: {
        options: [{ value: '', text: 'Select a sheet' }]
      },
      data: {
        loaded: false,
        items: [],
        columns: []
      },
      disableButton: true,
      loading: false,
      error: '',
      fileErrors: []
    };
  }

  getInitialConfigureStepState() {
    const pipelineInit = {
      pipelineMethod: 'skip',
      pipeline: '',
      pipelineInput: 0,
      pipelineResult: ''
    };
    const savedConfigInit = {
      id: '',
      name: '',
      description: ''
    };
    return {
      pipelineConfig: pipelineInit,
      indexName: '',
      savedConfiguration: savedConfigInit,
      indexNameError: false,
      blockIngestion: false,
      indexExists: false,
      deleteIndexBeforeImport: false,
      indexNameErrors: [],
      options: [],
      delimiters: [],
      selectedAnonOptions: [],
      kbnId: {
        useKbnId: false,
        model: '',
        preview: ''
      },
      uploadButton: {
        text: 'Import',
        loading: false
      },
      switchMap: {
        value: true
      },
      header: [],
      items: [],
      mappingInfo: [],
      examples: {}
    };
  }
  stepChangeHook(newStepIndex, clickedNext) {
    //Do anything that should be done before a step change.
    if (this.state.currentStep === STEP_CONFIGURE) {
      this.updateMappingInfo();
    }
    if (this.state.steps[newStepIndex].id === STEP_CONFIGURE && clickedNext) {
      const sheetname = this.state[STEP_FILE].sheetname;
      const header = this.state[STEP_FILE].workbook.getHeaderRow(sheetname, false);
      const items = this.state[STEP_FILE].workbook.getHeaderWithType(sheetname);
      items.forEach((ele, index) => {
        items[index].sanitizedFieldName = header[index];
      });
      return {
        ...this.getInitialConfigureStepState(),
        ...{
          indexName: setESIndexName(this.state[STEP_FILE].filename),
          header,
          items,
          examples: this.state[STEP_FILE].data.items
        }
      };
    } else if (this.state.steps[newStepIndex].id === STEP_DONE && clickedNext) {
      return {
        indexName: this.state[STEP_CONFIGURE].indexName,
        sheetName: this.state[STEP_FILE].sheetname,
        fileName: this.state[STEP_FILE].filename
      };
    }
    return {};
  }

  goToStep(stepId) {
    const _goToStep = () => {
      const nextStep = this.state.steps.findIndex(s => s.id === stepId);
      if (this.state.steps[nextStep].disabled) {
        return;
      }
      this.stepChangeHook(nextStep);
      this.setState({
        currentStep: stepId,
        hasNextStep: this.hasNextStep(nextStep)
      });
    };
    return () => {
      if (this.state.currentStep === STEP_CONFIGURE && this.state[STEP_CONFIGURE].uploadButton.loading) {
        this.setState({ abortConfirmFunction: _goToStep });
        return;
      }
      _goToStep();
    };
  }

  previousStep = () => {
    const previousStep = this.state.steps.findIndex(s => s.id === this.state.currentStep) - 1;
    if (previousStep >= 0) {
      this.stepChangeHook(previousStep, false);
    }
    this.setState(state => {
      return {
        currentStep: state.steps[previousStep].id,
      };
    });
    return;
  }

  hasNextStep(id) {
    return id <= this.state.steps.length - 1;
  }

  updateStepsOnNext(steps, nextStepIndex) {
    steps[nextStepIndex - 1].isComplete = true;
    steps[nextStepIndex].disabled = false;
    steps[nextStepIndex].isComplete = false;

    if (steps[nextStepIndex].id === STEP_DONE) {
      steps[nextStepIndex].isComplete = true;
    }

    for (let i = nextStepIndex + 1; i < steps.length; i++) {
      steps[i].disabled = true;
      steps[i].isComplete = false;
    }
    return steps;
  }

  nextStep = (passToNextState = {}) => {
    const nextStep = this.state.steps.findIndex(s => s.id === this.state.currentStep) + 1;
    passToNextState = {
      ...this.stepChangeHook(nextStep, true),
      ...passToNextState
    };
    this.setState((state => {
      const steps = this.updateStepsOnNext(this.state.steps, nextStep);
      const NEXT_STEP = steps[nextStep].id;
      return {
        currentStep: state.steps[nextStep].id,
        steps: this.updateStepsOnNext(this.state.steps, nextStep),
        hasNextStep: this.hasNextStep(nextStep),
        [NEXT_STEP]: { ...state[NEXT_STEP], ...passToNextState }
      };
    }));
  }

  onStepValueChange(step, fieldName, value) {
    this.setState(state => {
      const stepState = state[step];
      stepState[fieldName] = value;
      return { [step]: stepState };
    });
  }

  onStepStateUpdate(step, stepState) {
    if (typeof stepState === 'function') {
      stepState = stepState.bind(null, step);
      this.setState(stepState);
      return;
    }
    this.setState(state => {
      const updatedState = { ...state[step], ...stepState };
      return { [step]: updatedState };
    });
  }

  renderStepFile() {
    if (!this.stepFileOnStepValueChange) {
      this.stepFileOnStepValueChange = this.onStepValueChange.bind(this, STEP_FILE);
      this.stepFileOnStepStateUpdate = this.onStepStateUpdate.bind(this, STEP_FILE);
    }
    return (
      <StepOne
        nextStep={this.nextStep}
        state={this.state[STEP_FILE]}
        onFieldValueChange={this.stepFileOnStepValueChange}
        stateUpdate={this.stepFileOnStepStateUpdate}
      />
    );
  }

  renderAbortConfirmationModel() {
    if (this.state.abortConfirmFunction) {
      const title = 'Are you sure?';
      const body = (
        <div>
          <p>Leaving the page would cause current import to abort!</p>
        </div>
      );
      const onCancel = () => {
        this.setState({ abortConfirmFunction: null });
      };
      const onConfirm = () => {
        this.state.abortConfirmFunction();
        onCancel();
      };
      return confirmDestructModal(title, body, onCancel, onConfirm, 'Leave page');
    }
    return '';
  }

  renderStepConfigure() {
    const sheetname = this.state[STEP_FILE].sheetname;
    if (!this.stepConfigureOnStepValueChange) {
      this.stepConfigureOnStepValueChange = this.onStepValueChange.bind(this, STEP_CONFIGURE);
      this.stepConfigureOnStepStateUpdate = this.onStepStateUpdate.bind(this, STEP_CONFIGURE);
    }
    return (
      <StepTwo
        state={this.state[STEP_CONFIGURE]}
        onFieldValueChange={this.stepConfigureOnStepValueChange}
        stateUpdate={this.stepConfigureOnStepStateUpdate}
        nextStep={this.nextStep}
        workbook={this.state[STEP_FILE].workbook}
        sheetname={sheetname}
        previousStep={this.previousStep}
        injector={this.props.injector}
        onBeforeUnload={this.onBeforeUnload}
      />
    );
  }

  renderStepDone() {
    return (
      <StepThree
        state={this.state[STEP_DONE]}
      />
    );
  }

  renderCurrentStep() {
    switch (this.state.currentStep) {
      case STEP_FILE:
        return this.renderStepFile();
      case STEP_CONFIGURE:
        return this.renderStepConfigure();
      case STEP_DONE:
        return this.renderStepDone();
      default:
        return this.renderStepFile();
    }
  }

  renderHorizontalSteps() {
    const horizontalSteps = this.state.steps.map((step) => {
      return {
        onClick: this.goToStep(step.id),
        title: step.name,
        isSelected: this.state.currentStep === step.id,
        isComplete: step.isComplete,
        disabled: step.disabled
      };
    });
    return (
      <EuiStepsHorizontal
        steps={horizontalSteps}
        style={{ backgroundColor: 'white' }}
      />
    );
  }

  renderApp() {
    return (
      <React.Fragment>
        <Prompt
          when={!!this.state[STEP_CONFIGURE].indexName && !this.state[STEP_CONFIGURE].savedConfiguration.id
            && !this.state[STEP_DONE]}
          message={this.onBeforeUnload()}
        />
        <EuiPageBody>
          {this.renderAbortConfirmationModel()}
          <EuiPageContent>
            <EuiPageContentHeader>
              {this.renderHorizontalSteps()}
            </EuiPageContentHeader>
            <EuiPageContentBody>
              {this.renderCurrentStep()}
            </EuiPageContentBody>
          </EuiPageContent>
        </EuiPageBody>
      </React.Fragment>
    );
  }

  render() {
    return (
      <React.Fragment>
        <Header
          sections={this.props.sections}
          currentSection={this.props.currentSection}
        />
        {this.renderApp()}
      </React.Fragment>
    );
  }

}

export {
  Excel
};