import Boom from 'boom';
import Joi from 'joi';
import {
  DataModelGenerator,
  IngestionJobCreator,
  pendingJobsCache
} from './module_neo4j';

export default function (server, dataCluster, apiPath) {
  const BASE_PATH = `${apiPath}neo4j/`;

  // Get pending jobs
  server.route({
    path: `${BASE_PATH}pendingJobs`,
    method: 'GET',
    handler: function (req, reply) {
      const authUser = req.auth.credentials;
      reply(pendingJobsCache.getJobs(authUser && authUser.username));
    }
  });

  // Get job
  server.route({
    path: `${BASE_PATH}job/{jobId}`,
    method: 'GET',
    handler: function (req, reply) {
      const authUser = req.auth.credentials;
      reply(pendingJobsCache.getJob(authUser && authUser.username, req.params.jobId) ||
        Boom.notFound(`No pending job found with Job Id: ${req.params.jobId}`));
    }
  });

  //Create Ingestion Jobs
  server.route({
    path: `${BASE_PATH}createJobs`,
    method: 'POST',
    handler: function (req, reply) {
      const { jobs, datasourceId, neo4jDocumentField } = req.payload;
      const ingestionJobCreator = new IngestionJobCreator(jobs, datasourceId, neo4jDocumentField, dataCluster, req);
      ingestionJobCreator.createIngestionJobs()
        .then(reply)
        .catch(err => {
          reply({
            acknowledged: false,
            error: err.toString()
          });
        });
    },
    config: {
      validate: {
        payload: {
          neo4jDocumentField: Joi.string().max(75).required(),
          datasourceId: Joi.string().max(150).required(),
          jobs: Joi.array().items(
            Joi.object().keys({
              nodeName: Joi.string().required(),
              cypherQuery: Joi.string().required(),
              targetIndex: Joi.string().max(255).required(),
              jobName: Joi.string().required(),
              relations: Joi.array().items(
                Joi.object().keys({
                  startNode: Joi.string().required(),
                  endNode: Joi.string().required()
                })
              )
            }).required()
          ).required()
        }
      }
    }
  });

  //Create Ontology for Ingested data
  server.route({
    path: `${BASE_PATH}createOntology`,
    method: 'POST',
    handler: function (req, reply) {
      const { indexEntities, jobId } = req.payload;
      const datamodelGenerator = new DataModelGenerator(indexEntities, server, jobId, req.auth.credentials);
      datamodelGenerator.generateDataModel()
        .then(reply)
        .catch(err => {
          reply({
            acknowledged: false,
            error: err.toString()
          });
        });
    },
    config: {
      validate: {
        payload: {
          jobId: Joi.string().required(),
          indexEntities: Joi.object().pattern(/./,
            Joi.object().keys({
              jobId: Joi.string().required(), // Id of reflection job
              savedSearch: Joi.string().required(),
              es: Joi.string().max(255).required(),
              label: Joi.string().required(),
              inverseLabel: Joi.string(),
              color: Joi.string().required(),
              icon: Joi.string().required(),
              nodeName: Joi.string().required(),
              relations: Joi.array().items(
                Joi.object().keys({
                  startNode: Joi.string().required(),
                  endNode: Joi.string().required()
                })
              )
            })
          ).required()
        }
      }
    }
  });
}