import d3 from 'd3';
import $ from 'jquery';
import _ from 'lodash';
import angular from 'angular';
import { uiModules } from 'ui/modules';
import { VisColorMappedColorsProvider } from 'ui/vis/components/color/mapped_colors';

const module = uiModules.get('kibana/scatterplot_vis');

module.directive('scatterPlot', function (Private, createNotifier) {
  return {
    scope: {
      options: '=',
      data: '='
    },
    restrict: 'E',
    link: _link
  };

  function _link($scope, $element, attrs) {
    let circles;
    const notify = createNotifier({
      location: 'Scatterplot'
    });
    const mappedColors = Private(VisColorMappedColorsProvider);

    $scope.$watch('data', function (newData, oldData) {
      _render(newData);
    });

    $scope.$on('scatterplot:redraw', function () {
      _render($scope.data);
    });

    function _render(responseData) {
      if (!responseData) return;

      const $chart = angular.element($element[0]).find('svg');
      if ($chart.length > 0) {
        $chart[0].remove();
      }

      const data = responseData.data;
      if (data.length === 0) return;

      try {
        // Define the div for the tooltip
        let divTooltip = d3.select('body').select('div[class="scatterplot_tooltip"]');

        if ($(divTooltip[0])[0] === null) {
          divTooltip = d3.select('body').append('div')
            .attr('class', 'scatterplot_tooltip')
            .style('opacity', 0);
        }

        let x;
        let	y;

        switch ($scope.options.xField.type) {
          case 'string':
            x = d3.scale.ordinal();
            x.domain(data.sort(function (a, b) {
              return d3.ascending(a.x, b.x);
            }));
            break;
          case 'date':
            x = d3.time.scale();
            break;
          default:
            if ($scope.options.xAxisScale === 'linear') {
              x = d3.scale.linear();
            } else if ($scope.options.xAxisScale === 'log') {
              x = d3.scale.log();
            } else if ($scope.options.xAxisScale === 'square root') {
              x = d3.scale.sqrt();
            }
        }

        switch ($scope.options.yField.type) {
          case 'string':
            y = d3.scale.ordinal();
            y.domain(data.sort(function (a, b) {
              return d3.ascending(a.y, b.y);
            }));
            break;
          case 'date':
            y = d3.time.scale();
            break;
          default:
            if ($scope.options.yAxisScale === 'linear') {
              y = d3.scale.linear();
            } else if ($scope.options.yAxisScale === 'log') {
              y = d3.scale.log();
            } else if ($scope.options.yAxisScale === 'square root') {
              y = d3.scale.sqrt();
            }
        }

        let dotSizeScale;
        if ($scope.options.dotSizeScale === 'linear') {
          dotSizeScale = d3.scale.linear();
        } else if ($scope.options.dotSizeScale === 'log') {
          dotSizeScale = d3.scale.log();
        } else if ($scope.options.dotSizeScale === 'square root') {
          dotSizeScale = d3.scale.sqrt();
        }

        let xJitterScale;
        if ($scope.options.xJitterFieldName) {
          if ($scope.options.xJitterField.type === 'string') {
            xJitterScale = d3.scale.ordinal();
          } else {
            if ($scope.options.xJitterScale === 'linear') {
              xJitterScale = d3.scale.linear();
            } else if ($scope.options.xJitterScale === 'log') {
              xJitterScale = d3.scale.log();
            } else if ($scope.options.xJitterScale === 'square root') {
              xJitterScale = d3.scale.sqrt();
            }
          }
        }

        const colorSet = _.uniq(_.map(data, d => d.color));
        mappedColors.mapKeys(colorSet);

        // set the stage
        const margin = {t:30, r:30, b:10, l:100 };
        const legendWidth = $scope.options.colorFieldName !== '' ? margin.l + margin.r - 5 : 0;
        // if the legend was created keep some space on rendering of SVG by removing of legendWidth
        // to render scatterplot without 100% filling and add legend without overlapping later
        const parent = $element.parent();
        let w = parent.width() - 20 - margin.l - margin.r - legendWidth;
        let h = parent.height() - 20 - margin.t - margin.b;
        if (w < 0) w = 600;
        if (h < 0) h = 500;

        if ($scope.options.xField.type !== 'string') {
          x.range([0, w]);
        } else {
          x.rangeBands([0, w], 0);
        }
        if ($scope.options.yField.type !== 'string') {
          y.range([h - 60, 0]);
        } else {
          y.rangeBands([h - 60, 0], 1);
        }

        dotSizeScale.range([2, 20]);

        if ($scope.options.xJitterFieldName) {
          if ($scope.options.xJitterField.type !== 'string') {
            xJitterScale.range([1, 20]);
          } else {
            xJitterScale.rangeBands([0, w], 0);
          }
        }

        const svg = d3.select($element[0]).append('svg')
        .attr('width', w + margin.l + margin.r)
        .attr('height', h + margin.t + margin.b);

      	// set axes, as well as details on their ticks
        const xAxis = d3.svg.axis()
    		.scale(x)
    		.ticks(10)
        .tickFormat($scope.options.xField.format.getConverterFor('text'))
    		.tickSubdivide(true)
        .tickSize(6, 3)
        .orient('bottom');

        const yAxis = d3.svg.axis()
    		.scale(y)
    		.ticks(10)
        .tickFormat($scope.options.yField.format.getConverterFor('text'))
    		.tickSubdivide(true)
    	  .tickSize(6, 3)
    		.orient('left');

        if ($scope.options.xField.type !== 'string') {
          x.domain([
            d3.min(data, function (d) { return d.x; }),
            d3.max(data, function (d) { return d.x; })]
          );
          if ($scope.options.xAxisScale === 'log' && x.domain()[0] <= 0) {
            x.domain()[0] = 1;
            x = x.nice();
          }
        } else {
          x.domain(data.map(function (d) { return d.x; }));
        }

        if ($scope.options.yField.type !== 'string') {
          y.domain([
            d3.min(data, function (d) { return d.y; }),
            d3.max(data, function (d) { return d.y; })]
          );
          if ($scope.options.yAxisScale === 'log' && y.domain()[0] <= 0) {
            y.domain()[0] = 1;
            y = y.nice();
          }
        } else {
          y.domain(data.map(function (d) { return d.y; }));
        }

        dotSizeScale.domain([
          d3.min(data, function (d) { return d.size; }),
          d3.max(data, function (d) { return d.size; })]
        );
        if ($scope.options.dotSizeScale === 'log' && dotSizeScale.domain()[0] <= 0) {
          dotSizeScale.domain()[0] = 1;
          dotSizeScale = dotSizeScale.nice();
        }

        if ($scope.options.xJitterFieldName) {
          if ($scope.options.xJitterField.type !== 'string') {
            xJitterScale.domain([
              d3.min(data, function (d) { return d.xJitter; }),
              d3.max(data, function (d) { return d.xJitter; })]
            );
            if ($scope.options.xJitterScale === 'log' && xJitterScale.domain()[0] <= 0) {
              xJitterScale.domain()[0] = 1;
              xJitterScale = xJitterScale.nice();
            }
          } else {
            const sortedData = data.sort((a, b) => {
              if (!a.xJitter || _.isNull(a.xJitter)) return -1;
              if (!b.xJitter || _.isNull(b.xJitter)) return 1;
              return d3.ascending(a.xJitter.toUpperCase(), b.xJitter.toUpperCase());
            });
            xJitterScale.domain(sortedData.map(function (d) { return d.xJitter; }));
            _.forEach(sortedData, d => {
              xJitterScale(d.xJitter);
            });
          }
        }

        // y Axis margin adjust to label data length
        let maxCharLength = 0;
        y.domain().forEach(function (domain) {
          if ($scope.options.yField.type !== 'date' && domain.toString().length > maxCharLength) {
            maxCharLength = domain.toString().length;
          }
        });
        if (maxCharLength > 15) {
          margin.l += (maxCharLength - 15) * 5;
          if (margin.l > 200) margin.l = 200;
        }

        // group that will contain all of the plots
        const groups = svg.append('g')
        .attr('transform', 'translate(' + margin.l + ',' + margin.t + ')');

        if ($scope.options.aggMode === 'Straight data') {
          const brush = d3.svg.brush()
              .x(x)
              .y(y)
              .on('brushend', brushended); // eslint-disable-line memoryleaks

          groups.append('g')
            .attr('class', 'brush')
            .call(brush);

          function brushended() {
            const isXOrdinalSerie = $scope.options.xField.type === 'string';
            const xSelected = x.domain().filter(function (d) {
              return (brush.extent()[0][0] <= x(d)) && (x(d) <= brush.extent()[1][0]);
            });
            const xRange = !isXOrdinalSerie ? [brush.extent()[0][0], brush.extent()[1][0]] : xSelected;

            const isYOrdinalSerie = $scope.options.yField.type === 'string';
            const ySelected = y.domain().filter(function (d) {
              return (brush.extent()[0][1] <= y(d)) && (y(d) <= brush.extent()[1][1]);
            });
            const yRange = !isYOrdinalSerie ? [brush.extent()[0][1], brush.extent()[1][1]] : ySelected;

            divTooltip.style('opacity', 0);
            $scope.$emit('scatterplot_select', [xRange, yRange]);
          }
        }

  	    // style the circles, set their locations based on data
        circles =
        groups.selectAll('circle')
        .data(data)
        .enter()
        .append('circle')
        .attr('class', 'circles')
        .attr('style', 'opacity: ' + $scope.options.shapeOpacity)
        .attr({
          cx: function (d) { return x.rangeBand ? _xJitterCalc(d) : x(safeX(d.x)); },
          cy: function (d) { return y(safeY(d.y)); },
          r: function (d) { return _dotSizeCalc(d); },
          _r: function (d) { return _dotSizeCalc(d); },
          id: function (d) { return d._id; }
        })
        .on('click', function () { // eslint-disable-line memoryleaks
          d3.event.stopPropagation();
          divTooltip.style('opacity', 0);
          $scope.$emit('scatterplot_select', this.id);
        });

        function safeX(value) {
          if ($scope.options.xAxisScale === 'log') {
            return value <= 0 ? 1 : value;
          } else {
            return value;
          }
        }

        function safeY(value) {
          if ($scope.options.yAxisScale === 'log') {
            return value <= 0 ? 1 : value;
          } else {
            return value;
          }
        }

        function _xJitterCalc(d) {
          if ($scope.options.xJitterFieldName) {
            const jitter = xJitterScale(d.xJitter);
            const bucketsSize = x.rangeBand() / 2;
            const bucketPadding = x.rangeBand() / 4;
            let jitterPercentage;
            let bucket;
            let offset;

            if ($scope.options.xJitterField.type === 'string') {
              const buckets = 7;
              const bucketSize = bucketsSize / buckets;
              jitterPercentage = Math.abs(jitter) / (xJitterScale(xJitterScale.domain()[xJitterScale.domain().length - 1]) + 1);
              bucket = Math.round(buckets * jitterPercentage);
              offset = bucketSize * bucket;
            } else {
              if (isNaN(jitter)) {
                return x(d.x) + x.rangeBand() / 2;
              }
              jitterPercentage = Math.abs(jitter) / xJitterScale(xJitterScale.domain()[1]);
              offset = bucketsSize * jitterPercentage;
            }
            return x(d.x) + bucketPadding + offset;
          } else {
            return x(d.x) + x.rangeBand() / 2;
          }
        }

        function _dotSizeCalc(d) {
          if ($scope.options.dotSizeFieldName !== '') {
            const size = dotSizeScale(d.size);
            if (isNaN(size)) return 0;
            return Math.abs(size);
          } else {
            return $scope.options.dotSize;
          }
        }

        if ($scope.options.colorFieldName !== '') {
          circles.style('fill', function (d) { return mappedColors.get(d.color); });
        } else if ($scope.options.color) {
          circles.style('fill', function (d) { return d3.rgb(d.color); });
        }

        // tooltips
        if ($scope.options.labelEnabled && !$scope.options.labelHoverEnabled) {
          groups.selectAll('labels')
          .data(data)
          .enter().append('text')
          .attr('class', 'labels')
          .attr('x', function (d) { return x(d.x) - 8; })
          .attr('y', function (d) { return y(d.y) - 12; })
          .attr('dy', '.35em')
          .text(function (d) { return d.label; });
        }

        // legends box
        if ($scope.options.colorFieldName !== '') {
          const regions = colorSet.slice(0, 10);
          const legendX = w + legendWidth;

          const legends = svg.append('g')
            .attr('class', 'legends');

          legends.append('rect')
            .attr('class', 'legend-box')
            .attr({
              x: legendX,
              y: 0,
              width: legendWidth,
              height: regions.length * 14 + 30
            });

          const legendBody = legends.append('g');

          // the legend color guide
          const legend = legendBody.selectAll('rect')
            .data(regions)
            .enter().append('circle')
            .attr({
              cx: legendX + 8,
              cy: function (d, i) { return (i * 14 + 25); },
              r: 5,
            })
            .style('fill', function (d) { return mappedColors.get(d); });

        	// legend labels
          legendBody.selectAll('text')
            .data(regions)
            .enter().append('text')
            .attr({
              x: legendX + 18,
              y: function (d, i) { return (i * 14 + 29); }
            })
            .text(function (d) {
              if (d === null || d === undefined) {
                return '';
              }

              const text = d.toString().trim();
              return  text.length > 18 ? text.slice(0, 16) + '...' : text;
            });

          legends
            .append('text')
            .attr({
              x: legendX + 8,
              y: 13,
            })
            .attr('class', 'color-legend')
            .text('Top 10 color codes');

          // update SVG container to give a space for the legend
          svg.attr('width', w + margin.l + margin.r + legendWidth);
        }

        // draw axes
        svg.append('g')
        .attr('class', 'x axis')
        .attr('transform', 'translate(' + margin.l + ',' + (h - 60 + margin.t) + ')')
        .call(xAxis)
        .selectAll('text')
          .attr('transform', 'rotate(90)')
          .attr('x', '10')
          .attr('y', '-4')
          .style('text-anchor', 'start');

        svg.append('g')
        .attr('class', 'y axis')
        .attr('transform', 'translate(' + margin.l + ',' + margin.t + ')')
        .call(yAxis);

        // draw x axis label
        if ($scope.options.xAxisLabel) {
          svg.append('text')
          .attr('class', 'x label')
          .attr('text-anchor', 'end')
          .attr('x', w + 50)
          .attr('y', h - margin.t - 45)
          .text($scope.options.xAxisLabel);
        }

        // draw y axis label
        if ($scope.options.yAxisLabel) {
          svg.append('text')
          .attr('class', 'y label')
          .attr('text-anchor', 'end')
          .attr('x', -30)
          .attr('y', margin.l + 5)
          .attr('dy', '.75em')
          .attr('transform', 'rotate(-90)')
          .text($scope.options.yAxisLabel);
        }

        // draw result stats
        if (responseData.totalHits > 0 && $scope.options.dataSize < responseData.totalHits && $scope.options.aggMode === 'Straight data') {
          const infoResult = svg.append('g')
            .attr('class', 'sample-result');

          infoResult.append('text')
            .attr('text-anchor', 'end')
            .attr({
              x: w + 30,
              y: h - margin.t
            })
            .text($scope.options.dataSize + ' samples out of ' + responseData.totalHits + ' total');

          infoResult.append('rect')
            .attr({
              x: w + 35,
              y: h - margin.t - 16,
              height: 20,
              width: 65,
              rx: 3,
              ry: 3
            })
            .on('click', function () { // eslint-disable-line memoryleaks
              d3.event.stopPropagation();
              $scope.$emit('scatterplot_refresh');
            });

          infoResult.append('text')
            .attr('class', 'button-text')
            .attr({
              x: w + 43,
              y: h - margin.t - 3
            })
            .text('Resample')
            .on('click', function () { // eslint-disable-line memoryleaks
              d3.event.stopPropagation();
              divTooltip.style('opacity', 0);
              $scope.$emit('scatterplot_refresh');
            });
        }

        // what happens when we enter a bubble?
        const mouseOn = function (d) {
          const circle = d3.select(this);

          let formattedLabel;
          if (d.label && $scope.options.labelField) {
            formattedLabel = $scope.options.labelField.format.convert(d.label, 'html');
          }

          if ($scope.options.labelEnabled &&
              $scope.options.labelHoverEnabled &&
              $scope.options.labelFieldName !== '') {
            let html = '';
            html += '<table>';
            html += '<tr><td colspan="2"><b>' + formattedLabel || d.label + '</b></td></tr>';
            html += '<tr><td><b>' + $scope.options.xFieldName + '</b></td><td>' + d.x + '</td></tr>';
            html += '<tr><td><b>' + $scope.options.yFieldName + '</b></td><td>' + d.y + '</td></tr>';
            if ($scope.options.colorFieldName && d.color) {
              html += '<tr><td><b>' + $scope.options.colorFieldName + '</b></td><td>' + d.color + '</td></tr>';
            }
            if ($scope.options.dotSizeFieldName && d.size) {
              html += '<tr><td><b>' + $scope.options.dotSizeFieldName + '</b></td><td>' + d.size + '</td></tr>';
            }
            if ($scope.options.xJitterFieldName && d.xJitter) {
              html += '<tr><td><b>' + $scope.options.xJitterFieldName + '</b></td><td>' + d.xJitter + '</td></tr>';
            }
            html += '</table>';

            divTooltip.transition()
              .duration(200)
              .style('opacity', .9);

            const el = divTooltip.html(html);
            let left = d3.event.pageX;
            let top = d3.event.pageY;
            const width = $(el[0]).width();
            const height = $(el[0]).height();
            const graphTop = ($element[0]).getBoundingClientRect().top - $element.parent().height() + document.body.scrollTop;
            const graphLeft = ($element[0]).getBoundingClientRect().left + $element.parent().width() + document.body.scrollLeft;
            const graphBottom = ($element[0]).getBoundingClientRect().bottom + document.body.scrollTop;

            if (left + width > graphLeft) {
              left = left - width;
              top = d3.event.pageY - height;
            }
            if (top + height > graphBottom - margin.t - margin.b - 50) {
              top = d3.event.pageY - height;
            }
            if (top < graphTop) {
              top = d3.event.pageY;
            }
            el.style('left', left + 'px');
            el.style('top', top + 'px');
          }

          // transition to increase size/opacity of bubble
          circle.transition()
      		.duration(800).style('opacity', 1)
      		.attr('r', (+circle.attr('_r')) * 1.5).ease('elastic');

          if (!isNaN(circle.attr('cx')) && !isNaN(circle.attr('cy'))) {
        		// append lines to bubbles that will be used to show the precise data points.
        		// translate their location based on margins
            svg.append('g')
          		.attr('class', 'guide')
          		.append('line')
          			.attr('x1', circle.attr('cx'))
          			.attr('x2', circle.attr('cx'))
          			.attr('y1', +circle.attr('cy'))
          			.attr('y2', h - margin.t - margin.b - 20)
          			.attr('transform', 'translate(' + margin.l + ',' + margin.r + ')')
          			.style('stroke', circle.style('fill'))
          			.transition()
                  .delay(200)
                  .duration(400)
                  .styleTween('opacity', function () {
                    return d3.interpolate(0, .5);
                  });

            svg.append('g')
              .attr('class', 'guide')
          		.append('line')
          			.attr('x1', +circle.attr('cx'))
          			.attr('x2', 0)
          			.attr('y1', circle.attr('cy'))
          			.attr('y2', circle.attr('cy'))
          			.attr('transform', 'translate(' + margin.l + ',' + margin.r + ')')
          			.style('stroke', circle.style('fill'))
          			.transition()
                  .delay(200)
                  .duration(400)
                  .styleTween('opacity', function () {
                    return d3.interpolate(0, .5);
                  });
          }
        };

      	// what happens when we leave a bubble?
        const mouseOff = function (d) {
          const circle = d3.select(this);

          if ($scope.options.labelEnabled && $scope.options.labelHoverEnabled) {
            divTooltip.transition()
              .duration(200)
              .style('opacity', 0);
          }

      		// go back to original size and opacity
          circle.transition()
      		.duration(800).style('opacity', $scope.options.shapeOpacity)
      		.attr('r', circle.attr('_r')).ease('elastic');

      		// fade out guide lines, then remove them
          d3.selectAll('.guide').transition()
            .duration(100)
            .styleTween('opacity', function () {
              return d3.interpolate(.5, 0);
            })
            .remove();
        };

        // run the mouseon/out functions
        circles.on('mouseover', mouseOn); // eslint-disable-line memoryleaks
        circles.on('mouseout', mouseOff); // eslint-disable-line memoryleaks
      }
      catch (error) {
        notify.error(error);
      }
    }

    $scope.$on('$destroy', () => {
      if (circles) {
        circles.on('mouseover', null); // eslint-disable-line memoryleaks
        circles.on('mouseout', null); // eslint-disable-line memoryleaks
      }
    });

  }

});
