'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _lodash = require('lodash');

var _elasticdump = require('elasticdump');

var _elasticdump2 = _interopRequireDefault(_elasticdump);

var _path = require('path');

var _request = require('request');

var _request2 = _interopRequireDefault(_request);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class Dump {
  constructor(config, backupDir) {
    this._config = config;
    this._backupDir = backupDir;
    this._elasticdumpOptions = {
      scrollTime: '1m',
      offset: 0,
      limit: 100
    };
    this._originalTlsRejectUnauthorizedValue = undefined;
    this.baseRequest = _request2.default.defaults({ headers: { 'User-Agent': 'elasticdump' } });
  }

  async getElasticsearchURL() {
    let url = (0, _lodash.get)(this._config, 'elasticsearch.url');

    // If no elasticsearch.url set in config, set it to a default
    if (!url) {
      const defaultURL = 'http://localhost:9220';
      console.log(`Using the default Elasticsearch URL (${defaultURL}) as elasticsearch.url is not set`);
      url = defaultURL;
    }

    if ((0, _lodash.has)(this._config, 'elasticsearch.username') && (0, _lodash.has)(this._config, 'elasticsearch.password')) {
      const [protocol, ...rest] = url.split('//');
      // If request is being set over SSL and verificationMode is not set to none, warn the user
      // that the certificates are not being examined in the TLS handshake
      if (protocol === 'https:') {
        if (this._config.elasticsearch.ssl.verificationMode !== 'none') {
          console.warn('SSL WARNING: The certificates of the cluster will not be verified during this operation');
        }

        // If process.env.NODE_TLS_REJECT_UNAUTHORIZED is not undefined, hold its value temporarily
        if (process.env.NODE_TLS_REJECT_UNAUTHORIZED) {
          this._originalTlsRejectUnauthorizedValue = process.env.NODE_TLS_REJECT_UNAUTHORIZED;
        }

        // Setting this to 0 means the certificates are not used in TLS authentication
        process.env.NODE_TLS_REJECT_UNAUTHORIZED = 0;
      }

      const username = (0, _lodash.get)(this._config, 'elasticsearch.username');
      const password = (0, _lodash.get)(this._config, 'elasticsearch.password');

      return `${protocol}//${username}:${password}@${rest.join('//')}`;
    }

    return url;
  }

  async deleteIndex(index) {
    const elasticsearchURL = await this.getElasticsearchURL();
    const deleteIndexURL = elasticsearchURL + '/' + index;
    const esRequest = {
      'url': deleteIndexURL,
      'method': 'DELETE'
    };

    return await this.baseRequest.del(esRequest);
  }

  async fromElasticsearchToFile(index, type) {
    const input = await this.getElasticsearchURL();
    const output = (0, _path.join)(this._backupDir, `${type}-${index}.json`);

    const options = (0, _lodash.merge)({
      'input-index': index,
      type,
      input,
      output
    }, this._elasticdumpOptions);

    await this._dump(input, output, options);
  }

  async fromFileToElasticsearch(index, type, useUpgradedFile) {
    const upgradedFragment = useUpgradedFile ? '-upgraded' : '';
    const input = (0, _path.join)(this._backupDir, `${type}-${index}${upgradedFragment}.json`);
    const output = await this.getElasticsearchURL();

    const options = (0, _lodash.merge)({
      'output-index': index,
      type,
      input,
      output
    }, this._elasticdumpOptions);
    await this._dump(input, output, options);
  }

  async _getDumpPromise(elasticdump) {
    return new _bluebird2.default(function (fulfill, reject) {
      elasticdump.dump(function (err) {
        if (err) {
          return reject(err);
        }
        fulfill();
      });
    });
  }

  async _dump(input, output, options) {
    const elasticdump = new _elasticdump2.default(input, output, options);
    elasticdump.on('log', message => console.log(message));
    elasticdump.on('error', message => console.error(message));

    try {
      await this._getDumpPromise(elasticdump);
    } finally {
      // once the request is complete, set the env variable back to its original value, if set
      // or just delete it if it was not set initially
      if (this._originalTlsRejectUnauthorizedValue !== undefined) {
        process.env.NODE_TLS_REJECT_UNAUTHORIZED = this._originalTlsRejectUnauthorizedValue;
      } else {
        delete process.NODE_TLS_REJECT_UNAUTHORIZED;
      }
    }
  }
}
exports.default = Dump;
module.exports = exports['default'];
