'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = mapUri;

var _lodash = require('lodash');

var _url = require('url');

var _filter_headers = require('./filter_headers');

var _filter_headers2 = _interopRequireDefault(_filter_headers);

var _set_headers = require('./set_headers');

var _set_headers2 = _interopRequireDefault(_set_headers);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// kibi: server and sirenAction added for
// replacing '_search' with '_msearch' to use siren-federate when available

function mapUri(cluster, proxyPrefix, server, sirenAction) {
  const serverConfig = server.config();

  function joinPaths(pathA, pathB) {
    return (0, _lodash.trimRight)(pathA, '/') + '/' + (0, _lodash.trimLeft)(pathB, '/');
  }

  return function (request, done) {
    const elasticsearchPlugins = serverConfig.get('investigate_core.clusterplugins');
    const {
      protocol: esUrlProtocol,
      slashes: esUrlHasSlashes,
      auth: esUrlAuth,
      hostname: esUrlHostname,
      port: esUrlPort,
      pathname: esUrlBasePath,
      query: esUrlQuery
    } = (0, _url.parse)(cluster.getUrl(), true);

    // copy most url components directly from the elasticsearch.url
    const mappedUrlComponents = {
      protocol: esUrlProtocol,
      slashes: esUrlHasSlashes,
      auth: esUrlAuth,
      hostname: esUrlHostname,
      port: esUrlPort
    };

    // pathname
    const reqSubPath = request.path.replace(proxyPrefix, '');
    mappedUrlComponents.pathname = joinPaths(esUrlBasePath, reqSubPath);

    // kibi: replace _search with _msearch to use siren-federate when available
    if (sirenAction && elasticsearchPlugins && ((0, _lodash.contains)(elasticsearchPlugins, 'siren-federate') || (0, _lodash.contains)(elasticsearchPlugins, 'siren-vanguard'))) {
      if (!reqSubPath.startsWith('/_siren/') && (reqSubPath.endsWith('_search') || reqSubPath.endsWith('_search/scroll') || reqSubPath.endsWith('_msearch'))) {
        mappedUrlComponents.pathname = joinPaths(esUrlBasePath, `siren/${(0, _lodash.trimLeft)(reqSubPath, '/')}`);
      }
    }
    // kibi: end

    // kibi: remove request parameters used only for differentiating requests
    const requestQuery = request.query;
    if (requestQuery) {
      delete requestQuery.getMeta;
      delete requestQuery.getEntitiesFromDashboard;
    }
    // kibi: end

    // querystring
    const mappedQuery = (0, _lodash.defaults)((0, _lodash.omit)(requestQuery, '_'), esUrlQuery);
    if (Object.keys(mappedQuery).length) {
      mappedUrlComponents.query = mappedQuery;
    }

    const filteredHeaders = (0, _filter_headers2.default)(request.headers, cluster.getRequestHeadersWhitelist());

    // kibi: inject header to tract user sessions
    const customHeaders = cluster.getCustomHeaders() || {};
    if (server.plugins && server.plugins.siren_federate && server.plugins.siren_federate.federateClient) {
      const sessionHeader = server.plugins.siren_federate.federateClient.getFederateSessionHeader(request);
      if (sessionHeader) {
        customHeaders[sessionHeader.name] = sessionHeader.value;
      }
    }
    // kibi: end
    const mappedHeaders = (0, _set_headers2.default)(filteredHeaders, customHeaders);
    const mappedUrl = (0, _url.format)(mappedUrlComponents);
    done(null, mappedUrl, mappedHeaders);
  };
}
module.exports = exports['default'];
