import { DashboardViewMode } from '../dashboard_view_mode';
import { TopNavIds } from './top_nav_ids';

/**
 * @param {DashboardMode} dashboardMode.
 * @param actions {Object} - A mapping of TopNavIds to an action function that should run when the
 * corresponding top nav is clicked.
 * @return {Array<kbnTopNavConfig>} - Returns an array of objects for a top nav configuration, based on the
 * mode.
 */
export function getTopNavConfig(dashboardMode, actions) {
  switch (dashboardMode) {
    case DashboardViewMode.VIEW:
      return [
        getShareConfig(),
        getCloneConfig(actions[TopNavIds.CLONE]),
        getEditConfig(actions[TopNavIds.ENTER_EDIT_MODE]),
        getExportConfig()];
    case DashboardViewMode.EDIT:
      return [
        getSaveConfig(),
        getViewConfig(actions[TopNavIds.EXIT_EDIT_MODE]),
        getDashboardDatamodelConfig(),
        getAddConfig(),
        getOptionsConfig(),
        getShareConfig()];
    default:
      return [];
  }
}

/**
 * @returns {kbnTopNavConfig}
 */
function getEditConfig(action) {
  return {
    key: 'edit',
    description: 'Switch to edit mode',
    testId: 'dashboardEditMode',
    run: action,
    dataUiAcl: 'ui:kibi@dashboard_edit' // kibi: added to control button via UI ACL rules
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getDashboardDatamodelConfig(action) {
  return {
    key: 'data model',
    description: 'Show dashboard 360 graph',
    testId: 'dashboardRelationsItem',
    template: require('plugins/kibana/dashboard/top_nav/dashboard_datamodel.html')
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getSaveConfig() {
  return {
    key: 'save',
    description: 'Save your dashboard',
    testId: 'dashboardSaveMenuItem',
    template: require('plugins/kibana/dashboard/top_nav/save.html')
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getViewConfig(action) {
  return {
    key: 'cancel',
    description: 'Cancel editing and switch to view-only mode',
    testId: 'dashboardViewOnlyMode',
    run: action
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getCloneConfig(action) {
  return {
    key: 'clone',
    description: 'Create a copy of your dashboard',
    testId: 'dashboardClone',
    run: action,
    dataUiAcl: 'ui:kibi@dashboard_clone' // kibi: added to control button via UI ACL rules
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getAddConfig() {
  return {
    key: TopNavIds.ADD,
    description: 'Add a panel to the dashboard',
    testId: 'dashboardAddPanelButton',
    template: require('plugins/kibana/dashboard/top_nav/add_panel.html')
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getShareConfig() {
  return {
    key: TopNavIds.SHARE,
    description: 'Share Dashboard',
    testId: 'dashboardShareButton',
    template: require('plugins/kibana/dashboard/top_nav/share.html'),
    dataUiAcl: 'ui:kibi@dashboard_share' // kibi: added to control button via UI ACL rules
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getOptionsConfig() {
  return {
    key: TopNavIds.OPTIONS,
    description: 'Options',
    testId: 'dashboardOptionsButton',
    template: require('plugins/kibana/dashboard/top_nav/options.html')
  };
}

/**
 * @returns {kbnTopNavConfig}
 */
function getExportConfig() {
  return {
    key: TopNavIds.EXPORT,
    description: 'Export',
    testId: 'dashboardExportButton',
    template: require('plugins/kibana/dashboard/top_nav/export.html')
  };
}
