'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FieldFormat = undefined;

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _content_types = require('../content_types');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const contentTypes = (0, _content_types.contentTypesProvider)();
class FieldFormat {
  constructor(params = {}) {
    this.type = this.constructor;
    this._params = params;
    this._paramDefaults = this.type.paramDefaults || {};

    if (_lodash2.default.isFunction(this._convert)) {
      this._convert = { text: this._convert };
    }

    contentTypes.setup(this);
  }

  from(converter) {
    class FieldFormatFromConvert extends FieldFormat {
      constructor(params) {
        super(params);
        this._convert = converter;
      }
    }

    return FieldFormatFromConvert;
  }

  /**
   * Convert a raw value to a formatted string
   * @param  {any} value
   * @param  {string} [contentType=text] - optional content type, the only two contentTypes
   *                                currently supported are "html" and "text", which helps
   *                                formatters adjust to different contexts
   * @return {string} - the formatted string, which is assumed to be html, safe for
   *                    injecting into the DOM or a DOM attribute
   */

  // kibi: mapping was added to make correct label for date fields with custom mapping
  convert(value, contentType, mapping) {
    return this.getConverterFor(contentType)(value, mapping);
  }

  /**
   * Get a convert function that is bound to a specific contentType
   * @param  {string} [contentType=text]
   * @return {function} - a bound converter function
   */
  getConverterFor(contentType = 'text') {
    return this._convert[contentType];
  }

  /**
   * Get the value of a param. This value may be a default value.
   *
   * @param  {string} name - the param name to fetch
   * @return {any}
   */
  param(name) {
    const val = this._params[name];
    if (val || val === false || val === 0) {
      // truthy, false, or 0 are fine
      // '', NaN, null, undefined, etc are not
      return val;
    }

    return this._paramDefaults[name];
  }

  /**
   * Get all of the params in a single object
   * @return {object}
   */
  params() {
    return _lodash2.default.cloneDeep(_lodash2.default.defaults({}, this._params, this._paramDefaults));
  }

  /**
   * serialize this format to a simple POJO, with only the params
   * that are not default
   *
   * @return {object}
   */
  toJSON() {
    const type = this.type;
    const defaults = this._paramDefaults;

    let params = _lodash2.default.transform(this._params, (uniqParams, val, param) => {
      if (val !== defaults[param]) {
        uniqParams[param] = val;
      }
    }, {});

    if (!_lodash2.default.size(params)) {
      params = undefined;
    }

    return {
      id: type.id,
      params: params
    };
  }
}
exports.FieldFormat = FieldFormat;
