'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GeoshapeType = exports.GeopointType = undefined;
exports.createFieldType = createFieldType;

var _ngeohash = require('ngeohash');

var _ngeohash2 = _interopRequireDefault(_ngeohash);

var _terraformerWktParser = require('terraformer-wkt-parser');

var _terraformerWktParser2 = _interopRequireDefault(_terraformerWktParser);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class FieldType {
  constructor(obj) {
    this._fieldObj = this._normalize(obj);
  }

  _normalize(obj) {
    return obj;
  }

  normalized() {
    return this._fieldObj;
  }
}

/* Geo-points can be specified in four different ways:
    1. Object {"lat": latitude, "lon": longitude}
    2. String "latitude, longitude"
    3. Geohash "drm3btev3e86"
    4. Array [longitude, latitude]
*/
class GeopointType extends FieldType {
  constructor(obj) {
    super(obj);
  }

  // NOTE:
  // Geo-points expressed as strings are formatted as "lat,lon",
  // whereas geo-points expressed as arrays are formatted as [lon,lat].
  // (Source: https://www.elastic.co/guide/en/elasticsearch/reference/current/geo-point.html#CO242-4)
  _normalize(geopoint) {
    if (typeof geopoint === 'string') {
      if (geopoint.includes(',')) {
        const geopointArray = geopoint.replace(/ /g, '').split(',').map(coord => parseFloat(coord));
        return { 'lat': geopointArray[0], 'lon': geopointArray[1] };
      } else {
        const coords = _ngeohash2.default.decode(geopoint);
        return { 'lat': coords.latitude, 'lon': coords.longitude };
      }
    } else if (Array.isArray(geopoint)) {
      if (!geopoint.length) {
        return [];
      } else if (typeof geopoint[0] === 'string') {
        // Array of geopoints
        return geopoint.map(gp => this._normalize(gp));
      } else {
        return { 'lat': geopoint[1], 'lon': geopoint[0] };
      }
    } else {
      return { 'lat': geopoint.lat, 'lon': geopoint.lon };
    }
  }
}

exports.GeopointType = GeopointType;
class GeoshapeType extends FieldType {
  constructor(obj) {
    super(obj);
  }

  _normalize(geoshape) {
    const typeToLowerCase = obj => {
      const newObj = JSON.parse(JSON.stringify(obj));
      newObj.type = newObj.type.toLowerCase();
      return newObj;
    };

    if (typeof geoshape === 'string') {
      const geoshapeJSON = typeToLowerCase(_terraformerWktParser2.default.parse(geoshape));
      return geoshapeJSON;
    } else {
      return geoshape;
    }
  }
}

exports.GeoshapeType = GeoshapeType;
function createFieldType(obj, type) {
  switch (type) {
    case 'geo_point':
      return new GeopointType(obj);
    case 'geo_shape':
      return new GeoshapeType(obj);
    default:
      return new FieldType(obj);
  }
}
