'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FederateResolver = undefined;

var _constants = require('./constants');

var _ = require('./');

const formatPluginsObj = function (elasticsearchPlugins) {
  return {
    node1: {
      plugins: elasticsearchPlugins.map(plugin => {
        return {
          name: plugin.component,
          version: plugin.version
        };
      })
    }
  };
};

class FederateResolver {
  constructor(elasticsearchPlugins, compatibilityMatrix) {
    let federateVersion;
    for (let i = 0; i < elasticsearchPlugins.length; i++) {
      if (elasticsearchPlugins[i].component === _constants.SIREN_FEDERATE_PLUGIN) {
        federateVersion = elasticsearchPlugins[i].version;
        break;
      }
    }
    if (federateVersion) {
      this._featureCompatibilityService = new _.FeatureCompatibilityService(federateVersion);
      this._apiCompatibilityService = new _.ApiCompatibilityService(federateVersion);
      this._compatibilityResolver = new _.CompatibilityResolver(formatPluginsObj(elasticsearchPlugins), compatibilityMatrix);
    }
  }

  /**
   * @return {[{ key, object }]} Returns a list of { key, object } of disabled features
   */
  getDisabledFeatures() {
    if (this._featureCompatibilityService) {
      return this._featureCompatibilityService.getDisabledFeatureList();
    }
    return [];
  }

  /**
   * Check if feature is available
   * @param  {String} feature, As registered in feature_compatibility_matrix
   * @return {Boolean} True if feature is available, otherwise False in case of unknown or disabled features
   */
  isAvailable(feature) {
    if (this._featureCompatibilityService) {
      return this._featureCompatibilityService.isAvailable(feature);
    }
    return false;
  }

  /**
   * Check is backend API is available.
   * @param  {String} apiKey, As registered in api_compatibility_matrix
   * @return {Boolean} True if api is available, otherwise False in case of unknown or disabled apis
   */
  checkApiAvailability(apiKey) {
    if (this._apiCompatibilityService) {
      return this._apiCompatibilityService.isAvailable(apiKey);
    }
    return false;
  }

  /**
   * Get the compatibility status of Investigate
   * @return {COMPATIBILITY_STATUS}
   */
  getCompatibilityStatus() {
    return this._compatibilityResolver.getCompatibilityStatus();
  }
}

exports.FederateResolver = FederateResolver;
