'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _federate_version_parser = require('./utils/federate_version_parser');

var _lodash = require('lodash');

var _compatibility = require('./compatibility');

class ModuleCompatibilityService {

  static _checkESSupport(compatibilityDefinition, elasticsearchVersion) {
    if (compatibilityDefinition.supported_es_exceptions) {
      compatibilityDefinition.supported_es_exceptions.forEach(version => {
        if (elasticsearchVersion.rawVersion === version) {
          return true;
        }
      });
    }

    if (compatibilityDefinition.min_es) {
      compatibilityDefinition.min_es = new _federate_version_parser.Version(compatibilityDefinition.min_es);
      if (compatibilityDefinition.min_es.compareWith(elasticsearchVersion) >= 0) {
        if (compatibilityDefinition.max_es) {
          compatibilityDefinition.max_es = new _federate_version_parser.Version(compatibilityDefinition.max_es);
          return compatibilityDefinition.max_es.compareWith(elasticsearchVersion) < 0;
        }
        return true;
      }
      return false;
    }

    if (compatibilityDefinition.max_es) {
      compatibilityDefinition.max_es = new _federate_version_parser.Version(compatibilityDefinition.max_es);
      return compatibilityDefinition.max_es.compareWith(elasticsearchVersion) < 0;
    }
    return true;
  }

  static _checkFederateSupport(compatibilityDefinition, currentFederateVersion) {
    if (compatibilityDefinition.supported_federate_exceptions) {
      compatibilityDefinition.supported_federate_exceptions.forEach(version => {
        if (currentFederateVersion.rawVersion === version) {
          return true;
        }
      });
    }

    if (compatibilityDefinition.min_federate) {
      compatibilityDefinition.min_federate = new _federate_version_parser.FederateVersionParser(compatibilityDefinition.min_federate);
      if (compatibilityDefinition.min_federate.compareWith(currentFederateVersion) >= 0) {
        if (compatibilityDefinition.max_federate) {
          compatibilityDefinition.max_federate = new _federate_version_parser.FederateVersionParser(compatibilityDefinition.max_federate);
          return compatibilityDefinition.max_federate.compareWith(currentFederateVersion) < 0;
        }
        return true;
      }
      return false;
    }

    if (compatibilityDefinition.max_federate) {
      compatibilityDefinition.max_federate = new _federate_version_parser.FederateVersionParser(compatibilityDefinition.max_federate);
      return compatibilityDefinition.max_federate.compareWith(currentFederateVersion) < 0;
    }
    return true;
  }

  static _isModuleAvailable(compatibilityDefinition, currentFederateVersion) {
    if (ModuleCompatibilityService._checkESSupport(compatibilityDefinition, currentFederateVersion.elasticsearchVersion)) {
      compatibilityDefinition.isAvailable = ModuleCompatibilityService._checkFederateSupport(compatibilityDefinition, currentFederateVersion);
    } else {
      compatibilityDefinition.isAvailable = false;
    }
    return compatibilityDefinition;
  }

  static _getComputedModuleMatrix(federateVersion, compatibilityMatrix) {
    const computedMatrix = {};
    const COMPATIBILITY_MATRIX = (0, _lodash.cloneDeep)(compatibilityMatrix);
    Object.keys(COMPATIBILITY_MATRIX).forEach(key => {
      computedMatrix[key] = ModuleCompatibilityService._isModuleAvailable(COMPATIBILITY_MATRIX[key], federateVersion);
    });
    return Object.freeze(computedMatrix);
  }

  /**
   * @param  {FederateVersionParser} federateVersion
   * @param  {COMPATIBILITY_MATRIX} compatibilityMatrix
   */
  constructor(federateVersion, compatibilityMatrix) {
    this.federateVersion = new _federate_version_parser.FederateVersionParser(federateVersion);
    this.moduleMatrix = ModuleCompatibilityService._getComputedModuleMatrix(this.federateVersion, compatibilityMatrix);
  }

  /**
   * Check if module is available
   * @return {Boolean} True if module is available, otherwise False in case of unknown or disabled modules
   */
  isAvailable(moduleKey) {
    return (this.moduleMatrix[moduleKey] || false) && this.moduleMatrix[moduleKey].isAvailable;
  }

  /**
   * @return {[{ key, object }]} Returns a list of { key, object } of disabled modules
   */
  getDisabledModuleList() {
    if (this.disabledModules) {
      return this.disabledModules;
    }
    this.disabledModules = [];
    Object.keys(this.moduleMatrix).forEach(key => {
      if (!this.moduleMatrix[key].isAvailable) {
        this.disabledModules.push({
          key,
          object: this.moduleMatrix[key]
        });
      }
    });
    return this.disabledModules;
  }

}
exports.default = ModuleCompatibilityService;
module.exports = exports['default'];
