'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 17.
 *
 * Looks for:
 *
 * - configuration objects with the id 'kibi'
 *
 * Then:
 *
 * - changes that id to 'siren'
 * - deletes the old 'kibi' object
 * - and PUTs the new 'siren' object
 *  (effectively overwriting the old 'kibi' object)
 */
class Migration17 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._type = 'config';
    this._server = configuration.server;
    this._querySiren = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'siren'
            }
          }]
        }
      }
    };
    this._queryKibi = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'kibi'
            }
          }]
        }
      }
    };
  }

  static get description() {
    return 'Migrate config object id from "kibi" to "siren"';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const configsSiren = await this.scrollSearch(this._index, this._type, this._querySiren);
    const configsKibi = await this.scrollSearch(this._index, this._type, this._queryKibi);
    return configsSiren.length === 0 && configsKibi.length > 0 ? 1 : 0;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const count = await this.count();
    if (count === 0) {
      return 0;
    }
    const configurations = await this.scrollSearch(this._index, this._type, this._queryKibi);
    if (configurations.length === 0) {
      return 0;
    }

    let body = '';
    for (const config of configurations) {
      body += JSON.stringify({
        index: {
          _index: config._index,
          _type: config._type,
          _id: 'siren'
        }
      }) + '\n' + JSON.stringify(config._source) + '\n';
    }

    if (body.length) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return configurations.length;
  }
}
exports.default = Migration17;
module.exports = exports['default'];
