'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 23.
 *
 * Looks for:
 *
 * - the refreshInterval object within the kibana.index dashboard mapping
 *
 * Then:
 *
 * - if not found, adds the refreshInterval mapping object
 */
class Migration23 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._type = 'dashboard';
    this._server = configuration.server;
    this._defaultRefreshIntervalMapping = {
      properties: {
        display: {
          type: 'text',
          fields: {
            keyword: {
              type: 'keyword',
              ignore_above: 256
            }
          }
        },
        pause: {
          type: 'boolean'
        },
        section: {
          type: 'long'
        },
        value: {
          type: 'long'
        }
      }
    };
  }

  static get description() {
    return 'Add the refreshInterval mapping type to the mappings if it doesn\'t exist';
  }

  /**
   * Checks if dashboard mapping contains a property called "refreshInterval".
   */
  _isUpgradeable(mappings) {
    const dashboardMappingsProperties = mappings.dashboard.properties;
    return dashboardMappingsProperties.refreshInterval ? 0 : 1;
  }

  /**
   * Exposes the default refreshInterval mapping object
   */
  getDefaultRefreshIntervalMapping() {
    return this._defaultRefreshIntervalMapping;
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    try {
      const dashboardMapping = await this._client.indices.getMapping({
        index: this._index,
        type: this._type,
        ignoreUnavailable: true,
        allowNoIndices: true
      });
      return this._isUpgradeable(dashboardMapping[this._index].mappings);
    } catch (e) {
      if (e.status === 404) {
        return 0;
      }
      throw e;
    }
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }

    let dashboardMapping;
    try {
      dashboardMapping = await this._client.indices.getMapping({ index: this._index, type: this._type });
    } catch (e) {
      if (e.status === 404) {
        return 0;
      }
      throw e;
    }

    const mapping = dashboardMapping[this._index].mappings;

    if (this._isUpgradeable(mapping) !== 0) {
      mapping.dashboard.properties.refreshInterval = this.getDefaultRefreshIntervalMapping();

      await this._client.indices.putMapping({
        index: this._index,
        type: this._type,
        body: mapping
      });
      return 1;
    } else {
      return 0;
    }
  }
}
exports.default = Migration23;
module.exports = exports['default'];
