'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class Migration26 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._query = {
      query: {
        bool: {
          should: [{ exists: { field: 'kibanaSavedObjectMeta' } }, { exists: { field: 'uiStateJSON' } }]
        }
      }
    };
    this._visualizations = [];
    this._dashboards = [];
    this._searches = [];
  }

  static get description() {
    return 'Fix and deduplicate broken wildcard queries (containing "$state) in saved objects';
  }

  countReplacer(countObject) {
    return function (key, val) {
      let innerCount = 0;

      if (val && val.query && val.$state && val.$state.store && !val.meta) {
        if (countObject.dryrun && innerCount === 0) {
          innerCount++;
        } else {
          delete val.$state;
        }
      }

      if (countObject.dryrun) {
        countObject.count += innerCount;
      }

      return val;
    };
  }

  checkAndReplaceDuplicateFilters(key, val) {
    if ((key === 'filters' || key === 'filter') && (0, _lodash.isArray)(val)) {
      return val.reduce((previous, current) => {
        const duplicate = previous.filter(query => (0, _lodash.isEqual)(query, current));

        if (duplicate.length === 0) {
          previous.push(current);
        }
        return previous;
      }, []);
    }

    return val;
  }

  clearCollections() {
    this._visualizations = [];
    this._searches = [];
    this._dashboards = [];
  }

  async _fetchEntities(collection, type, query) {
    let entities;
    try {
      entities = await this.scrollSearch(this._index, type, query);
    } catch (error) {
      if (error.status === 404) {
        return 0;
      } else {
        throw error;
      }
    }

    (0, _lodash.each)(entities, entity => {
      let upgradeableEntity = null;

      if (entity._source.kibanaSavedObjectMeta.searchSourceJSON) {
        const searchSourceJSON = JSON.parse(entity._source.kibanaSavedObjectMeta.searchSourceJSON);
        const searchCountObject = { count: 0, dryrun: true };
        JSON.stringify(searchSourceJSON, this.countReplacer(searchCountObject));

        if (searchCountObject.count > 0 && !upgradeableEntity) {
          searchCountObject.count = 0;
          upgradeableEntity = entity;
        }
      }

      if (entity && entity._source && entity._source.uiStateJSON && entity._source.uiStateJSON !== '{}') {
        const parsedUiStateJSON = JSON.parse(entity._source.uiStateJSON);
        const countObject = { count: 0, dryrun: true };
        JSON.stringify(parsedUiStateJSON, this.countReplacer(countObject));
        if (countObject.count > 0 && !upgradeableEntity) {
          countObject.count = 0;
          upgradeableEntity = entity;
        }
      }

      if (upgradeableEntity) {
        collection.push(upgradeableEntity);
      }
    });
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let count = 0;
    await this._fetchEntities(this._visualizations, 'visualization', this._query);
    await this._fetchEntities(this._dashboards, 'dashboard', this._query);
    await this._fetchEntities(this._searches, 'search', this._query);
    if (!(0, _lodash.isEmpty)(this._visualizations)) {
      count += this._visualizations.length;
    }
    if (!(0, _lodash.isEmpty)(this._dashboards)) {
      count += this._dashboards.length;
    }
    if (!(0, _lodash.isEmpty)(this._searches)) {
      count += this._searches.length;
    }

    this.clearCollections();
    return count;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let upgraded = 0;
    await this._fetchEntities(this._visualizations, 'visualization', this._query);
    await this._fetchEntities(this._dashboards, 'dashboard', this._query);
    await this._fetchEntities(this._searches, 'search', this._query);

    const upgradeableObjects = this._visualizations.concat(this._dashboards).concat(this._searches);

    if (upgradeableObjects.length === 0) {
      return upgraded;
    }
    let body = '';

    (0, _lodash.each)(upgradeableObjects, obj => {
      if (obj._source && obj._source.kibanaSavedObjectMeta.searchSourceJSON) {
        const newSearchSourceJSON = JSON.parse(obj._source.kibanaSavedObjectMeta.searchSourceJSON);
        const replacedSearchSourceJSON = JSON.stringify(newSearchSourceJSON, this.countReplacer({ count: 0, dryrun: false }));

        const checkForDuplicateFiltersSearchSourceJSON = JSON.parse(replacedSearchSourceJSON);
        const dedupedSearchSourceJSON = JSON.stringify(checkForDuplicateFiltersSearchSourceJSON, this.checkAndReplaceDuplicateFilters);
        obj._source.kibanaSavedObjectMeta.searchSourceJSON = dedupedSearchSourceJSON;
      }

      if (obj._source.uiStateJSON && obj._source.uiStateJSON !== '{}') {
        const parsedUiStateJSON = JSON.parse(obj._source.uiStateJSON);
        obj._source.uiStateJSON = JSON.stringify(parsedUiStateJSON, this.countReplacer({ count: 0, dryrun: false }));
        const dedupeFiltersUiStateJSON = JSON.parse(obj._source.uiStateJSON);
        obj._source.uiStateJSON = JSON.stringify(dedupeFiltersUiStateJSON, this.checkAndReplaceDuplicateFilters);
      }

      body += JSON.stringify({
        update: {
          _index: obj._index,
          _type: obj._type,
          _id: obj._id
        }
      }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
      upgraded++;
    });
    if (upgraded > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }

    this.clearCollections();
    return upgraded;
  }
}
exports.default = Migration26; /*eslint max-len: 0*/

module.exports = exports['default'];
