'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 7.
 *
 * Looks for:
 *
 * - configuration objects
 *
 * Then:
 *
 * - migrate latest configuration to the new config object singleton
 */
class Migration7 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');
  }

  static get description() {
    return 'Move configuration object to singleton';
  }

  async _getConfigurations() {
    const configurations = await this._client.search({
      index: this._index,
      type: 'config',
      size: 1000
    });
    return configurations.hits.hits.sort((confa, confb) => {
      const buildNumA = parseInt(confa._source.buildNum) || 0;
      const buildNumB = parseInt(confb._source.buildNum) || 0;
      return buildNumA < buildNumB;
    });
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const existingConfigs = await this._client.count({
      index: this._index,
      type: 'config',
      ignoreUnavailable: true
    });
    if (existingConfigs.count === 0) {
      return 0;
    }
    try {
      await this._client.get({
        index: this._index,
        type: 'config',
        id: 'kibi'
      });
    } catch (err) {
      if (err.status === 404) {
        const configurations = await this._getConfigurations();
        let configIdIsSnapshotOrNotSiren = true;
        for (const config of configurations) {
          if (!config._id.endsWith('-SNAPSHOT')) {
            configIdIsSnapshotOrNotSiren = false;
          }

          if (config._id === 'siren') {
            configIdIsSnapshotOrNotSiren = true;
            break;
          }
        }
        return configIdIsSnapshotOrNotSiren ? 0 : 1;
      }
      throw err;
    }
    return 0;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const count = await this.count();
    if (count === 0) {
      return 0;
    }
    const configurations = await this._getConfigurations();
    let configuration;
    for (const config of configurations) {
      if (!config._id.endsWith('-SNAPSHOT')) {
        configuration = config._source;
        break;
      }
    }
    delete configuration.buildNum;
    await this._client.create({
      refresh: true,
      index: this._index,
      type: 'config',
      id: 'kibi',
      body: configuration
    });
    return 1;
  }

}
exports.default = Migration7;
module.exports = exports['default'];
