'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _entity_type = require('../../ontology/entity_type');

var _gremlin_server = require('../../../../../server/gremlin_server/gremlin_server');

var _gremlin_server2 = _interopRequireDefault(_gremlin_server);

var _gremlin_server_errors = require('../../../../../server/gremlin_server/gremlin_server_errors');

var _server_ontology_client = require('../../ontology/server_ontology_client');

var _server_ontology_client2 = _interopRequireDefault(_server_ontology_client);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 38.
 *
 * This migration will retrieve the ontology and convert all EIDs and relations into saved objects.
 * Additionally it will update the saved searches with properties that were in the ontology only (icon, color etc)
 * and move the relationa graph layout into a separate saved object
 *
 */

class Migration38 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');

    this._ontologyType = 'ontology-model';
    this._ontologyId = 'default-ontology';
    this._gremlinStartTimeout = configuration.gremlinStartTimeout;
  }

  static get description() {
    return 'Migrated old ontology object. Separated it into searches, relations, EIDs and saved graph objects.';
  }

  logMissingRelationInverse(relation) {
    this._logger.warning(`Couldn't find inverse of relation "${relation.directLabel}",` + ` it will be skipped. Full relation: ${JSON.stringify(relation)}`);
  }

  logMissingRelationEndpoint(relation, endpoint) {
    this._logger.warning(`Couldn't find relation endpoint "${endpoint}" for relation` + ` "${relation.directLabel}", it will be skipped. Full relation:` + ` ${JSON.stringify(relation)}`);
  }

  getRelationsWithNoBroken(relations, entities, { logSkipped }) {
    const relationsIds = new Set((0, _lodash.map)(relations, 'id'));

    return (0, _lodash.filter)(relations, rel => {
      if (!relationsIds.has(rel.inverseOf)) {
        if (logSkipped) {
          this.logMissingRelationInverse(rel);
        }
        return false;
      }

      if (!(0, _lodash.find)(entities, 'id', rel.domain.id)) {
        if (logSkipped) {
          this.logMissingRelationEndpoint(rel, rel.domain.id);
        }
        return false;
      }

      if (!(0, _lodash.find)(entities, 'id', rel.range.id)) {
        if (logSkipped) {
          this.logMissingRelationEndpoint(rel, rel.range.id);
        }
        return false;
      }

      return true;
    });
  }

  async count() {
    const gremlin = new _gremlin_server2.default(this._server, this._gremlinStartTimeout);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);

    try {
      const ontologyExists = await this._ontologyExists();
      if (ontologyExists) {
        await gremlin.start('Migration 38');

        // Find entities
        const entities = await serverOntologyClient.getEntities();

        // Find Relations
        const relations = await serverOntologyClient.getRelations();
        // get the list of relations without the ones without an inverse (broken)
        const cleanRelations = this.getRelationsWithNoBroken(relations, entities, { logSkipped: false });

        // Find Relational Graph
        let relationalGraphCounter = 0;
        const object = await this._client.get({
          index: this._index,
          type: 'doc',
          id: `${this._ontologyType}:${this._ontologyId}`

        });
        if (object.found && object._source[this._ontologyType]['relational-graph']) {
          ++relationalGraphCounter;
        }

        await gremlin.stop('Migration 38 clean');

        // We add 1 for the ontology object that has to be deleted
        const relNum = cleanRelations.length > 0 ? cleanRelations.length / 2 : 0;
        return entities.length + relNum + relationalGraphCounter + 1;
      } else {
        return 0;
      }
    } catch (e) {
      this._logger.error('Migration 38 - count operation');
      if (e instanceof _gremlin_server_errors.GremlinStopError) {
        this._logger.error('Could not stop the Siren Gremlin Server for Migration 38');
        this._logger.error(e);
      } else if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 38');
        this._logger.error(e);
        await gremlin.stop('Migration 38 error');
      } else {
        this._logger.error('Could not perform Migration 38');
        this._logger.error(e);
        await gremlin.stop('Migration 38 error');
      }
      throw e;
    }
  }

  async upgrade() {
    const gremlin = new _gremlin_server2.default(this._server, this._gremlinStartTimeout);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);

    try {
      const ontologyExists = await this._ontologyExists();
      if (ontologyExists) {
        await gremlin.start('Migration 38');

        const entities = await serverOntologyClient.getEntities();
        // EIDs
        const eids = (0, _lodash.reduce)(entities, (total, entity) => {
          if (entity.type === _entity_type.EntityType.VIRTUAL_ENTITY) {
            total.push(entity);
          }
          return total;
        }, []);

        if (eids.length) {
          await this._createEidSavedObjects(eids);
        }

        // Searches
        const searches = (0, _lodash.reduce)(entities, (total, entity) => {
          if (entity.type === _entity_type.EntityType.SAVED_SEARCH) {
            total.push(entity);
          }
          return total;
        }, []);

        if (searches.length) {
          await this._updateSearchSavedObjects(searches);
        }

        // Relations
        const relations = await serverOntologyClient.getRelations();
        const cleanRelations = this.getRelationsWithNoBroken(relations, entities, { logSkipped: true });

        if (cleanRelations.length) {
          await this._createRelationSavedObjects(cleanRelations, entities);
        }

        // Relational Graph
        let relationalGraphCounter = 0;
        const object = await this._client.get({
          index: this._index,
          type: 'doc',
          id: `${this._ontologyType}:${this._ontologyId}`
        });
        if (object.found && object._source[this._ontologyType]['relational-graph']) {
          const relationalGraph = object._source[this._ontologyType]['relational-graph'];
          ++relationalGraphCounter;
          await this._createRelationalGraphSavedObjects(relationalGraph, searches);
        }

        this._deleteSavedOntology();

        await gremlin.stop('Migration 38 clean');

        // We add 1 as we deleted the ontology object
        const relNum = cleanRelations.length > 0 ? cleanRelations.length / 2 : 0;
        return eids.length + searches.length + relNum + 1 + relationalGraphCounter;
      } else {
        return 0;
      }
    } catch (e) {
      this._logger.error('Migration 38 - count operation');
      if (e instanceof _gremlin_server_errors.GremlinStopError) {
        this._logger.error('Could not stop the Siren Gremlin Server for Migration 38');
        this._logger.error(e);
      } else if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 38');
        this._logger.error(e);
        await gremlin.stop('Migration 38 error');
      } else {
        this._logger.error('Could not perform Migration 38');
        this._logger.error(e);
        await gremlin.stop('Migration 38 error');
      }
      throw e;
    }
  }

  // check if there is a saved object for the ontology model
  async _ontologyExists() {
    return this._client.exists({
      index: this._index,
      type: 'doc',
      id: `${this._ontologyType}:${this._ontologyId}`
    });
  }

  async _createEidSavedObjects(eids) {
    for (const eid of eids) {
      const id = this._normalizeEidId(eid.id);
      const source = {
        eid: {
          id: id,
          title: eid.label,
          version: 1,
          siren: {
            ui: {
              icon: eid.icon,
              color: eid.color,
              shortDescription: eid.shortDescription
            }
          }
        },
        type: 'eid'
      };

      await this._indexDoc(id, source);
    }
  }

  async _updateSearchSavedObjects(searches) {
    const searchQuery = {
      query: {
        match: {
          type: 'search'
        }
      }
    };
    const savedSearches = await this.scrollSearch(this._index, 'doc', searchQuery);

    for (const search of searches) {
      const savedSearch = (0, _lodash.find)(savedSearches, '_id', search.id);

      if (savedSearch) {
        let parentId = null;
        if (search.parentId && search.parentId !== 'null' && search.parentId !== 'undefined') {
          parentId = decodeURIComponent(search.parentId);
        }
        savedSearch._source.search.siren = {
          parentId: parentId,
          ui: {
            icon: search.icon,
            color: search.color,
            shortDescription: search.shortDescription,
            instanceLabel: {
              type: search.instanceLabelType,
              value: search.instanceLabelValue
            }
          }
        };

        await this._indexDoc(search.id, savedSearch._source);
      }
    };
  }

  async _createRelationSavedObjects(relations, entities) {
    // We store only one saved object for each couple (direct + inverse) of relations
    const storedRel = new Set();
    for (const relation of relations) {
      // Add the prefix if needed
      const id = this._normalizeRelationId(relation.id);

      if (!storedRel.has(id)) {
        // Add the prefix if needed
        const inverseOf = this._normalizeRelationId(relation.inverseOf);

        const domainEntity = (0, _lodash.find)(entities, 'id', relation.domain.id);
        const domainId = domainEntity.type === _entity_type.EntityType.SAVED_SEARCH ? domainEntity.id : this._normalizeEidId(domainEntity.id);
        const rangeEntity = (0, _lodash.find)(entities, 'id', relation.range.id);
        const rangeId = rangeEntity.type === _entity_type.EntityType.SAVED_SEARCH ? rangeEntity.id : this._normalizeEidId(rangeEntity.id);

        const source = {
          relation: {
            id: id,
            inverseOf: inverseOf,
            title: `${relation.domain.label} -> ${relation.directLabel} -> ${relation.range.label}`,
            version: 1,
            timeout: relation.timeout,
            joinType: relation.joinType,
            directLabel: relation.directLabel,
            inverseLabel: relation.inverseLabel,
            domainField: relation.domain.field,
            rangeField: relation.range.field,
            domainId,
            rangeId
          },
          type: 'relation'
        };

        await this._indexDoc(id, source);
        storedRel.add(inverseOf);
      }
    };
  }

  async _createRelationalGraphSavedObjects(relationalGraph, searches) {
    for (const search of searches) {

      const id = 'relational-graph:' + search.id;
      const source = {
        "relational-graph": {
          model: JSON.stringify(relationalGraph),
          title: search.label,
          version: 1
        },
        type: 'relational-graph'
      };
      await this._indexDoc(id, source);
    };
  }

  async _deleteSavedOntology() {
    const body = JSON.stringify({
      delete: {
        _index: this._index,
        _type: 'doc',
        _id: this._ontologyType + ':' + this._ontologyId
      }
    }) + '\n';
    await this._client.bulk({
      refresh: true,
      body: body
    });
  }

  _normalizeEidId(id) {
    return id.startsWith('eid:') ? id : 'eid:' + id;
  }

  _normalizeRelationId(id) {
    return id.startsWith('relation:') ? id : 'relation:' + id;
  }

  async _indexDoc(id, source) {
    await this._client.index({
      index: this._index,
      type: 'doc',
      id: id,
      body: source,
      refresh: true
    });
  }
}
exports.default = Migration38;
module.exports = exports['default'];
