'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _requirefrom = require('requirefrom');

var _requirefrom2 = _interopRequireDefault(_requirefrom);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 40.
 *
 * Changes the mappings for "url" objects, property "sirenSession" becomes an opaque
 * "object", with no indexed sub-property.
 */

class Migration40 extends _migration2.default {
  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._type = 'doc';
    this._query = {
      query: {
        term: {
          type: 'url'
        }
      }
    };
  }

  static get description() {
    return 'Changes the mappings for "url" objects, property "sirenSession" becomes an opaque JSON "object", with no indexed sub-property';
  }

  async getMappings() {
    let result;

    try {
      result = await this._client.indices.get({
        index: this._index
      });
    } catch (err) {
      if (err.status !== 404) {
        throw err;
      }
    }

    return result && result[this._index].mappings;
  }

  getUrlProperty(mappings) {
    return mappings && _lodash2.default.get(mappings[this._type], 'properties.url');
  }

  getUrlSirenSessionProperty(mappings) {
    return mappings && _lodash2.default.get(mappings[this._type], 'properties.url.properties.sirenSession');
  }

  async count() {
    const mappings = await this.getMappings();

    const sirenSessionProperty = this.getUrlSirenSessionProperty(mappings);
    if (!sirenSessionProperty) {
      // If the url or sirenSession mapping type is not present, it should be
      // automatically created with the right type when a document is first
      // indexed in
      return 0;
    }

    if (sirenSessionProperty.enabled === false) {
      // The sirenSession property is already disabled - nothing to do
      return 0;
    }

    // Otherwise, mappings for property sirenSession must be updated
    return 1;
  }

  async upgrade() {
    const mappings = await this.getMappings();

    const sirenSessionProperty = this.getUrlSirenSessionProperty(mappings);
    if (!sirenSessionProperty || sirenSessionProperty.enabled === false) {
      return 0;
    }

    // Retrieve all documents in the kibana index
    const allObjects = await this.scrollSearch(this._index);
    if (allObjects.length === 0) {
      return 0;
    }

    // Build the updated mappings for all retrieved document types
    this.getUrlProperty(mappings).properties.sirenSession = {
      type: 'object',
      enabled: false
    };

    // Retrieve all documents
    const body = [];

    for (const obj of allObjects) {
      body.push({
        index: {
          _index: obj._index,
          _type: obj._type,
          _id: obj._id
        }
      });
      body.push(obj._source);
    }

    // Remove the kibana index
    await this._client.indices.delete({
      index: this._index
    });

    // Recreate it with the new, fixed mappings
    await this._client.indices.create({
      index: this._index,
      body: {
        mappings
      }
    });

    // Bulk indexing request on newly created index
    await this._client.bulk({
      refresh: true,
      body: body
    });

    return 1;
  }
}
exports.default = Migration40;
module.exports = exports['default'];
