import _ from 'lodash';
import uiRoutes from 'ui/routes';
import { uiModules } from 'ui/modules';
import template from './advanced_options.html';

uiRoutes.when('/management/siren/relations/:entity/:relId', {
  template,
  resolve: {
    entity: function ($route, ontologyModel, courier) {
      return ontologyModel
        .getEntityMap(entityMap => {
          const entityId = decodeURIComponent($route.current.params.entity);
          return entityMap[entityId];
        })
        .catch(courier.redirectWhenMissing('/management/siren/datamodel'));
    },
    relations: function ($route, ontologyModel, courier) {
      return ontologyModel
        .getRelationList()
        .then(relations => {
          const relation = _.find(relations,
            rel => rel.id === decodeURIComponent($route.current.params.relId));
          const inverseRelation = _.find(relations,
            rel => rel.id === relation.inverseOf);

          return { relation, inverseRelation };
        });
    }
  },
  mapBreadcrumbs($route, breadcrumbs) {
    const { entity, relations } = $route.current.locals;
    const { relation } = relations;

    return breadcrumbs.map(crumb => {
      const crumbId = decodeURIComponent(crumb.id);

      if (crumbId === entity.id) {
        crumb = { ...crumb, display: entity.label };
      } else if (crumbId === relation.id) {
        crumb = { ...crumb, display: relation.directLabel };
      }

      return crumb;
    });
  },
});

uiModules
  .get('apps/management')
  .controller('RelationAdvancedSettingsController',
    function ($scope, $route, config, kbnUrl, createNotifier, kibiVersion, ontologyModel, savedRelations) {
      const notify = createNotifier({
        name: 'Relation Advanced Settings'
      });
      const docsBaseLink = `https://docs.support.siren.io/${kibiVersion}/platform/en/index-en.html`;
      $scope.docsLinks = {
        joinType: `${docsBaseLink}?contextId=004_JNTP`,
        joinTaskTimeout: `${docsBaseLink}?contextId=002_JNTT`
      };

      $scope.joinTypes = [
        {
          value: null,
          label: 'Automatic'
        },
        {
          value: 'HASH_JOIN',
          label: 'Distributed join using hash join algorithm'
        },
        {
          value: 'BROADCAST_JOIN',
          label: 'Broadcast join'
        }
      ];

      const redirectToEntity = function (entityId) {
        return ontologyModel.getEntityMap()
          .then(entityMap => {
            const entity = entityMap[decodeURIComponent(entityId)];
            kbnUrl.change('/management/siren/datamodel/' + entity.type + '/' + entity.id + '/relations');
          });
      };

      $scope.relation = $route.current.locals.relations.relation;
      $scope.inverseRelation = $route.current.locals.relations.inverseRelation;

      // check if the limit property is present
      if (typeof $scope.relation.timeout === 'undefined') {
        const defaultTimeout = config.get('siren:joinTaskTimeout');
        if (defaultTimeout === 0) {
          $scope.relation.timeout = -1; // -1 to take the global one from kibi:joinTaskTimeout
        } else {
          $scope.relation.timeout = defaultTimeout;
        }
      }

      // cancel button
      $scope.cancel = function () {
        redirectToEntity($route.current.params.entity);
      };

      // save button
      $scope.submit = function () {
        // note: if we plan to allow the user to have different parameter in the direct and
        // inverse relation, we need switch to a "deleteRelationsById" + "insertRelations(false)"
        // that does not automatically add the inverse. Both the API are not yet implemented.
        $scope.inverseRelation.joinType = $scope.relation.joinType;
        $scope.inverseRelation.timeout = $scope.relation.timeout;
        if (!$scope.relation.joinType || $scope.relation.joinType === '') {
          delete $scope.relation.joinType;
          delete $scope.inverseRelation.joinType;
        }

        Promise.all([
          savedRelations.get($scope.relation.id),
          savedRelations.get($scope.inverseRelation.id)
        ])
          .then(([ savedRelation, savedRelationInverse ]) => {
            savedRelation.timeout = $scope.relation.timeout;
            savedRelationInverse.timeout = $scope.relation.timeout;

            savedRelation.joinType = $scope.relation.joinType || null;

            savedRelationInverse.joinType = savedRelation.joinType;

            return Promise.all([
              savedRelation.save(),
              savedRelationInverse.save()
            ])
              .then(() => {
                $scope.relation.joinType = $scope.relation.joinType ? $scope.relation.joinType : null;
                notify.info(`Advanced settings for the ${$scope.relation.directLabel} relation successfuly updated`);
                redirectToEntity($route.current.params.entity);
              });
          })
          .catch(notify.error);
      };
    }
  );
