import _ from 'lodash';

import { IsRequestProvider } from './is_request';
import { MergeDuplicatesRequestProvider } from './merge_duplicate_requests';
import { ReqStatusProvider } from './req_status';
import { uniqFilters } from 'ui/filter_bar/lib/uniq_filters';

// kibi: import
import { extractHighestTaskTimeoutFromMsearch } from 'ui/kibi/helpers/extract_highest_task_timeout_from_msearch';
import { GetUUIDProvider } from 'ui/kibi/helpers/get_uuid';
import { JobCancelProvider } from 'ui/kibi/helpers/job_cancel';
// kibi: end

export function CallClientProvider(Private, Promise, esAdmin, es, config) {

  const isRequest = Private(IsRequestProvider);
  const mergeDuplicateRequests = Private(MergeDuplicatesRequestProvider);
  const getUUID = Private(GetUUIDProvider);
  const jobCancel = Private(JobCancelProvider);

  const ABORTED = Private(ReqStatusProvider).ABORTED;
  const DUPLICATE = Private(ReqStatusProvider).DUPLICATE;

  function callClient(strategy, requests) {
    // merging docs can change status to DUPLICATE, capture new statuses
    const statuses = mergeDuplicateRequests(requests);

    // get the actual list of requests that we will be fetching
    let requestsToFetch = statuses.filter(isRequest);
    let execCount = requestsToFetch.length;

    if (!execCount) return Promise.resolve([]);

    // resolved by respond()
    let esPromise;
    const defer = Promise.defer();

    // for each respond with either the response or ABORTED
    const respond = function (responses) {
      responses = responses || [];
      return Promise.map(requests, function (request, i) {
        switch (statuses[i]) {
          case ABORTED:
            return ABORTED;
          case DUPLICATE:
          // kibi: when using segmented request, it should also check 'request._uniq._mergedResp' too
            return request._uniq.resp || request._uniq._mergedResp;
          default:
            const index = _.findIndex(requestsToFetch, request);
            if (index < 0) {
              // This means the request failed.
              return ABORTED;
            }
            return responses[index];
        }
      })
        .then(
          (res) => defer.resolve(res),
          (err) => defer.reject(err)
        );
    };

    // kibi: global object to be able to share cancelId through the promise chain
    const sirenData = {};
    // kibi: end

    // handle a request being aborted while being fetched
    const requestWasAborted = Promise.method(function (req, i) {
      if (statuses[i] === ABORTED) {
        defer.reject(new Error('Request was aborted twice?'));
      }

      execCount -= 1;
      if (execCount > 0) {
        // the multi-request still contains other requests
        return;
      }

      if (esPromise && _.isFunction(esPromise.abort)) {
        esPromise.abort();
        // kibi: here the whole http request get aborted
        if (sirenData.cancelId) {
          jobCancel.cancel(sirenData.cancelId);
        }
      }

      esPromise = ABORTED;

      return respond();
    });


    // attach abort handlers, close over request index
    statuses.forEach(function (req, i) {
      if (!isRequest(req)) return;
      req.whenAborted(function () {
        requestWasAborted(req, i).catch(defer.reject);
      });
    });

    // Now that all of THAT^^^ is out of the way, lets actually
    // call out to elasticsearch
    Promise.map(requestsToFetch, function (request) {
      return Promise.try(request.getFetchParams, void 0, request)
      // kibi: call to requestAdapter function of the related visualization
        .then(function (fetchParams) {
        // If the request is a default wildcard query
        // convert it to a match_all (and if there is another match_all, dedupe)
        // This conversion is done here - just pre-request - to limit the need
        // to make multiple checks/modifications in the state of the dashboards
        // when getting or setting the state. Making modifications at that point
        // would propagate through the dashboards with the getCounts requests for the
        // dashboard state metadata leading to inconsistent state between front and backend.
          if (fetchParams.body && fetchParams.body.query && fetchParams.body.query.bool && fetchParams.body.query.bool.must) {
          // Cloning to prevent modification of the upstream filter objects
            const mustClone = _.cloneDeep(fetchParams.body.query.bool.must);
            fetchParams.body.query.bool.must = uniqFilters(mustClone.map(query => {
              if (_.isEqual(query, { query_string: { query: "*", analyze_wildcard: true } })) {
                query = { match_all: {} };
              }

              return query;
            }));
          }

          if (fetchParams.getSource) {
            const source = fetchParams.getSource();
            if (source && source.vis && source.vis.requestAdapter) {
              source.vis.requestAdapter(fetchParams);
            }
          }
          return fetchParams;
        })
      // kibi: end
        .then(function (fetchParams) {
          return (request.fetchParams = fetchParams);
        })
        .then(value => ({ resolved: value }))
        .catch(error => ({ rejected: error }));
    })
      .then(function (results) {
        const requestsWithFetchParams = [];
        // Gather the fetch param responses from all the successful requests.
        results.forEach((result, index) => {
          if (result.resolved) {
            requestsWithFetchParams.push(result.resolved);
          } else {
            const request = requestsToFetch[index];
            request.handleFailure(result.rejected);
            requestsToFetch[index] = undefined;
          }
        });
        // The index of the request inside requestsToFetch determines which response is mapped to it. If a request
        // won't generate a response, since it already failed, we need to remove the request
        // from the requestsToFetch array so the indexes will continue to match up to the responses correctly.
        requestsToFetch = requestsToFetch.filter(request => request !== undefined);
        return strategy.reqsFetchParamsToBody(requestsWithFetchParams);
      })
      .then(function (body) {
      // while the strategy was converting, our request was aborted
        if (esPromise === ABORTED) {
          throw ABORTED;
        }

        const id = strategy.id;
        let client = (id && id.includes('admin')) ? esAdmin : es;
        // kibi:
        // if the strategy provides a client use it instead of the default one.
        client = strategy.client ? strategy.client : client;


        // kibi:
        // if strategy.clientMethod == 'msearch' and task_timeout detected
        // add a task_timeout parameter to the msearch
        const params = {
          body
        };

        // flag to search request to avoid spurious warn log in elasticsearch
        if (
          config.get('siren:elasticsearch:searchErrorTrace') &&
        (strategy.clientMethod === 'msearch' || strategy.clientMethod === 'search')
        ) {
          params.error_trace = true;
        };

        if (strategy.clientMethod === 'msearch') {
          const results = extractHighestTaskTimeoutFromMsearch(body);
          if (results.taskTimeout !== 0) {
            params.task_timeout = results.taskTimeout;
            params.body = results.body;
          }
        }

        if (strategy.clientMethod === 'search' || strategy.clientMethod === 'msearch') {
          sirenData.cancelId = getUUID.get();
          params.headers = {
            'X-Opaque-Id': sirenData.cancelId
          };
        }
        // kibi: end

        return (esPromise = client[strategy.clientMethod](params));
      })
      .then(function (clientResp) {
        return strategy.getResponses(clientResp);
      })
      .then(respond)
      .catch(function (err) {
        if (err === ABORTED) respond();
        else defer.reject(err);
      });

    // return our promise, but catch any errors we create and
    // send them to the requests
    return defer.promise
      .catch(function (err) {
        requests.forEach(function (req, i) {
          if (statuses[i] !== ABORTED) {
            req.handleFailure(err);
          }
        });
      });

  }

  return callClient;
}
