import _ from 'lodash';
import chrome from 'ui/chrome';
import { Scanner } from 'ui/utils/scanner';
import { StringUtils } from 'ui/utils/string_utils';
import { SavedObjectsClient } from 'ui/saved_objects';

// kibi: imports
import { jdbcDatasourceTranslate } from 'plugins/investigate_core/management/sections/kibi_datasources/services/jdbc_datasource_translate';
import angular from 'angular';
// kibi: end

export class SavedObjectLoader {
  constructor(
    SavedObjectClass, kbnIndex, kbnUrl,
    { $http, caching: { cache, find, get, scanAll } = {}, mapHit, exclude, jdbcDatasources } = {},
  ) {
    // kibi: kibi properties
    this.mapHit = mapHit;
    this.cache = cache;
    this.cacheGet = get;
    this.cacheFind = find;
    this.cacheScanAll = scanAll;
    this.exclude = exclude;
    this.reservedCharactersRegex = new RegExp('[+\\-=&|><!(){}[\\\]^"~*?:]', 'g');
    this.jdbcDatasources = jdbcDatasources;
    // kibi: end

    this.type = SavedObjectClass.type;
    this.Class = SavedObjectClass;
    this.lowercaseType = this.type.toLowerCase();
    this.kbnIndex = kbnIndex;
    this.kbnUrl = kbnUrl;
    // kibi: removed this.esAdmin

    // kibi: removed scanner

    this.loaderProperties = {
      name: `${ this.lowercaseType }s`,
      noun: StringUtils.upperFirst(this.type),
      nouns: `${ this.lowercaseType }s`,
    };

    this.savedObjectsClient = new SavedObjectsClient($http, chrome.getBasePath(), Promise); // kibi: added parameters
  }

  /**
   * Retrieve a saved object by id. Returns a promise that completes when the object finishes
   * initializing.
   * @param id
   * @returns {Promise<SavedObject>}
   */
  get(id) {
    let cacheKey;
    if (id) {
      cacheKey = `${this.lowercaseType}-id-${id}`;
    }
    // kibi: get from cache
    if (this.cacheGet && cacheKey && this.cache && this.cache.has(cacheKey)) {
      return this.cache.get(cacheKey);
    }
    const promise = (new this.Class(id)).init();
    if (this.cacheGet && cacheKey && this.cache) {
      // kibi: put into cache
      this.cache.set(cacheKey, promise);
    }
    return promise;
  }

  urlFor(id) {
    return this.kbnUrl.eval(`#/${ this.lowercaseType }/{{id}}`, { id: id });
  }

  delete(ids) {
    ids = !_.isArray(ids) ? [ids] : ids;

    const deletions = ids.map(id => {
      const savedObject = new this.Class(id);
      return savedObject.delete();
    });

    // kibi: clear the cache
    this.clearCache();

    return Promise.all(deletions);
  }

  /**
   * Updates source to contain an id and url field, and returns the updated
   * source object.
   * @param source
   * @param id
   * @returns {source} The modified source object, with an id and url field.
   */
  mapHitSource(source, id) {
    source.id = id;
    source.url = this.urlFor(id);
    // kibi: alter the contents of a source
    if (this.mapHit) {
      this.mapHit(source);
    }
    return source;
  }

  /**
   * Deprecated - use find instead
   * Kept here for backward compatibility with existing plugins
   */
  scanAll(queryString, pageSize = 1000) {
    // kibi: use savedObjectsClient
    const cacheKey = `${this.lowercaseType}-${queryString || ''}-scanAll`;
    if (this.cacheScanAll && this.cache && this.cache.has(cacheKey)) {
      return Promise.resolve(this.cache.get(cacheKey));
    }

    const promise = this.savedObjectsClient.find({
      search: queryString,
      type: this.lowercaseType,
      perPage: pageSize
    })
      .then(resp => {
        const result = Object.assign(resp, resp.hits);
        if (this.cacheScanAll && this.cache) {
          this.cache.set(cacheKey, result);
        }
        return result;
      });
    if (this.cacheScanAll && this.cache) {
      this.cache.set(cacheKey, promise);
    }
    return promise;
    // kibi: end
  }

  /**
   * Updates hit.attributes to contain an id and url field, and returns the updated
   * attributes object.
   * @param hit
   * @returns {hit.attributes} The modified hit.attributes object, with an id and url field.
   */
  mapSavedObjectApiHits(hit) {
    const object = this.mapHitSource(hit.attributes, hit.id);
    // check type of object and add extra params if it is 'visualization'
    if (_.isFunction(this.mapHits)) {
      return this.mapHits(object);
    };
    return object;
  }

  /**
   * kibi: get dashboards from the Saved Object API.
   * TODO: Rather than use a hardcoded limit, implement pagination. See
   * https://github.com/elastic/kibana/issues/8044 for reference.
   *
   * @param searchString
   * @param removeReservedChars - //kibi: flag for removing reserved characters
   * @param size - The number of documents to return
   * @param exclude - A list of fields to exclude
   * @returns {Promise}
   */

  find(searchString, removeReservedChars = true, size = 100, doNotFetchJDBC = false, ignoreBroken = false) {
    if (!searchString) {
      searchString = null;
    }

    // kibi: cache results
    const cacheKey = `${this.lowercaseType}-${searchString || ''}`;
    if (!ignoreBroken && this.cacheFind && this.type !== 'datasource' && this.cache && this.cache.has(cacheKey)) {
      return Promise.resolve(this.cache.get(cacheKey));
    }

    //kibi: if searchString contains reserved characters, split it with reserved characters
    // combine words with OR operator
    let safeQuery = '';
    if (removeReservedChars && searchString && searchString.match(this.reservedCharactersRegex)) {
      const words = searchString.split(this.reservedCharactersRegex).map((item) => item.trim());
      _.each(words, function (word, index) {
        if (index === 0) {
          safeQuery = word;
        } else {
          safeQuery += ' || ' + word;
        }
      });
    };

    //kibi: added searchFields param to search just in titles
    const promise = this.savedObjectsClient.find({
      type: this.lowercaseType,
      search: safeQuery || searchString ? `${searchString}*` : undefined,
      searchFields: 'title',
      perPage: size,
      exclude: this.exclude,
      ignoreBroken: ignoreBroken
    })
      .then((resp) => {
        const result = {
          total: resp.total,
          hits: resp.savedObjects.map((savedObject) => this.mapSavedObjectApiHits(savedObject))
        };

        if (this.type === 'datasource' && !doNotFetchJDBC) {
          return this.jdbcDatasources.list().then(datasources => {
            _.each(datasources, datasource => {
              result.hits.push(jdbcDatasourceTranslate.datasourceToSavedDatasource(datasource));
            });
            return result;
          }).catch(err => {
            if (err.status === 403) {
              if (err.data && err.data.error && err.data.error.type === 'security_exception') {
              // ignore access error
              // user will simply not see the JDBC datasources if s/he has no access
                return result;
              } else if (err.data && err.data.error) {
                throw new Error('Access to JDBC datasources forbidden.' + JSON.stringify(err.data.error));
              } else {
                throw new Error('Access to JDBC datasources forbidden.' + JSON.stringify(err));
              }
            } else {
              throw err;
            }
          });
        }

        if (!ignoreBroken && this.cache && this.cacheFind) {
          this.cache.set(cacheKey, result);
        }

        //kibi: when escaping reserved characters in query, we do not filter with reserved chracters
        //if searchString contains reserved characters check titles of hits
        if (safeQuery) {
          result.hits = _.filter(result.hits, function (hit) {
            return angular.lowercase(hit.title).includes(angular.lowercase(searchString));
          });
          result.total = result.hits.length;
        }
        //kibi: end
        return result;
      });
    if (!ignoreBroken && this.cache && this.cacheFind) {
      this.cache.set(cacheKey, promise);
    }
    return promise;
  }

  /**
   * kibi: clears the cache
   */
  clearCache() {
    if (this.cache) {
      this.cache.invalidate();
    }
  }
}
