import { has, get } from 'lodash';
import { FilterMetaPropertyIndexError, FilterMetaPropertyKeyError } from 'ui/errors';

export function FilterBarLibMapRangeProvider(Promise, courier) {
  return function (filter) {
    const isScriptedRangeFilter = isScriptedRange(filter);
    if (!filter.range && !isScriptedRangeFilter) {
      return Promise.reject(filter);
    }

    return courier
      .indexPatterns
      .get(filter.meta.index)
      .then(function (indexPattern) {
        const type = 'range';
        const key = isScriptedRangeFilter ? filter.meta.field : Object.keys(filter.range)[0];
        // kibi: handle case where fields are not present (indexPattern no longer exists);
        if (!indexPattern.fields) {
          return Promise.reject(new FilterMetaPropertyIndexError(filter.meta.index));
        }
        // kibi: end
        const field = indexPattern.fields.byName[key];
        // kibi: handle case where the field is no longer present in the index-pattern
        if (!field) {
          return Promise.reject(new FilterMetaPropertyKeyError(key, filter.meta.index));
        }
        // kibi: end
        const convert = field.format.getConverterFor('text');
        const range = isScriptedRangeFilter ? filter.script.script.params : filter.range[key];

        let left = has(range, 'gte') ? range.gte : range.gt;
        if (left == null) left = -Infinity;

        let right = has(range, 'lte') ? range.lte : range.lt;
        if (right == null) right = Infinity;
        // kibi: added to make correct label for date fields with custom mapping
        let value = `${convert(left, field.mappingFormat)} to ${convert(right, field.mappingFormat)}`;
        if (field.mappingFormat) {
          value += ' [' + field.mappingFormat + ']';
        };
        // kibi: end
        return { type, key, value };
      })
      .catch(err => {
        if (err instanceof FilterMetaPropertyIndexError || err instanceof FilterMetaPropertyKeyError) {
          return Promise.reject(err);
        }
        return Promise.reject(new FilterMetaPropertyIndexError(filter.meta.index));
      });
  };

}

function isScriptedRange(filter) {
  const params = get(filter, ['script', 'script', 'params']);
  return params && Object.keys(params).find(key => ['gte', 'gt', 'lte', 'lt'].includes(key));
}
