import _ from 'lodash';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { translateToQuery } from './lib/translate_to_query';

function stripSearchItem(item) {
  // remove all except
  Object.keys(item)
    .forEach(function (key) {
      if (['type', 'id', 'x', 'y', 'd'].indexOf(key) === -1) {
        delete item[key];
      }
    });

  if (item.d) {
    Object.keys(item.d)
      .forEach(function (key) {
        if (['isRoot', 'entity', 'widgets', 'range'].indexOf(key) === -1) {
          delete item.d[key];
        }
        if (key === 'isRoot' && !item.d.isRoot) {
          delete item.d[key];
        }
      });
  }

  if (item.d.widgets) {
    for (let i = 0; i < item.d.widgets.length; i++) {
      delete item.d.widgets[i].title;
    }
  }

  // remove all except id and useGlobalTimeFilter from entity
  if (item.d.entity) {
    Object.keys(item.d.entity)
      .forEach(function (key) {
        if (['id', 'useGlobalTimeFilter'].indexOf(key) === -1) {
          delete item.d.entity[key];
        }
        if (key === 'useGlobalTimeFilter' && !item.d.entity.useGlobalTimeFilter) {
          delete item.d.entity[key];
        }
      });
  }
}

function stripRelationItem(item) {
  Object.keys(item)
    .forEach(function (key) {
      if (['type', 'id', 'id1', 'id2', 'd'].indexOf(key) === -1) {
        delete item[key];
      }
    });

  if (item.d) {
    Object.keys(item.d)
      .forEach(function (key) {
        if (['rel'].indexOf(key) === -1) {
          delete item.d[key];
        }
      });
  }

  if (item.d.rel) {
    Object.keys(item.d.rel)
      .forEach(function (key) {
        if (['id'].indexOf(key) === -1) {
          delete item.d.rel[key];
        }
      });
  }
}

function decorateRelationsItem(item, relations) {

  if (item.id === undefined || item.id1 === undefined || item.id2 === undefined) {
    throw Error('Relations should have id, id1 and id2 properties');
  }

  const rel = _.find(relations, 'id', item.d.rel.id);
  if (rel) {
    const inverseRel = _.find(relations, 'id', rel.inverseOf);
    item.d.rel = _.cloneDeep(rel);
    item.d.inverseRel = _.cloneDeep(inverseRel);
  }
}

function decorateSearchItem(item, searches, entities, visualizations, visualizationIdTitleMap) {

  if (item.id === undefined) {
    throw Error('Search should have id property');
  }

  const entity = _.find(entities, 'id', item.d.entity.id);
  if (entity && entity.type === EntityType.SAVED_SEARCH) {
    const useGlobalTimeFilter = item.d && item.d.entity && item.d.entity.useGlobalTimeFilter;
    const search = _.find(searches, 'id', item.d.entity.id);

    item.d.entity = _.cloneDeep(entity);
    item.d.entity.indices = search.indices;

    if (search.timeFieldName) {
      if (useGlobalTimeFilter) {
        item.d.entity.useGlobalTimeFilter = true;
      }
      item.d.entity.timeFieldName = search.timeFieldName;
    }

    if (search.filters && search.filters.length > 0) {
      item.d.entity.filters = search.filters;
    }

    // add title for each vis
    if (item.d.widgets) {
      for (const vis of item.d.widgets) {
        const found = _.find(visualizations, 'id', vis.id);
        if (found) {
          vis.title = found.title;
        } else if (visualizationIdTitleMap.get(vis.id)) {
          vis.title = visualizationIdTitleMap.get(vis.id);
        } else {
          vis.title = 'Missing';
        }
      }
    }
  } else {
    // EID
  }
}

export async function getCoatVisualizations(savedVisualization, savedSearches, panels) {
  if (!panels || panels.length === 0) {
    return [];
  }
  const searches = await savedSearches.find();
  const visualisations = await savedVisualization.find();

  return panels
    .map(panel => {
      let item = _.find(visualisations.hits, 'id', panel.id);
      if (!item) {
        item = _.find(searches.hits, 'id', panel.id);
      }

      if (!item) {
        return {
          id: panel.id,
          title: 'Missing',
          panelIndex: panel.panelIndex,
          type: panel.type,
          visType: panel.type === 'visualization' || null
        };
      }

      let indexPattern = null;
      const visSearch =  _.find(searches.hits, 'id', item.savedSearchId);
      if (visSearch) {
        indexPattern = JSON.parse(visSearch.kibanaSavedObjectMeta.searchSourceJSON).index;
      }


      const ret = {
        id: panel.id,
        panelIndex: panel.panelIndex,
        type: panel.type,
        visType: panel.type === 'visualization' ? item.type.name : null,
        title: item.title,
        savedSearchId: panel.type === 'visualization' ? item.savedSearchId : item.id,
        indexPattern: indexPattern,
        dashboard360Compatible: item.type && item.type.dashboard360Compatible ? item.type.dashboard360Compatible : false
      };
      return ret;
    });
}

export async function getCoatSearches(savedSearches, indexPatterns) {
  const searches = await savedSearches.find();
  const ret = [];
  for (const search of searches.hits) {
    const searchSource = JSON.parse(search.kibanaSavedObjectMeta.searchSourceJSON);
    const indexPattern = await indexPatterns.get(searchSource.index);


    let timeFieldName;
    if (indexPattern.timeFieldName) {
      timeFieldName = indexPattern.timeFieldName;
    }
    let filters = [];
    if (searchSource.filter && searchSource.filter.length > 0) {
      filters = searchSource.filter.map(filter => {
        return translateToQuery(filter);
      });
    }
    if (searchSource.query && !searchSource.query.match_all) {
      // TODO: maybe add as yet another filter
      filters.push(translateToQuery(searchSource.query));
    }
    ret.push({
      id: search.id,
      title: search.title,
      indices: [indexPattern.title],
      timeFieldName,
      filters
    });
  }

  return ret;
}

export function decorateCoatItems(searches, entities, relations, visualizations, items, visualizationIdTitleMap) {
  if (!items) return;

  for (const item of items) {
    if (item.type === 'node' && item.d && item.d.entity && item.d.entity.id) {
      decorateSearchItem(item, searches, entities, visualizations, visualizationIdTitleMap);
    } else if (item.type === 'link' && item.d && item.d.rel && item.d.rel.id) {
      decorateRelationsItem(item, relations);
    }
  }
}

export function stripCoatItems(items) {
  for (const item of items) {
    if (item.type === 'node' && item.d && item.d.entity && item.d.entity.type === EntityType.SAVED_SEARCH) {
      stripSearchItem(item);
    } else if (item.type === 'link') {
      stripRelationItem(item);
    }
  }
}

export function isThereAssignedVisBasedOnMainSearch(items) {
  const mainNode = findMainCoatNode(items);
  if (mainNode && mainNode.d && mainNode.d.widgets) {
    return  mainNode.d.widgets.length > 0;
  } else {
    return false;
  }
}

export function findMainCoatNode(items) {
  if (items && items.length > 0) {
    for (const item of items) {
      if (item.type === 'node' && item.d && item.d.entity && item.d.entity.type === EntityType.SAVED_SEARCH && item.d.isRoot) {
        return item;
      }
    }
  }
  return null;
}

export function setCountForMainSearch(items, count) {
  const mainNode = findMainCoatNode(items);
  if (mainNode) {
    if (count !== undefined && count !== null) {
      mainNode.d.entity.count = count;
    } else {
      mainNode.d.entity.count = undefined;
    }
  }
}

export function findMainCoatNodeSearchId(coatJSON) {
  if (!coatJSON) {
    return null;
  }

  const coat = JSON.parse(coatJSON);

  if (coat.items && coat.items.length > 0) {
    for (const item of coat.items) {
      if (item.type === 'node' && item.d && item.d.entity && item.d.isRoot) {
        return item.d.entity.id;
      }
    }
  }
  return null;
}

export async function getAllSearchBasedDashboardIds(savedDashboards) {
  return savedDashboards.find().then(function (dashboards) {
    return _(dashboards.hits)
      .map(hit => {
        const mainSearchId = findMainCoatNodeSearchId(hit.coatJSON);
        if (mainSearchId) {
          return hit.id;
        }
        return null;
      })
      .filter(id => id !== null)
      .value();
  });
}

export function getParsedDecoratedCoat(coatJSON, siren, visualizationIdTitleMap) {
  const coat = JSON.parse(coatJSON);
  if (!_.isEmpty(coat)) {
    decorateCoatItems(
      siren.searches,
      siren.entities,
      siren.relations,
      siren.visualizations,
      coat.items,
      visualizationIdTitleMap
    );
  }
  return coat;
}

export function shouldTimefilterBeEnabled(items) {
  if (!items) {
    return false;
  }
  if (items && items.length > 0) {
    for (const item of items) {
      if (item.type === 'node' && item.d && item.d.entity && item.d.entity.useGlobalTimeFilter) {
        return true;
      }
    }
  }
  return false;
}

export function checkTreeItems(dash) {
  const entities = dash.siren.entities;
  const relations = dash.siren.relations;
  _.each(dash.siren.coat.items, item => {
    if (item.type === 'link') {
      if (!item.d.rel.domain || !item.d.rel.range) {
        item.d.rel.notFound = true;
      } else {
        const relation = _.find(relations, 'id', item.d.rel.id);
        const inverseRelation = _.find(relations, 'id', item.d.inverseRel.id);
        if (!relation) {
          item.d.rel.notFound = true;
        }
        if (!inverseRelation) {
          item.d.inverseRel.notFound = true;
        }
      }
    } else if (item.type === 'node') {
      const entity = _.find(entities, 'id', item.d.entity.id);
      if (!entity) {
        item.d.entity.notFound = true;
      }
    }
  });
}
