import { AbstractOntologyModel } from 'ui/kibi/components/ontology_model/ontology_model_abstract';
import _ from 'lodash';
import Promise from 'bluebird';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { hash } from 'ui/kibi/utils/hash';
import { DataModelErrorType } from 'plugins/investigate_core/management/sections/data_model/controllers/error_type';

function getAllIndexPatterns(indexPatterns) {
  return indexPatterns.getIds()
    .then(ids => {
      return Promise.map(ids, id => {
        return indexPatterns.get(id);
      });
    });
};

function getAllSavedObjects(service) {
  return service.find()
    .then(foundObjects => {
      return Promise.map(foundObjects.hits, foundObject => {
        return service.get(foundObject.id);
      });
    });
};

function buildRelation(savedRelation, entityMap, { isInverse = false }) {
  const relation = {
    joinType: savedRelation.joinType ? savedRelation.joinType : null,
    timeout: savedRelation.timeout,
    _objects: {
      savedRelation: savedRelation
    }
  };

  if (!isInverse) {
    relation.id = savedRelation.id;
    relation.directLabel = savedRelation.directLabel;
    relation.inverseLabel = savedRelation.inverseLabel;
    relation.inverseOf = savedRelation.inverseOf;
  } else {
    relation.id = savedRelation.inverseOf;
    relation.directLabel = savedRelation.inverseLabel;
    relation.inverseLabel = savedRelation.directLabel;
    relation.inverseOf = savedRelation.id;
  }

  const domain = _.cloneDeep(entityMap[savedRelation.domainId]);
  domain.field = savedRelation.domainField;
  const range = _.cloneDeep(entityMap[savedRelation.rangeId]);
  range.field = savedRelation.rangeField;

  if (!isInverse) {
    relation.domainField = savedRelation.domainField;
    relation.rangeField = savedRelation.rangeField;
    relation.domain = domain;
    relation.range = range;
    relation.title = `${domain.label} -> ${savedRelation.directLabel} -> ${range.label}`;
  } else {
    relation.domainField = savedRelation.rangeField;
    relation.rangeField = savedRelation.domainField;
    relation.domain = range;
    relation.range = domain;
    relation.title = `${range.label} -> ${savedRelation.inverseLabel} -> ${domain.label}`;
  }

  return relation;
}

/**
 * This is a simple implementation of the ontology model.
 * It does not use triples nor inference nor SPARQL queries.
 * It's just based on simple JSON objects
 */
class OntologyModelSimple extends AbstractOntologyModel {

  constructor(savedSearches, savedEids, indexPatterns, savedRelations) {
    super(savedSearches, savedEids, indexPatterns, savedRelations);
  }

  formatSavedSearches(savedSearchesArray, indexPatternsArray, entities, broken) {
    for (let i = 0; i < savedSearchesArray.length; i++) {
      const savedSearch = savedSearchesArray[i];
      const entity = {
        id: savedSearch.id,
        label: savedSearch.title,
        type: EntityType.SAVED_SEARCH,
        primaryKeys: [],
        singleValues: [],
        parentId: null,
        instanceLabel: {},
        _objects: {}
      };

      const sirenAttibutes = savedSearch.siren;
      if (sirenAttibutes) {
        if (sirenAttibutes.parentId) {
          entity.parentId = sirenAttibutes.parentId;
        }

        if (sirenAttibutes.ui) {
          const ui = sirenAttibutes.ui;
          if (ui.icon) {
            entity.icon = ui.icon;
          }

          if (ui.color) {
            entity.color = ui.color;
          }

          if (ui.shortDescription) {
            entity.shortDescription = ui.shortDescription;
          }

          if (ui.instanceLabelType) {
            entity.instanceLabel.type = ui.instanceLabelType;
          }

          if (ui.instanceLabelValue) {
            entity.instanceLabel.value = ui.instanceLabelValue;
          }
        }
      }

      if (savedSearch.kibanaSavedObjectMeta && savedSearch.kibanaSavedObjectMeta.searchSourceJSON) {
        const kibanaSavedObjectMeta = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);

        if (kibanaSavedObjectMeta.index) {
          entity.indexPattern = kibanaSavedObjectMeta.index;
        }
        if (kibanaSavedObjectMeta.filter) {
          entity.filter = JSON.stringify(kibanaSavedObjectMeta.filter);
        }
        if (kibanaSavedObjectMeta.query) {
          entity.query = JSON.stringify(kibanaSavedObjectMeta.query);
        }
      }

      if (entity.indexPattern) {
        const indexPattern = _.find(indexPatternsArray, 'id', entity.indexPattern);

        if (!indexPattern || broken) {
          entity._objects.error = 'Missing data indices';
          entity._objects.errorType = DataModelErrorType.INDEX_PATTERN_MISSING_DATA_INDICES;
          entity._objects.errorRecoveredIndexPatternAttributes = {
            title: entity.label || entity.indexPattern.split(':')[1]
          };
          entities[i] = entity;
          continue;
        }

        const fields = indexPattern.fields;
        for (let i = 0; i < fields.length; i++) {
          const field = fields[i];

          if (field.primaryKey) {
            entity.primaryKeys.push(field.name);
          }
          if (field.singleValue) {
            entity.singleValues.push(field.name);
          }
        }

        entity._objects.indexPattern = indexPattern;
      }

      entity._objects.savedSearch = savedSearch;

      entities[i] = entity;
    }
  }

  getEntityList() {
    return Promise.all([
      getAllSavedObjects(this.savedSearches),
      getAllSavedObjects(this.savedEids),
      getAllIndexPatterns(this.indexPatterns)
    ])
      .then(([ savedSearchesArray, savedEidsArray, indexPatternsArray ]) => {
        const savedSearchesLength = savedSearchesArray.length;
        const savedEidsLength = savedEidsArray.length;
        const entities = new Array(savedSearchesLength + savedEidsLength);

        // Saved searches
        this.formatSavedSearches(savedSearchesArray, indexPatternsArray, entities, false);

        // EIDs
        for (let i = 0; i < savedEidsLength; i++) {
          const savedEid = savedEidsArray[i];
          const entity = {
            id: savedEid.id,
            label: savedEid.title,
            type: EntityType.VIRTUAL_ENTITY,
            parentId: null,
            _objects: {}
          };

          const sirenAttibutes = savedEid.siren;
          if (sirenAttibutes && sirenAttibutes.ui) {
            const ui = sirenAttibutes.ui;
            if (ui.icon) {
              entity.icon = ui.icon;
            }
            if (ui.color) {
              entity.color = ui.color;
            }

            if (ui.shortDescription) {
              entity.shortDescription = ui.shortDescription;
            }
          }

          entity._objects.savedEid = savedEid;

          entities[i + savedSearchesLength] = entity;
        }

        return entities;
      });
  };

  getBrokenSavedSearches() {
    return Promise.all([
      getAllSavedObjects(this.savedSearches),
      this.savedSearches.find('', true, 100, false, true),
      getAllIndexPatterns(this.indexPatterns)
    ])
      .then(([ returnedSavedSearches, savedSearchesArrayWithBrokenOnes, indexPatternsArray ]) => {
        const onlyBrokenOnes = _.filter(savedSearchesArrayWithBrokenOnes.hits, hit => {
          const index = _.findIndex(returnedSavedSearches, 'id', hit.id);
          return index === -1;
        });
        const entities = new Array(onlyBrokenOnes.length);

        this.formatSavedSearches(onlyBrokenOnes, indexPatternsArray, entities, true);
        return entities;
      });
  }

  getEntityMap() {
    return this.getEntityList()
      .then(entityList => {
        const entityMap = _.reduce(entityList, (map, entity) => {
          map[entity.id] = entity;
          return map;
        }, {});

        return entityMap;
      });
  };

  getRelationList() {
    return Promise.all([
      getAllSavedObjects(this.savedRelations),
      this.getEntityMap()
    ])
      .then(([ savedRelationsArray, entityMap ]) => {
        const relations = new Array();

        for (const savedRelation of savedRelationsArray) {
          relations.push(buildRelation(savedRelation, entityMap, { isInverse: false }));
          relations.push(buildRelation(savedRelation, entityMap, { isInverse: true }));
        }

        return relations;
      });
  }

  getRelationMap() {
    return this.getRelationList()
      .then(relationList => {
        const relationMap = _.reduce(relationList, (map, relation) => {
          map[relation.id] = relation;
          return map;
        }, {});

        return relationMap;
      });
  };

  getRelationsByDomain(entityId) {
    return this.getRelationList()
      .then(relationList => {
        const relations = [];
        for (let i = 0; i < relationList.length; i++) {
          const relation = relationList[i];

          if (relation.domain.id === entityId) {
            relations.push(relation);
          }
        }

        return relations;
      });
  };

  getUniqueRelationLabels() {
    return this.getRelationList()
      .then(relationList => {
        const relationLabelsSet = _.reduce(relationList, (total, rel) => {
          total.add(rel.directLabel);
          total.add(rel.inverseLabel);
          return total;
        }, new Set());
        return Array.from(relationLabelsSet);
      });
  };

  getUniqueRelationLabelPairs() {
    return this.getRelationList()
      .then(relationList => {
        const relationLabelPairs = _.reduce(relationList, (total, rel) => {
          total.push({
            directLabel : rel.directLabel,
            inverseLabel: rel.inverseLabel
          });
          return total;
        }, []);
        const uniqueRelationLabelPairs = _.uniq(relationLabelPairs, pair => {
          return pair.directLabel + pair.inverseLabel;
        });
        return uniqueRelationLabelPairs;
      });
  };

  getRangesForEntityId(entityId) {
    return this.getRelationList()
      .then(relations => {
        const ranges = _.reduce(relations, (total, relation) => {
          if (relation.domain.id === entityId) {
            total.push(relation.range);
          }
          return total;
        }, []);

        return ranges;
      });
  };

  getAllPaths() {
    return Promise.all([
      this.getEntityMap(),
      this.getRelationList()
    ])
      .then(([ entityMap, relations ]) => {
        // a path is <e1 p0 e2 p1 e3>
        const paths = [];

        for (const p0 of relations) {
          const e1 = entityMap[p0.domain.id];
          const e2 = entityMap[p0.range.id];

          // + "  ?e2 " + getPrefixed(TYPE) + " \"" + EntityType.SAVED_SEARCH.toString() + "\" .\n"
          if (e2.type !== EntityType.SAVED_SEARCH) {
            continue;
          }

          // We go to the middle entity via one of his primary keys.
          // Nothing to aggregate as the primary key will lead to 1 document only.
          // + "  FILTER NOT EXISTS {\n"
          // + "    ?p0 " + getPrefixed(IN_FIELD) + " ?p0inField .\n"
          // + "    ?e2 " + getPrefixed(PRIMARY_KEY) + " ?p0inField .\n"
          // + "  }\n"
          if (e2.primaryKeys && e2.primaryKeys.length &&
            e2.primaryKeys.indexOf(p0.range.field) >= 0) {
            continue;
          }

          for (const p1 of relations) {
            if (p0.range.id !== p1.domain.id) {
              continue;
            }

            // The first relation is the opposite of the second one, and we use a single value field
            // to go to the middle entity. This means a single value to go to entity 2 and to go out
            // from entity 2. Nothing aggregatable here.
            // + "  FILTER NOT EXISTS {\n"
            // + "    ?p0 " + OWL_INVERSE_OF + " ?p1 .\n"
            // + "    ?p0 " + getPrefixed(IN_FIELD) + " ?p0inField .\n"
            // + "    ?e2 " + getPrefixed(SINGLE_VALUE) + " ?p0inField .\n"
            // + "  }\n"
            if (p0.inverseOf === p1.id &&
              e2.singleValues && e2.singleValues.length &&
              e2.singleValues.indexOf(p0.range.field) >= 0) {
              continue;
            }

            const e3 = entityMap[p1.range.id];

            // This is the rule that enforces the "being on the same index" of the info needed
            // to perform the aggregation.
            // If we're going from entity 2 via its primary key and we're not targeting
            // the same index, the relation is filtered.
            // + "  FILTER NOT EXISTS {\n"
            // + "    ?p1 " + getPrefixed(OUT_FIELD) + " ?p1outField .\n"
            // + "    ?e2 " + getPrefixed(PRIMARY_KEY) + " ?p1outField .\n"
            // + "    FILTER NOT EXISTS {\n"
            // + "      ?e2 " + getPrefixed(INDEX_PATTERN) + " ?e2e3IndexPattern .\n"
            // + "      ?e3 " + getPrefixed(INDEX_PATTERN) + " ?e2e3IndexPattern .\n"
            // + "    }\n"
            // + "  }\n"
            if (e2.primaryKeys && e2.primaryKeys.length &&
              e2.primaryKeys.indexOf(p1.domain.field) >= 0) {
              if (e2.indexPattern !== e3.indexPattern) {
                continue;
              }
            }

            // https://github.com/sirensolutions/kibi-internal/issues/7068
            // + "  FILTER NOT EXISTS {\n"
            // + "    ?e3 " + getPrefixed(TYPE) + " \"" + EntityType.SAVED_SEARCH.toString() + "\" .\n"
            // + "    ?p1 " + getPrefixed(IN_FIELD) + " ?p1inFieldPk .\n"
            // + "    FILTER NOT EXISTS {\n"
            // + "    ?e3 " + getPrefixed(PRIMARY_KEY) + " ?p1inFieldPk .\n"
            // + "    }\n"
            // + "  }\n"
            if (e3.type === EntityType.SAVED_SEARCH &&
              !(e3.primaryKeys && e3.primaryKeys.length &&
              e3.primaryKeys.indexOf(p1.range.field) >= 0)) {
              continue;
            }

            paths.push({
              id: hash([e1.id, p0.id, e2.id, p1.id, e3.id]),
              sourceEntity: e1,
              sourceToMiddleRelation: p0,
              middleEntity: e2,
              middleToTargetRelation: p1,
              targetEntity: e3
            });
          }
        }

        return paths;
      });
  };

  clearCache() {
    return Promise.resolve();
  }
};


export function OntologyModelSimpleProvider(savedSearches, savedEids, indexPatterns, savedRelations) {
  return new OntologyModelSimple(savedSearches, savedEids, indexPatterns, savedRelations);
}
