import angular from 'angular';
import _ from 'lodash';
import { DelayExecutionHelperFactory } from 'ui/kibi/helpers/delay_execution_helper';
import { onDashboardPage } from 'ui/kibi/utils/on_page';
import { SpinnerStatus } from 'ui/kibi/spinners/spinner_status';
import { getAllSearchBasedDashboardIds } from 'ui/kibi/components/dashboards360/coat_tree';

export function KibiNavBarHelperFactory(dashboardGroups, kibiState, globalState, getAppState, createNotifier, Private, $rootScope,
  savedDashboards, timefilter, useVisualizationForCountOnDashboard) {

  const notify = createNotifier({
    location: 'Siren Navbar Helper'
  });

  const DelayExecutionHelper = Private(DelayExecutionHelperFactory);

  const NO_METADATA_UPDATE = Symbol.for('no metadata to update on any dashboard');

  /*
  * Private Methods
  */
  const updateCounts = function (dashboardsIds, reason, forceUpdate = false) {
    if (!dashboardsIds.length) {
      return;
    }
    if (console.debug) {  // eslint-disable-line no-console
      const msg = `KibiNavBar: requesting metadata update for following dashboards: ${JSON.stringify(dashboardsIds)} because: [${reason}]`;
      console.debug(msg); // eslint-disable-line no-console
    }

    _.each(dashboardGroups.getGroups(), g => {
      _.each(g.dashboards, d => {
        if (dashboardsIds.indexOf(d.id) > -1) {
          if (useVisualizationForCountOnDashboard.get(d.id)) {
            d.count = SpinnerStatus.FETCHING;
            dashboardGroups.emit('dashboardMetadataUpdated', { id: d.id });
            dashboardsIds = _.filter(dashboardsIds, dashboardId => {
              return d.id !== dashboardId;
            });
          }
        }
      });
    });

    return this.delayExecutionHelper.addEventData({
      forceUpdate: forceUpdate,
      ids: dashboardsIds
    });
  };


  // =================
  // Group computation and counts updates
  // =================

  function KibiNavBarHelper() {
    const updateCountsOnAppStateChange = function (diff) {
      if (diff.indexOf('query') === -1 && diff.indexOf('filters') === -1) {
        return;
      }
      // when appState changed get connected and selected dashboards
      const currentDashboard = kibiState.getDashboardOnView();
      if (!currentDashboard) {
        return;
      }
      this.updateAllCounts([currentDashboard.id], 'AppState change ' + angular.toJson(diff));
    };

    const updateCountsOnGlobalStateChange = function (diff) {
      const currentDashboard = kibiState.getDashboardOnView();
      if (!currentDashboard) {
        return;
      }

      if (diff.indexOf('filters') !== -1) {
        // the pinned filters changed, update counts on all selected dashboards
        this.updateAllCounts(null, 'GlobalState pinned filters change');
      } else if (diff.indexOf('time') !== -1) {
        this.updateAllCounts([currentDashboard.id], 'GlobalState time changed');
      } else if (diff.indexOf('refreshInterval') !== -1) {
        // force the count update to refresh all tabs count
        this.updateAllCounts(null, 'GlobalState refreshInterval changed', true);
      }
    };
    const updateCountsOnKibiStateReset = function (dashboardsIds) {
      this.updateAllCounts(dashboardsIds, 'KibiState reset');
    };

    const updateCountsOnKibiStateTime = function (dashboardId, newTime, oldTime) {
      this.updateAllCounts([dashboardId], `KibiState time changed on dashboard ${dashboardId}`);
    };

    const updateCountsOnKibiStateChange = function (diff) {
      // when kibiState changes get connected and selected dashboards
      const currentDashboard = kibiState.getDashboardOnView();
      if (!currentDashboard) {
        return;
      }
      // We are firing this function updateCountsOnKibiStateChange
      // for example when we are on 'dashboard a' but we click to 'dashboard b'.
      // And in this situation, we are starting the counting process for 'dashboard a'
      // but before we get the latest count for 'dashboard a' we are already switching to 'dashboard b'.
      // And It causes forever spinning error. To prevent that we are setting false and firing dashboard count query as it was.
      useVisualizationForCountOnDashboard.set(currentDashboard.id, false);
      if (diff.indexOf(kibiState._properties.groups) !== -1 || diff.indexOf(kibiState._properties.dashboards) !== -1) {
        this.updateAllCounts([currentDashboard.id], `KibiState change ${JSON.stringify(diff)}`);
      }
    };

    $rootScope.$listen(globalState, 'save_with_changes', (diff) => updateCountsOnGlobalStateChange.call(this, diff));

    this.appStateHandlerOff;
    const appStateHandler = (diff) => {
      updateCountsOnAppStateChange.call(this, diff);
    };
    $rootScope.$watch(getAppState, appState => {
      if (appState) {
        if (this.appStateHandlerOff) {
          // kibi: remove the old handler before attaching the new one
          this.appStateHandlerOff();
        }
        appState.on('save_with_changes', appStateHandler);
        this.appStateHandlerOff = function () {
          appState.off('save_with_changes', appStateHandler);
        };
      }
    });
    $rootScope.$listen(kibiState, 'save_with_changes', (diff) => updateCountsOnKibiStateChange.call(this, diff));
    $rootScope.$listen(kibiState, 'reset', (dashboardsIds) => updateCountsOnKibiStateReset.call(this, dashboardsIds));
    $rootScope.$listen(kibiState, 'time', updateCountsOnKibiStateTime.bind(this));

    // everywhere use this event !!! to be consistent
    // make a comment that it was required because not all components can listen to
    // esResponse
    $rootScope.$on('courier:searchRefresh', (event) => {
      if ((timefilter.refreshInterval.display !== 'Off')
        && (timefilter.refreshInterval.pause === false)) {
        const currentDashboard = kibiState.getDashboardOnView();
        if (!currentDashboard) {
          return;
        }
        this.updateAllCounts(null, 'courier:searchRefresh event', true);
      }
    });

    this.delayExecutionHelper = new DelayExecutionHelper(
      (data, alreadyCollectedData) => {
        if (alreadyCollectedData.ids === undefined) {
          alreadyCollectedData.ids = [];
        }
        if (alreadyCollectedData.forceUpdate === undefined) {
          alreadyCollectedData.forceUpdate = false;
        }
        if (data.forceUpdate) {
          alreadyCollectedData.forceUpdate = data.forceUpdate;
        }
        _.each(data.ids, (d) => {
          if (alreadyCollectedData.ids.indexOf(d) === -1) {
            alreadyCollectedData.ids.push(d);
          }
        });
      },
      (data) => {
        if (onDashboardPage()) {
          const forceUpdate = data.forceUpdate;
          const filteredDashboardsIds = dashboardGroups.getVisibleDashboardIds(data.ids);

          if (!filteredDashboardsIds.length) {
            return Promise.resolve(NO_METADATA_UPDATE);
          }

          // Warnings suppressed when search refreshed as they are already shown when the dashboards are first loaded
          return dashboardGroups.updateMetadataOfDashboardIds(filteredDashboardsIds, forceUpdate, true)
            .catch(notify.warning);
        }
      },
      750,
      DelayExecutionHelper.DELAY_STRATEGY.RESET_COUNTER_ON_NEW_EVENT
    );
  }

  /*
  * Public Methods
  */
  KibiNavBarHelper.prototype.updateAllCounts = function (dashboardsIds, reason, forceUpdate = false) {
    if (!dashboardsIds) {
      return getAllSearchBasedDashboardIds(savedDashboards)
        .then(ids => updateCounts.call(this, ids, reason, forceUpdate))
        .catch(err => {
          if (err.message !== DelayExecutionHelper.ERRORS.CANCELLED) {
            // notify only if error different that
            notify.error(err);
          }
        });
    } else {
      return updateCounts.call(this, dashboardsIds, reason, forceUpdate);
    }
  };

  KibiNavBarHelper.prototype.destroy = function () {
    if (this.appStateHandlerOff) {
      this.appStateHandlerOff();
    }
    if (this.delayExecutionHelper) {
      this.delayExecutionHelper.cancel();
    }
    if (this.delayExecutionHelper) {
      this.delayExecutionHelper.destroy();
    }
  };

  return new KibiNavBarHelper();
};
