import _ from 'lodash';
import { CrudType } from 'plugins/investigate_core/management/sections/data_model/services/crud_type';
import { DataModelPermissionsProvider } from 'plugins/investigate_core/management/sections/data_model/services/data_model_permissions';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { DatasourceTypes } from 'plugins/investigate_core/management/sections/kibi_datasources/lib/datasources';

export function DeleteHelperFactory(
  Promise, dashboardGroups, savedVisualizations,
  Private, config, jdbcDatasources, dataModel, ontologyModel, confirmModal, kbnUrl) {

  function DeleteHelper() {
    this.dataModelPermissions = Private(DataModelPermissionsProvider);
  }

  DeleteHelper.prototype._isIndexPatternSearch = function (entity) {
    return entity.type === EntityType.SAVED_SEARCH && entity.parentId === null;
  };

  /**
   * GetVisualisations returns visualisations that are used by the list of queries
   */
  DeleteHelper.prototype._getVisualisations = function (queryIds) {
    if (!queryIds) {
      return Promise.reject(new Error('Empty argument'));
    }
    return savedVisualizations.find('').then(function (resp) {
      const selectedQueries = [];

      const queryIds2 = _.map(queryIds, function (id) {
        return '"queryId":"' + id + '"';
      });
      const vis = _.filter(resp.hits, function (hit) {
        const list = _.filter(queryIds2, function (id, index) {
          if (hit.visState.indexOf(id) !== -1) {
            selectedQueries.push(queryIds[index]);
            return true;
          }
          return false;
        });
        return !!list.length;
      });
      return [ _(selectedQueries).compact().uniq().value(), vis ];
    });
  };

  /**
   * Delete selected objects with pre-processing that depends on the type of the service
   */
  DeleteHelper.prototype.deleteByType = function (type, items, delcb) {
    if (!delcb) {
      throw new Error('delete method was not passed');
    }
    const self = this;

    const ids = _.map(items, 'id');

    const _delete = function (filteredIds) {
      let promise = delcb(filteredIds);
      if (!Promise.is(promise)) {
        promise = Promise.resolve();
      }
      return promise;
    };

    switch (type) {
      case 'search':
        const deleteBothObjectsAndEntities = function (ids) {
          if (ids.length > 0) {
            _delete(ids)
              .then(() => {
                const promises = _.map(ids, id => dataModel.deleteRelationsByDomainOrRange(id));
                return Promise.all(promises);
              });
          }
        };

        return Promise.all([
          self.dataModelPermissions.checkIndexPatternSearchPermissions(CrudType.DELETE),
          self.dataModelPermissions.checkSearchPermissions(CrudType.DELETE),
          ontologyModel.getEntityList()
        ])
          .then(res => {
            const rootDeletionAllowed = res[0];
            const searchDeletionAllowed = res[1];
            const entities = res[2];
            const filteredIds = [];
            const childrenEntitiesToDelete = [];

            // iterate over ids as this is the main source of truth
            for (const id of ids) {
              const entity = _.find(entities, 'id', id);
              if (!entity) {
              // broken ontology allow to delete orphan search
                filteredIds.push(id);
                continue;
              }

              if (self._isIndexPatternSearch(entity)) {
              // it is a root one do we have permissions
                if (rootDeletionAllowed) {
                  filteredIds.push(id);
                  // examine children
                  for (const e of entities) {
                    if (e.parentId === id) {
                    // entity is a child of one of the entities to delete
                    // check permission
                      if (searchDeletionAllowed) {
                        childrenEntitiesToDelete.push({
                          id: e.id,
                          label: e.label
                        });
                      }
                    }
                  }
                }

              } else {
              // is a child do we have permission
                if (searchDeletionAllowed) {
                  filteredIds.push(id);
                }
              }
            }

            if (rootDeletionAllowed && childrenEntitiesToDelete.length > 0) {
            // show modal, delete after user confirmation
              const childrenLabelsToDelete = _.map(childrenEntitiesToDelete, entity => entity.label);
              const childrenIdsToDelete = _.map(childrenEntitiesToDelete, entity => entity.id);

              const msg =
            `One or more selected searches was an Index Pattern Search.
             Deletion will also delete the following child search${childrenLabelsToDelete.length > 1 ? 'es' : ''}:
             ${JSON.stringify(childrenLabelsToDelete, null, ' ')}.
             Do you want to delete the selected searches and all child searches?`;

              const allIdsToDelete = filteredIds.concat(childrenIdsToDelete);
              return new Promise(function (fulfill, reject) {
                const params = {
                  confirmButtonText: `Delete Search${allIdsToDelete.length > 1 ? 'es' : ''}`,
                  onCancel: fulfill,
                  onConfirm: () => {
                    deleteBothObjectsAndEntities(allIdsToDelete).then(fulfill);
                  }
                };
                confirmModal(msg, params);
              });
            } else {
            // just delete
              return deleteBothObjectsAndEntities(filteredIds);
            }
          });
      case 'dashboardgroup':
        return _delete(ids).then(() => dashboardGroups.computeGroups(`deleted dashboard groups ${JSON.stringify(ids, null, ' ')}`));

      case 'datasource':
        // grab jdbc datasources ids
        const jdbcIds = _(items).filter(item => {
          return item.datasourceType === DatasourceTypes.JDBC;
        })
          .map('id')
          .value();

        const otherIds = _(items).filter(item => {
          return item.datasourceType !== DatasourceTypes.JDBC;
        })
          .map('id')
          .value();

        if (jdbcIds.length > 0) {
          const promises = [];
          _.each(jdbcIds, id => {
            promises.push(jdbcDatasources.delete(id));
          });
          return Promise.all(promises).then(() => {
            return _delete(otherIds);
          });
        }
        return _delete(otherIds);

      case 'dashboard':
        const dashboardGroupNames = dashboardGroups.getIdsOfDashboardGroupsTheseDashboardsBelongTo(ids);
        const defaultDashboard = config.get('siren:defaultDashboardId');
        if (dashboardGroupNames && dashboardGroupNames.length > 0) {
          const plural = dashboardGroupNames.length > 1;
          const msg =
            'Dashboard ' + JSON.stringify(ids, null, ' ') + ' is referred by the following dashboardGroup' +
            (plural ? 's' : '') + ':\n' + dashboardGroupNames.join(', ') + '\n' +
            'Please edit the group' + (plural ? 's' : '') +
            ' and remove the dashboard from its configuration first.';
          return new Promise(function (resolve, reject) {
            const confirmEditGroup = {
              confirmButtonText: 'Go to the dashboard group',
              onConfirm: () => {
                resolve();
                kbnUrl.redirect('/management/siren/objects/savedDashboardGroups/{{id}}', { id: dashboardGroupNames[0] });
              }
            };
            confirmModal(msg, confirmEditGroup);
          });
        }
        //kibi: if default dashboard is removed, remove it also from config
        _.each(ids, function (dashboardId) {
          if (defaultDashboard === dashboardId) {
            config.set('siren:defaultDashboardId', '');
          }
        });
        return _delete(ids).then(() => dashboardGroups.computeGroups(`deleted dashboards ${JSON.stringify(ids, null, ' ')}`));

      case 'query':
        return this._getVisualisations(ids).then(function (visData) {
          if (visData[0].length) {
            const plural = visData[0].length > 1;
            let msg = plural ? 'The queries ' : 'The query ';
            msg += JSON.stringify(visData[0], null, ' ') + (plural ? ' are' : ' is') + ' used in the following';
            msg += (visData[1].length === 1 ? ' visualization' : ' visualizations') + ': \n' +
              JSON.stringify(_.map(visData[1], 'title'), null, ' ') + '\n\nPlease edit or delete' +
              (visData[1].length === 1 ? ' this visualization ' : ' those visualizations ') + 'first.\n\n';
            return new Promise(function (resolve, reject) {
              const confirmEditVis = {
                confirmButtonText: 'Go to visualizations',
                onConfirm: () => {
                  resolve();
                  kbnUrl.redirect('/management/siren/objects/savedVisualizations/{{id}}', { id:  visData[1][0].id });
                }
              };
              confirmModal(msg, confirmEditVis);
            });
          } else {
            delcb(ids);
          }
        });

      default:
        return _delete(ids);
    }
  };

  return new DeleteHelper();
};
