import { QueryBuilderFactory } from 'ui/kibi/helpers/query_builder';
import { parseWithPrecision } from 'ui/kibi/utils/date_math_precision';
import { SirenExportVisualizationHelper } from 'ui/kibi/helpers/siren_export_visualization_helper';
import { fitSizeToContainerWithOptsHelper } from 'ui/kibi/helpers/fit_size_to_container_with_opts_helper';
import _ from 'lodash';
import pdfMake from 'pdfmake/build/pdfmake.js';
import pdfFonts from 'pdfmake/build/vfs_fonts.js';
import htmlToText from 'html-to-text';
import logo from '../../../../../src/ui/public/kibi/images/siren_full_logo.png';

export function SirenExportDashboardHelperProvider(Private, $rootScope, createNotifier, joinExplanation, $injector) {
  return function sirenExportDashboardHelper(panelsToExport, dashOptions) {

    const notify = createNotifier({
      location: 'Export'
    });

    pdfMake.vfs = pdfFonts.pdfMake.vfs;

    const visExportPromises = [];
    const graphEvents = {};
    const exportOptions = {
      method: 'convertToPng',
      title: false
    };

    let countExportedPanels = 0;
    let withGraph = false;

    panelsToExport.forEach(panel => {
      const $element = panel.$element;
      const sirenExportVisualizationHelper = new SirenExportVisualizationHelper(panel.vis, $element);
      if (
        (sirenExportVisualizationHelper.validToExport() || (dashOptions.exportWithTables && !$element.find('div#no-results-found-box')[0]))
        && !$element.find('visualize-spy[is-opened="true"]')[0]
      ) {
        countExportedPanels++;
        if (panel.vis.type.name !== 'graph_browser_vis') {
          visExportPromises.push(sirenExportVisualizationHelper.getVis(exportOptions));
        } else {
          withGraph = true;
          const timestamp = new Date().getTime();
          graphEvents[timestamp] = {};
          graphEvents[timestamp].registeredExportAsPngEvent = new CustomEvent('siren-export-as-png-call', { detail: {
            visId: panel.id,
            isExpanded: panel.expanded
          } });
          document.dispatchEvent(graphEvents[timestamp].registeredExportAsPngEvent);
          graphEvents[timestamp].exportAsSvgEventRespId = 'siren-export-as-png-response';
          graphEvents[timestamp].graphCallback = (event) => {
            const eventDetails = event.detail;

            if (panel.id !== eventDetails.visId) {
              return;
            };
            exportOptions.src = eventDetails.src;
            visExportPromises.push(sirenExportVisualizationHelper.getVis(exportOptions));
          };
          document.removeEventListener(graphEvents[timestamp].exportAsSvgEventRespId, graphEvents[timestamp].graphCallback);
          document.addEventListener(graphEvents[timestamp].exportAsSvgEventRespId, graphEvents[timestamp].graphCallback, false);
        };
      };
    });

    return new Promise((resolve, reject) => {
      let checks = 0;
      const intervalToWaitAllPanels = setInterval(function () {
        if (countExportedPanels === visExportPromises.length) {
          clearInterval(intervalToWaitAllPanels);
          resolve();
        } else if (checks === 10) {
          notify.error('Graph Browser visualization was not included in process due timeout.');
          clearInterval(intervalToWaitAllPanels);
          resolve();
        };
        checks++;
      }, 500);
    }).then(() => {
      const filters = _.cloneDeep(dashOptions.filters);
      filters.push(dashOptions.query);
      const promises = [];
      promises.push(joinExplanation.getFilterExplanations(filters));
      if ($injector.has('kibiAccessControl')) {
        promises.push($injector.get('kibiAccessControl').getUserInfo());
      };
      return Promise.all(promises);
    }).then(([explainedFilters, userInfo]) => {
      const queryString = htmlToText.fromString(explainedFilters.join(' '), {
        wordwrap: 130
      });
      return Promise.all(visExportPromises).then((imgElements) => {
        if (withGraph) {
          _.forEach(graphEvents, function (eventDetails, eventStamp) {
            document.removeEventListener(eventDetails.exportAsSvgEventRespId, eventDetails.graphCallback);
          });
        };

        // A4 72PPI
        const A4_PAGE_WIDTH_PIXELS = 595;
        const A4_PAGE_HEIGHT_PIXELS = 842;
        const SIZE_PROPORTION = A4_PAGE_WIDTH_PIXELS / A4_PAGE_HEIGHT_PIXELS;
        const PAGE_MARGIN = 40;
        const AVAILABLE_A4_PAGE_WIDTH_PIXELS = A4_PAGE_WIDTH_PIXELS - 2 * PAGE_MARGIN;
        const AVAILABLE_A4_PAGE_HEIGHT_PIXELS = A4_PAGE_HEIGHT_PIXELS - 2 * PAGE_MARGIN;
        const ADJUSTABLE_PROPORTION = 0.6;

        const content = [];

        if (isNaN(dashOptions.count)) {
          content.push({
            text: dashOptions.title + '\n\n',
            style: 'header'
          });
        } else {
          content.push({
            text: `${dashOptions.title} (${dashOptions.count})`,
            style: 'header'
          });
        };

        if (userInfo) {
          content.push({
            text: `Produced by ${userInfo.username} at ${new Date().toLocaleString()}\n\n`,
            style: 'small'
          });
        } else {
          content.push({
            text: `Produced at ${new Date().toLocaleString()}\n\n`,
            style: 'small'
          });
        };

        content.push({
          text: 'Dashboard filters:\n',
          style: 'section'
        });

        content.push({
          text: `${queryString}\n\n`,
          style: 'small'
        });

        if (dashOptions.exportWithQuery) {
          const filters = _.cloneDeep(dashOptions.filters);
          const queryBuilder = Private(QueryBuilderFactory);
          if (dashOptions.timefilter && dashOptions.timefilter.fieldName && dashOptions.timefilter.timeRange) {
            const timeFieldFilter = {
              range: {
                [dashOptions.timefilter.fieldName]: {
                  gte: parseWithPrecision(dashOptions.timefilter.timeRange.from, false, $rootScope.sirenTimePrecision).valueOf(),
                  lte: parseWithPrecision(dashOptions.timefilter.timeRange.to, true, $rootScope.sirenTimePrecision).valueOf(),
                  format: 'epoch_millis'
                }
              }
            };
            filters.push(timeFieldFilter);
          };

          content.push({
            text: 'Dashboard filters as query:\n',
            style: 'section'
          });

          content.push({
            text: JSON.stringify(queryBuilder(filters, [dashOptions.query])) + '\n\n',
            style: 'small'
          });
        };

        if (dashOptions.timefilter && dashOptions.timefilter.fieldName && dashOptions.timefilter.timeRange) {
          content.push({
            text: 'Time filter:\n',
            style: 'section'
          });

          content.push({
            text: `from: "${dashOptions.timefilter.timeRange.from}" to: "${dashOptions.timefilter.timeRange.to}"\n\n`,
            style: 'small'
          });
        };

        const docDefinition = {};
        docDefinition.pageSize = 'A4';
        docDefinition.pageMargins = [PAGE_MARGIN, PAGE_MARGIN];
        docDefinition.footer = {
          image: logo,
          width: 90,
          style: 'logo'
        };
        docDefinition.styles = {
          centered: {
            marginBottom: 20,
            alignment: 'center'
          },
          header: {
            fontSize: 24,
            bold: true
          },
          section: {
            fontSize: 10,
            bold: true
          },
          small: {
            fontSize: 8
          },
          logo: {
            alignment: 'right',
            marginLeft: PAGE_MARGIN
          }
        };

        content.push({
          text: 'Dashboard panels:\n\n',
          style: 'section'
        });

        imgElements.forEach(imgElement => {
          content.push({ ul: [imgElement.title], style: 'small' });

          const imgObject = {};

          const fittedImgSize = fitSizeToContainerWithOptsHelper({
            elementWidth: imgElement.width,
            elementHeight: imgElement.height,
            containerWidth: AVAILABLE_A4_PAGE_WIDTH_PIXELS,
            containerHeight: AVAILABLE_A4_PAGE_HEIGHT_PIXELS,
            adjustableProportion: ADJUSTABLE_PROPORTION,
            sizeProportion: SIZE_PROPORTION
          });

          imgObject.width = fittedImgSize.width;
          imgObject.height = fittedImgSize.height;
          imgObject.image = imgElement.imgSrc;
          imgObject.style = 'centered';
          content.push(imgObject);

        });

        docDefinition.content = content;
        if (dashOptions.toDownload) {
          pdfMake.createPdf(docDefinition).download(dashOptions.title + '.pdf');
        } else {
          pdfMake.createPdf(docDefinition).open();
        };
      });
    }).catch((e) => {
      notify.error(e);
    });
  };
}
