'use strict';

var _utils = require('../utils');

var _lodash = require('lodash');

var _path = require('path');

const arr = v => [].concat(v || []);

module.exports = class UiBundlerEnv {
  constructor(workingDir) {

    // the location that bundle entry files and all compiles files will
    // be written
    this.workingDir = workingDir;

    // the context that the bundler is running in, this is not officially
    // used for anything but it is serialized into the entry file to ensure
    // that they are invalidated when the context changes
    this.context = {};

    // the plugins that are used to build this environment
    // are tracked and embedded into the entry file so that when the
    // environment changes we can rebuild the bundles
    this.pluginInfo = [];

    // regular expressions which will prevent webpack from parsing the file
    this.noParse = [/node_modules[\/\\](angular|elasticsearch-browser)[\/\\]/, /node_modules[\/\\](mocha|moment)[\/\\]/];

    // webpack aliases, like require paths, mapping a prefix to a directory
    this.aliases = {
      ui: (0, _utils.fromRoot)('src/ui/public'),
      ui_framework: (0, _utils.fromRoot)('ui_framework'),
      test_harness: (0, _utils.fromRoot)('src/test_harness/public'),
      querystring: 'querystring-browser',
      // kibi: added test_utils alias to simplify imports
      test_utils: (0, _utils.fromRoot)('src/test_utils'),
      // kibi: added alias to test folder to simplify imports
      test_kibana: (0, _utils.fromRoot)('test'),
      // kibi: added alias to utils folder to simplify imports
      utils_kibana: (0, _utils.fromRoot)('src/utils')
    };

    // map of which plugins created which aliases
    this.aliasOwners = {};

    // loaders that are applied to webpack modules after all other processing
    // NOTE: this is intentionally not exposed as a uiExport because it leaks
    // too much of the webpack implementation to plugins, but is used by test_bundle
    // core plugin to inject the instrumentation loader
    this.postLoaders = [];
  }

  consumePlugin(plugin) {
    const tag = `${plugin.id}@${plugin.version}`;
    if ((0, _lodash.includes)(this.pluginInfo, tag)) return;

    if (plugin.publicDir) {
      this.aliases[`plugins/${plugin.id}`] = plugin.publicDir;
    }

    this.pluginInfo.push(tag);
  }

  exportConsumer(type) {
    switch (type) {
      case 'noParse':
        return (plugin, spec) => {
          for (const rule of arr(spec)) {
            this.noParse.push(this.getNoParseRegexp(rule));
          }
        };

      case '__globalImportAliases__':
        return (plugin, spec) => {
          for (const key of Object.keys(spec)) {
            this.aliases[key] = spec[key];
          }
        };
    }
  }

  addContext(key, val) {
    this.context[key] = val;
  }

  addPostLoader(loader) {
    this.postLoaders.push(loader);
  }

  getNoParseRegexp(rule) {
    if (typeof rule === 'string') {
      return new RegExp(`${(0, _path.isAbsolute)(rule) ? '^' : ''}${(0, _lodash.escapeRegExp)(rule)}`);
    }

    if (rule instanceof RegExp) {
      return rule;
    }

    throw new Error('Expected noParse rule to be a string or regexp');
  }
};
