import { cloneDeep } from 'lodash';
import ngMock from 'ng_mock';
import expect from 'expect.js';
import { getSearchParameters } from 'plugins/scatterplot_vis/helpers/get_search_parameters_helper';
import { stubbedLogstashIndexPatternService } from 'fixtures/stubbed_logstash_index_pattern';

describe('getSearchParameters', () => {
  let $scope;

  beforeEach(ngMock.module('kibana'));

  const rangeFilter = {
    range: {
      fieldA: {

      }
    }
  }

  const timeFilter = {
    get: () => {
      return rangeFilter
    }
  };

  function init(options) {
    ngMock.inject(function ($rootScope, Private) {
      $scope = $rootScope.$new();
      $scope.vis = {
        indexPattern: Private(stubbedLogstashIndexPatternService),
        params: options.params
      };
      $scope.options = options.params;
      Object.defineProperty($scope, 'appState', {
        get: () => {
          return {};
        },
        set: () => {}
      });
    });
  };

  it('body should include filters from both appState even if they are for different index-pattern (pinned filter)', () => {
    init({ params: {} });
    const meta = {
      disabled: false,
      index: 'index-pattern:different-than-stubbed-one' // id DIFFERENT from stubbed index-pattern which is logstash-*
    };

    const filterFromAppState = {
      query: {
        exists: {
          field: 'fieldA1'
        }
      },
      meta
    };

    const expectedMust = cloneDeep([
      filterFromAppState,
      rangeFilter,
    ]).map(f => {
      if (f.query && f.meta) {
        return f.query;
      }
      delete f.meta;
      return f;
    });

    const searchParametersBody = getSearchParameters(
      $scope, timeFilter,
      [ filterFromAppState ], null,
      [ ], null
    ).body;

    const actualMust = searchParametersBody.query.function_score.query.bool.must;
    expect(actualMust.length).to.equal(expectedMust.length);
    expect(actualMust).to.eql(expectedMust);
  });

  it('body should include filters from both appState and savedSearch meta', () => {
    init({ params: {} });
    const meta = {
      disabled: false,
      index: 'index-pattern:logstash-*' // id of the stubbed index-pattern
    };

    const filterFromAppState = {
      query: {
        exists: {
          field: 'fieldA1'
        }
      },
      meta
    };
    const queryFromAppState = {
      query: {
        exists: {
          field: 'fieldA2'
        }
      }
    };

    const filterFromSavedSearchState = {
      query: {
        exists: {
          field: 'fieldS1'
        }
      },
      meta
    };
    const queryFromSavedSearchState = {
      query: {
        exists: {
          field: 'fieldS2'
        }
      }
    };

    const expectedMust = cloneDeep([
      filterFromAppState,
      filterFromSavedSearchState,
      rangeFilter,
      queryFromAppState,
      queryFromSavedSearchState
    ]).map(f => {
      if (f.query && f.meta) {
        return f.query;
      }
      delete f.meta;
      return f;
    });

    const searchParametersBody = getSearchParameters(
      $scope, timeFilter,
      [ filterFromAppState ], queryFromAppState,
      [ filterFromSavedSearchState ], queryFromSavedSearchState
    ).body;

    const actualMust = searchParametersBody.query.function_score.query.bool.must;
    expect(actualMust.length).to.equal(expectedMust.length);
    expect(actualMust).to.eql(expectedMust);
  });

  it('should correctly determine the fields with dots', () => {
    const testMap = [
      {
        fieldA: 'fieldName.raw',
        fieldB: 'fieldName',
        _sourceExpectedLength: 1,
        docvalue_fieldsExpectedLength: 1,
      },
      {
        fieldA: 'fieldName.dot',
        fieldB: 'fieldName',
        _sourceExpectedLength: 1,
        docvalue_fieldsExpectedLength: 1
      },
      {
        fieldA: 'fieldName.dot',
        fieldB: 'fieldName.raw',
        _sourceExpectedLength: 0,
        docvalue_fieldsExpectedLength: 2
      },
      {
        fieldA: 'fieldName',
        fieldB: 'fieldName',
        _sourceExpectedLength: 1,
        docvalue_fieldsExpectedLength: 0
      },
      {
        fieldA: 'fieldName_1',
        fieldB: 'fieldName_2',
        _sourceExpectedLength: 2,
        docvalue_fieldsExpectedLength: 0
      },
      {
        fieldA: 'fieldName.dot',
        fieldB: 'fieldName.dot',
        _sourceExpectedLength: 0,
        docvalue_fieldsExpectedLength: 1
      },
    ];

    testMap.forEach(params => {
      init({
        params: {
          xFieldName: params.fieldA,
          yFieldName: params.fieldB,
          aggMode: 'Straight data'
        }
      });

      const searchParametersBody = getSearchParameters($scope, timeFilter, [], null).body;

      expect(searchParametersBody._source.length).to.be(params._sourceExpectedLength);
      expect(searchParametersBody.docvalue_fields.length).to.be(params.docvalue_fieldsExpectedLength);
      if (params.fieldA.split('.').length > 1) {
        expect(searchParametersBody.docvalue_fields.indexOf(params.fieldA) !== -1).to.be(true);
        expect(searchParametersBody._source.indexOf(params.fieldA)).to.be(-1);
      } else {
        expect(searchParametersBody.docvalue_fields.indexOf(params.fieldA)).to.be(-1);
        expect(searchParametersBody._source.indexOf(params.fieldA) !== -1).to.be(true);
      };
      if (params.fieldB.split('.').length > 1) {
        expect(searchParametersBody.docvalue_fields.indexOf(params.fieldB) !== -1).to.be(true);
        expect(searchParametersBody._source.indexOf(params.fieldB)).to.be(-1);
      } else {
        expect(searchParametersBody.docvalue_fields.indexOf(params.fieldB)).to.be(-1);
        expect(searchParametersBody._source.indexOf(params.fieldB) !== -1).to.be(true);
      };
    });
  });
});
