'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

exports.default = function (program) {

  /**
   * Waits for the kbnServer status to be green.
   *
   * @param {KbnServer} kbnServer A KbnServer instance.
   * @param {Number} retries The number of retries.
   */
  async function waitForGreenStatus(kbnServer, retries) {
    if (retries === 0) {
      throw new Error('Timed out while waiting for the server status to be ' + 'green, please check the logs and try again.');
    }
    if (!kbnServer.status.isGreen()) {
      await _bluebird2.default.delay(2500);
      await waitForGreenStatus(kbnServer, --retries);
    }
  }

  async function replaceEncryptionKey(currentKey, newKey, newCipher, options) {

    const config = (0, _read_yaml_config2.default)(options.config);
    const logger = new _logger2.default(options);

    (0, _lodash.merge)(config, {
      env: 'production',
      logging: {
        silent: options.silent,
        quiet: options.quiet,
        verbose: false
      },
      optimize: {
        enabled: false
      },
      server: {
        autoListen: false
      },
      plugins: {
        initialize: true,
        scanDirs: options.pluginDir
      }
    });

    const kbnServer = new _kbn_server2.default(config);

    await kbnServer.ready();

    try {
      await waitForGreenStatus(kbnServer, 10);
      try {
        const indexHelper = kbnServer.server.plugins.investigate_core.getIndexHelper();
        await indexHelper.rencryptAllValuesInKibiIndex(currentKey, newCipher, newKey, options.config);
        logger.log('Encryption key replaced successfully.');
      } catch (error) {
        logger.error(error.message ? error.message : error);
        process.exit(1);
      }
    } catch (error) {
      logger.error('An error occurred while waiting for the server to be ready, please check the logs.');
      logger.error(error.message ? error.message : error);
      process.exit(1);
    }

    await kbnServer.close();
    process.exit(0);
  }

  async function processCommand(currentKey, newKey, newCipher, options) {
    await replaceEncryptionKey(currentKey, newKey, newCipher, options);
  }

  program.command('replace_encryption_key <current_key> <new_key> <new_cipher>').description('Re-encrypts all the datasources and updates the configuration file. ' + 'The original configuration file is saved with the .bak extension.').option('-q, --quiet', 'Disable all process messaging except errors').option('-s, --silent', 'Disable all process messaging').option('-c, --config <path>', 'Path to the config file, can be changed with the CONFIG_PATH environment variable as well', process.env.CONFIG_PATH || (0, _from_root.fromRoot)('config/investigate.yml')).option('--plugin-dir <path>', 'A path to scan for plugins, this can be specified multiple ' + 'times to specify multiple directories', pluginDirCollector, [(0, _from_root.fromRoot)('plugins'), // installed plugins
  (0, _from_root.fromRoot)('src/siren_core_plugins'), // kibi plugins
  (0, _from_root.fromRoot)('siren_plugins'), // installed siren external plugins
  (0, _from_root.fromRoot)('src/core_plugins') // kibana plugins
  ]).action(processCommand);
};

var _kbn_server = require('../../server/kbn_server');

var _kbn_server2 = _interopRequireDefault(_kbn_server);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _lodash = require('lodash');

var _logger = require('./logger');

var _logger2 = _interopRequireDefault(_logger);

var _read_yaml_config = require('../serve/read_yaml_config');

var _read_yaml_config2 = _interopRequireDefault(_read_yaml_config);

var _path = require('path');

var _from_root = require('../../utils/from_root');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const pathCollector = function () {
  const paths = [];
  return function (path) {
    paths.push((0, _path.resolve)(process.cwd(), path));
    return paths;
  };
};

const pluginDirCollector = pathCollector();

/**
 * The command to replace the datasource encryption key.
 */
;
module.exports = exports['default'];
