'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = restoreCommand;

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _lodash = require('lodash');

var _read_yaml_config = require('../serve/read_yaml_config');

var _read_yaml_config2 = _interopRequireDefault(_read_yaml_config);

var _validate_config = require('../../cli/kibi/validate_config');

var _from_root = require('../../utils/from_root');

var _path = require('path');

var _restore_kibi = require('./_restore_kibi');

var _restore_kibi2 = _interopRequireDefault(_restore_kibi);

var _syswideCas = require('syswide-cas');

var _syswideCas2 = _interopRequireDefault(_syswideCas);

var _readlinePromise = require('readline-promise');

var _readlinePromise2 = _interopRequireDefault(_readlinePromise);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const pathCollector = function () {
  const paths = [];
  return function (path) {
    paths.push((0, _path.resolve)(process.cwd(), path));
    return paths;
  };
};

const pluginDirCollector = pathCollector();

/**
 * The command to restore a kibi instance
 */
function restoreCommand(program) {

  async function restore(options) {
    const config = (0, _read_yaml_config2.default)(options.config);
    const configFiles = [options.config];
    if (options.dev && (0, _validate_config.checkConfigYmlExists)('investigate', true)) {
      configFiles.push((0, _validate_config.getConfigYmlPath)('investigate', true));
    }

    if (options.dev) {
      try {
        (0, _lodash.merge)(config, (0, _read_yaml_config2.default)((0, _from_root.fromRoot)('config/investigate.dev.yml')));
      } catch (e) {
        // ignore
      }
    }

    const acceptAll = options.Y || false;

    (0, _lodash.merge)(config, {
      env: 'production',
      logging: {
        silent: false,
        quiet: false,
        verbose: false,
        dest: 'stdout'
      },
      optimize: {
        enabled: false
      },
      server: {
        autoListen: false
      },
      plugins: {
        initialize: true,
        scanDirs: options.pluginDir
      },
      configFiles: configFiles
    });

    if ((0, _lodash.has)(config, 'elasticsearch.ssl.ca')) {
      _syswideCas2.default.addCAs((0, _lodash.get)(config, 'elasticsearch.ssl.ca'));
    }

    let exitCode = 0;
    let deleteIndicesBeforeRestoring = false;

    const restoreKibi = new _restore_kibi2.default(config, options.backupDir);

    try {
      // Check one of the indices exists
      const indexExist = await restoreKibi.getIndicesInfo();

      if (acceptAll) {
        deleteIndicesBeforeRestoring = true;
      } else {
        if (indexExist.kibi || indexExist.acl) {
          const rlp = _readlinePromise2.default.createInterface(process.stdin, process.stdout);
          let resp;
          if (indexExist.indexNames.length === 1) {
            resp = await rlp.questionAsync('Delete ' + indexExist.indexNames[0] + ' index before restoring? [N/y]');
          } else {
            resp = await rlp.questionAsync('Delete [' + indexExist.indexNames.join(', ') + '] indexes before restoring? [N/y]');
          }

          const yes = resp.toLowerCase().trim()[0] === 'y';
          process.stderr.write('\n');
          if (yes) {
            deleteIndicesBeforeRestoring = true;
          }
        }
      }

      await restoreKibi.restore(deleteIndicesBeforeRestoring, indexExist);
      process.stdout.write('Restore successfully completed.\n');
    } catch (error) {
      process.stderr.write(`${error}\n`);
      exitCode = 1;
    }
    process.exit(exitCode);
  }

  async function processCommand(options) {
    await restore(options);
  }

  return program.command('restore').description('Restore an Investigate instance').option('--dev', 'Run the restore command using development mode configuration').option('-c, --config <path>', 'Path to the config file, can be changed with the CONFIG_PATH environment variable as well', process.env.CONFIG_PATH || (0, _from_root.fromRoot)('config/investigate.yml')).option('-y', 'Accept all options e.g. delete the index before restoring question').option('--backup-dir <path>', 'Path to the directory where the Investigate instance data was saved').option('--plugin-dir <path>', 'A path to scan for plugins, this can be specified multiple times to specify multiple directories', pluginDirCollector, [(0, _from_root.fromRoot)('src/core_plugins'), (0, _from_root.fromRoot)('src/siren_core_plugins'), (0, _from_root.fromRoot)('siren_plugins'), (0, _from_root.fromRoot)('plugins')]).action(processCommand);
};
module.exports = exports['default'];
