import _ from 'lodash';
import 'ui/visualize';
import 'ui/doc_table';
import * as columnActions from 'ui/doc_table/actions/columns';
import 'plugins/kibana/dashboard/panel/get_object_loaders_for_dashboard';
import { FilterManagerProvider } from 'ui/filter_manager';
import { uiModules } from 'ui/modules';
import panelTemplate from 'plugins/kibana/dashboard/panel/panel.html';
import { DoesVisDependsOnSelectedEntitiesProvider } from 'ui/kibi/components/commons/_does_vis_depends_on_selected_entities';
import { savedObjectManagementRegistry } from 'plugins/kibana/management/saved_object_registry';
import { getPersistedStateId } from 'plugins/kibana/dashboard/panel/panel_state';
import { loadSavedObject } from 'plugins/kibana/dashboard/panel/load_saved_object';
import { DashboardViewMode } from '../dashboard_view_mode';

// kibi: imports
import 'ui/kibi/components/dashboards360/coat_panel_register';
import { HashedItemStoreSingleton } from 'ui/state_management/state_storage';
import { SirenExportVisualizationHelper } from 'ui/kibi/helpers/siren_export_visualization_helper';
import { findItemByVisIdAndPanelIndex } from 'ui/kibi/components/dashboards360/lib/coat/find_item_by_vis_id_and_panel_index';
// kibi: end

uiModules
  .get('app/dashboard')
  .directive('dashboardPanel', function (savedVisualizations, savedSearches, Private,
    $injector, getObjectLoadersForDashboard, kibiState, $timeout, createNotifier, coatPanelRegister) {

    const filterManager = Private(FilterManagerProvider);
    const doesVisDependsOnSelectedEntities = Private(DoesVisDependsOnSelectedEntitiesProvider);

    const services = savedObjectManagementRegistry.all().map(function (serviceObj) {
      const service = $injector.get(serviceObj.service);
      return {
        type: service.type,
        name: serviceObj.service
      };
    });

    return {
      restrict: 'E',
      template: panelTemplate,
      scope: {
      // kibi: 'hideBorders' is added

      /**
       * toggle borders around panels
       * @author kibi
       */
        hideBorders: '=',
        /**
       * What view mode the dashboard is currently in - edit or view only.
       * @type {DashboardViewMode}
       */
        dashboardViewMode: '=',
        /**
       * Whether or not the dashboard this panel is contained on is in 'full screen mode'.
       * @type {boolean}
       */
        isFullScreenMode: '=',
        /**
       * Used to create a child persisted state for the panel from parent state.
       * @type {function} - Returns a {PersistedState} child uiState for this scope.
       */
        createChildUiState: '=',
        /**
       * Registers an index pattern with the dashboard app used by this panel. Used by the filter bar for
       * generating field suggestions.
       * @type {function(IndexPattern)}
       */
        registerPanelIndexPattern: '=',
        /**
       * Contains information about this panel.
       * @type {PanelState}
       */
        panel: '=',
        /**
       * Handles removing this panel from the grid.
       * @type {function}
       */
        remove: '&',
        /**
       * Expand or collapse the current panel, so it either takes up the whole screen or goes back to its
       * natural size.
       * @type {function}
       */
        toggleExpand: '&',
        /**
       * @type {boolean}
       */
        isExpanded: '=',
        /**
       * Returns a click handler for a visualization.
       * @type {function}
       */
        getVisClickHandler: '=',
        /**
       * Returns a brush event handler for a visualization.
       * @type {function}
       */
        getVisBrushHandler: '=',
        /**
       * Call when changes should be propagated to the url and thus saved in state.
       * @type {function}
       */
        saveState: '=',

        sirenData: '=?'
      },
      link: function ($scope, element) {
        if (!$scope.panel.id || !$scope.panel.type) return;

        const notify = createNotifier({
          location: 'Dashboard - Panel'
        });

        // kibi: allows restore the uiState after click edit visualization on dashboard
        $scope.edit = function () {
          if ($scope.panel.type === savedVisualizations.type && $scope.savedObj.vis) {
            const kibiPanelId = {
              visId: $scope.savedObj.vis.id,
              panelId: getPersistedStateId($scope.panel),
              updated: false
            };
            HashedItemStoreSingleton.setItem('kibi_panel_id', JSON.stringify(kibiPanelId));
            HashedItemStoreSingleton.setItem('kibi_ui_state', $scope.savedObj.vis.getUiState().toString());
          }
          window.location.href = $scope.editUrl;
        };
        // kibi: end


        function getSavedSearchId(savedObj) {
        // TODO check by type SavedSearch or SavedVis
          const isVis = savedObj.id.indexOf('visualization:') === 0;
          const isSearch = savedObj.id.indexOf('search:') === 0;
          let savedSearchId;
          if (isVis && savedObj.vis) {
            savedSearchId = savedObj.savedSearchId;
          } else if (isSearch) {
            savedSearchId = savedObj.id;
          }
          return savedSearchId;
        }

        // we should assign _siren even to vis which do no have single savedSearchId
        // - one that are using multiple search sources like relational navigator or timeline
        function addSirenPropertyToVisOrSearch(savedObj) {
          if (!savedObj || !$scope.sirenData || !$scope.sirenData.coat) {
            return;
          }
          const savedSearchId = getSavedSearchId(savedObj);
          const isVis = savedObj.id.indexOf('visualization:') === 0;

          const node = findItemByVisIdAndPanelIndex(
            $scope.sirenData.coat.items,
            isVis ? savedObj.id : savedSearchId,
            $scope.panel.panelIndex
          );

          const _siren = {
            vis: {
              id: savedObj.id,
              panelIndex: $scope.panel.panelIndex,
              title: savedObj.title
            },
            coat: {
              items: $scope.sirenData.coat.items,
              joinAlgorithmType: $scope.sirenData.coat.joinAlgorithmType,
              datamodelType: $scope.sirenData.coat.datamodelType,
              node
            }
          };

          savedObj.searchSource._siren = _siren;
          if (isVis) {
            savedObj.vis._siren = _siren;
          }
        };

        const coatPanelRegisterFunction = function () {
          return addSirenPropertyToVisOrSearch($scope.savedObj);
        };

        coatPanelRegister.add(coatPanelRegisterFunction);
        /**
       * Initializes the panel for the saved object.
       * @param {{savedObj: SavedObject, editUrl: String}} savedObjectInfo
       */
        function initializePanel(savedObjectInfo) {
          $scope.savedObj = savedObjectInfo.savedObj;
          $scope.editUrl = savedObjectInfo.editUrl;

          element.on('$destroy', function () {
            coatPanelRegister.remove(coatPanelRegisterFunction);
            $scope.savedObj.destroy();
            $scope.$destroy();
          });

          // kibi: For some unknown reason the vis object doesn't have his own id. This must be investigated in the future.
          // See issue https://github.com/sirensolutions/kibi-internal/issues/2909
          if ($scope.savedObj.vis) {
            $scope.savedObj.vis.id = $scope.panel.id;
            $scope.savedObj.vis.isPanelExpanded = () => $scope.isExpanded;
          }
          // kibi: end

          // kibi: added handle the entity selection events
          $scope.dependsOnSelectedEntities = false;
          if ($scope.panel.type === savedVisualizations.type && $scope.savedObj.vis) {
            // there could be no vis object if we visualise saved search
            doesVisDependsOnSelectedEntities($scope.savedObj.vis).then(function (does) {
              $scope.dependsOnSelectedEntities = does;
            });
          }

          $scope.markDependOnSelectedEntities = Boolean(kibiState.getEntityURI());
          $scope.selectedEntitiesDisabled = kibiState.isSelectedEntityDisabled();

          // react to changes about the selected entity
          $scope.$listen(kibiState, 'save_with_changes', (diff) => {
            if (diff.indexOf(kibiState._properties.selected_entity) !== -1 ||
            diff.indexOf(kibiState._properties.selected_entity_disabled) !== -1) {
              $scope.markDependOnSelectedEntities = Boolean(kibiState.getEntityURI());
              $scope.selectedEntitiesDisabled = kibiState.isSelectedEntityDisabled();
            }
          });
          // kibi: end

          // create child ui state from the savedObj
          const uiState = $scope.savedObj.uiStateJSON ? JSON.parse($scope.savedObj.uiStateJSON) : {};
          $scope.uiState = $scope.createChildUiState(getPersistedStateId($scope.panel), uiState);

          if ($scope.panel.type === savedVisualizations.type && $scope.savedObj.vis) {
            // kibi: allows restore the uiState after click edit visualization on dashboard
            const __panelid = HashedItemStoreSingleton.getItem('kibi_panel_id');
            if (__panelid) {
              const { visId, panelId, updated } = JSON.parse(__panelid);
              if (visId === $scope.panel.id && panelId === getPersistedStateId($scope.panel) && updated) {
                $scope.uiState.fromString(HashedItemStoreSingleton.getItem('kibi_ui_state'));
                HashedItemStoreSingleton.removeItem('kibi_panel_id');
                HashedItemStoreSingleton.removeItem('kibi_ui_state');
              }
            }
            // kibi: end

            $scope.savedObj.vis.setUiState($scope.uiState);
            $scope.savedObj.vis.listeners.click = $scope.getVisClickHandler();
            $scope.savedObj.vis.listeners.brush = $scope.getVisBrushHandler();
            // kibi: we are getting registerPanelIndexPattern function from dashboard.js
            // if dashboard has saved search we are not assigning this function
            // we need to check if this function was set
            if ($scope.registerPanelIndexPattern) {
              $scope.registerPanelIndexPattern($scope.panel.panelIndex, $scope.savedObj.vis.indexPattern);
            }
          } else if ($scope.panel.type === savedSearches.type) {
            if ($scope.savedObj.searchSource && $scope.registerPanelIndexPattern) {
              $scope.registerPanelIndexPattern($scope.panel.panelIndex, $scope.savedObj.searchSource.get('index'));
            }
            // This causes changes to a saved search to be hidden, but also allows
            // the user to locally modify and save changes to a saved search only in a dashboard.
            // See https://github.com/elastic/kibana/issues/9523 for more details.
            $scope.panel.columns = $scope.panel.columns || $scope.savedObj.columns;
            $scope.panel.sort = $scope.panel.sort || $scope.savedObj.sort;

            $scope.setSortOrder = function setSortOrder(columnName, direction) {
              $scope.panel.sort = [columnName, direction];
              $scope.saveState();
            };

            $scope.addColumn = function addColumn(columnName) {
              $scope.savedObj.searchSource.get('index').popularizeField(columnName, 1);
              columnActions.addColumn($scope.panel.columns, columnName);
              $scope.saveState();  // sync to sharing url
            };

            $scope.removeColumn = function removeColumn(columnName) {
              $scope.savedObj.searchSource.get('index').popularizeField(columnName, 1);
              columnActions.removeColumn($scope.panel.columns, columnName);
              $scope.saveState();  // sync to sharing url
            };

            $scope.moveColumn = function moveColumn(columnName, newIndex) {
              columnActions.moveColumn($scope.panel.columns, columnName, newIndex);
              $scope.saveState();  // sync to sharing url
            };
          }
        }

        $scope.filter = function (field, value, operator, _siren) {
          const index = $scope.savedObj.searchSource.get('index').id;
          filterManager.add(field, value, operator, index, { _siren });
        };


        $scope.$on('edit:vis', function (event, id, params) {
          if (id === $scope.panel.id) {
            $scope.savedObj.vis.params = params;
          }
        });

        // kibi: toggle borders around panels
        // Don't do this inside initialization to consistently show/hide borders
        // when there are errors during the panel initialization
        $scope.border = !$scope.hideBorders;
        $scope.$watch('hideBorders', hideBorders => {
          if (hideBorders !== undefined) {
            $scope.border = !$scope.hideBorders;
          }
        });
        // kibi: end


        $scope.loadedPanel = loadSavedObject(getObjectLoadersForDashboard(), $scope.panel)
          .then(savedObjectInfo => {
            if (savedObjectInfo.savedObj) {
              addSirenPropertyToVisOrSearch(savedObjectInfo.savedObj);
            }
            return savedObjectInfo;
          })
          .then(savedObjectInfo => initializePanel(savedObjectInfo))
          .catch(function (e) {
            $scope.error = e.message;

            // Dashboard listens for this broadcast, once for every visualization (pendingVisCount).
            // We need to broadcast even in the event of an error or it'll never fetch the data for
            // other visualizations.
            $scope.$root.$broadcast('ready:vis');

            // If the savedObjectType matches the panel type, this means the object itself has been deleted,
            // so we shouldn't even have an edit link. If they don't match, it means something else is wrong
            // with the object (but the object still exists), so we link to the object editor instead.
            const objectItselfDeleted = e.savedObjectType === $scope.panel.type;
            if (objectItselfDeleted) return;

            const type = $scope.panel.type;
            const id = $scope.panel.id;
            const service = _.find(services, { type: type });
            if (!service) return;

            // kibi: changed url to siren
            $scope.editUrl = '#management/siren/objects/' + service.name + '/' + id + '?notFound=' + e.savedObjectType;
          });

        /**
       * @returns {boolean} True if the user can only view, not edit.
       */
        $scope.isViewOnlyMode = () => {
          return $scope.dashboardViewMode === DashboardViewMode.VIEW || $scope.isFullScreenMode;
        };

        const exportIsCurrentKeyWatcher = $scope.$parent.$parent.$watch('exportIsCurrentKey', key => $scope.exportIsCurrentKey = key);

        let exportVisualizationLogic;
        $scope.$watch('savedObj', savedObj => {
          if (savedObj) {
            exportVisualizationLogic = new SirenExportVisualizationHelper($scope.savedObj.vis, element);
          };
        });

        $scope.exportInProcess = false;

        $scope.executeExport = () => {
          $scope.exportInProcess = true;
          $timeout(() => {
            exportVisualizationLogic.getVis()
              .then(() => {
                $scope.exportInProcess = false;
              }).catch((e) => {
                notify.error(e);
                $scope.exportInProcess = false;
              });
          });
        };

        $scope.exportVisualization = () => {
          if ($scope.savedObj.vis.type.name === 'graph_browser_vis') {
            $scope.exportInProcess = true;
            const exportAsSvgEventCallId = 'siren-export-as-png-call';
            const registeredExportAsPngEvent = new CustomEvent(exportAsSvgEventCallId, { detail: {
              visId: $scope.savedObj.vis.id,
              isExpanded: $scope.isExpanded
            } });
            document.dispatchEvent(registeredExportAsPngEvent);
            const exportAsSvgEventRespId = 'siren-export-as-png-response';
            const graphCallback = (e) => {
              const eventDetails = e.detail;
              if ($scope.isExpanded !== eventDetails.isExpanded && $scope.savedObj.vis.id !== eventDetails.visId) {
                return;
              };
              exportVisualizationLogic.getVis({ src: eventDetails.src })
                .then(() => {
                  $scope.exportInProcess = false;
                  document.removeEventListener(exportAsSvgEventRespId, graphCallback);
                }).catch((e) => {
                  notify.error(e);
                  $scope.exportInProcess = false;
                });
            };
            document.removeEventListener(exportAsSvgEventRespId, graphCallback);
            document.addEventListener(exportAsSvgEventRespId, graphCallback, false);
            return;
          };
          $scope.executeExport();
        };

        $scope.validToExport = _.throttle(() => {
          if ($scope.savedObj &&
           $scope.savedObj.vis &&
           $scope.savedObj.vis.type &&
           $scope.savedObj.vis.type.name === 'multi_chart_vis') {
            $scope.tableOnMultichart = !!element.find('kbn-agg-table-group')[0];
            return !$scope.tableOnMultichart && exportVisualizationLogic.validToExport();
          } else if ($scope.savedObj) {
            return exportVisualizationLogic.validToExport();
          } else {
            return false;
          };
        }, 500);

        $scope.$watch(() => {
          const visSpy = element.find('visualize-spy');
          return visSpy[0] && visSpy.attr('is-opened');
        }, (visSpy) => {
          $scope.visSpyIsOpened = visSpy ? true : false;
        });

        $scope.$on('$destroy', () => {
          exportIsCurrentKeyWatcher();
        });
      }
    };
  });
