import _ from 'lodash';
import angular from 'angular';
import moment from 'moment';
import { getSort } from 'ui/doc_table/lib/get_sort';
import * as columnActions from 'ui/doc_table/actions/columns';
import 'ui/doc_table';
import 'ui/visualize';
import 'ui/notify';
import 'ui/fixed_scroll';
import 'ui/directives/validate_json';
import 'ui/filters/moment';
import 'ui/courier';
import 'ui/index_patterns';
import 'ui/state_management/app_state';
import 'ui/timefilter';
import 'ui/share';
import { VisProvider } from 'ui/vis';
import { DocTitleProvider } from 'ui/doc_title';
import { UtilsBrushEventProvider } from 'ui/utils/brush_event';
import PluginsKibanaDiscoverHitSortFnProvider from 'plugins/kibana/discover/_hit_sort_fn';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { FilterManagerProvider } from 'ui/filter_manager';
import { AggTypesBucketsIntervalOptionsProvider } from 'ui/agg_types/buckets/_interval_options';
import { stateMonitorFactory } from 'ui/state_management/state_monitor_factory';
import uiRoutes from 'ui/routes';
import { uiModules } from 'ui/modules';
import indexTemplate from 'plugins/kibana/discover/index.html';
import { StateProvider } from 'ui/state_management/state';
import { documentationLinks } from 'ui/documentation_links/documentation_links';
import { SavedObjectsClientProvider } from 'ui/saved_objects';
import { getDefaultQuery } from 'ui/parse_query';

// kibi: imports
import { parseWithPrecision } from 'ui/kibi/utils/date_math_precision';
import { IndexPatternAuthorizationError, SavedObjectAuthorizationError, IndexPatternMissingIndices } from 'ui/errors';
import { QuickDashboardProvider } from 'ui/kibi/quick_dashboard/quick_dashboard';
import { GuessFieldsProvider } from 'ui/kibi/quick_dashboard/guess_fields';
import { SearchSourceProvider } from 'ui/courier/data_source/search_source';
import { SegmentedRequestProvider } from 'ui/courier/fetch/request/segmented.js';
import 'plugins/investigate_core/ui/directives/siren_entity_nav/siren_entity_nav';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { EntityValidatorProvider } from 'plugins/investigate_core/management/sections/data_model/controllers/entity_validator';
// kibi: end

const app = uiModules.get('apps/discover', [
  'kibana/notify',
  'kibana/courier',
  'kibana/index_patterns'
]);

uiRoutes
  .when('/discover/', {
    resolve: {
      // kibi:
      // This resolver is here just to redirect user to correct search or Data model page
      x: function (ontologyModel, kbnUrl, createNotifier, Private, Promise) {
      // first available Index Pattern Search entity
        const entityValidator = Private(EntityValidatorProvider);
        return entityValidator.findFirstValidEntity(EntityType.SAVED_SEARCH)
          .then(res => {
            if (res.entity && res.entity.type !== 'VIRTUAL_ENTITY') {
              kbnUrl.redirect('/discover/' + res.entity.id);
              return Promise.halt();
            }

            if (res.searches.total === 0) {
              createNotifier({ location: 'Discover - resolve' }).warning(`No Search found, please create first one on Data Model page`);
              kbnUrl.redirect('/management/siren/datamodel/');
              return Promise.halt();
            }
          });
      }
    // kibi: end
    }
  })
  .when('/discover/:id?', {
    template: indexTemplate,
    reloadOnSearch: false,
    resolve: {
    // kibi: added createNotifier, kbnUrl, kibiDefaultIndexPattern
      ip: function (Promise, courier, config, $location, Private, createNotifier, kbnUrl, kibiDefaultIndexPattern) {
        const State = Private(StateProvider);
        const savedObjectsClient = Private(SavedObjectsClientProvider);

        return savedObjectsClient.find({
          type: 'index-pattern',
          fields: ['title'],
          perPage: 10000
        })
          .then(({ savedObjects }) => {
            /**
         *  In making the indexPattern modifiable it was placed in appState. Unfortunately,
         *  the load order of AppState conflicts with the load order of many other things
         *  so in order to get the name of the index we should use, and to switch to the
         *  default if necessary, we parse the appState with a temporary State object and
         *  then destroy it immediately after we're done
         *
         *  @type {State}
         */
            const state = new State('_a', {});

            const specified = !!state.index;

            const exists = _.findIndex(savedObjects, o => o.id === state.index) > -1;

            // kibi: use our service in case we need default indexPattern
            let idPromise;
            if (exists) {
              idPromise = Promise.resolve(state.index);
            } else {
              idPromise = kibiDefaultIndexPattern.getDefaultIndexPattern().then(indexPattern => indexPattern.id);
            }
            // kibi: end

            state.destroy();

            // kibi: added this extra promise to handle default index fetching when exists === false
            return idPromise.then(id => {
              return Promise.props({
                list: savedObjects,
                loaded: courier.indexPatterns.get(id),
                stateVal: state.index,
                stateValFound: specified && exists
              })
                .catch(error => {
                  // kibi: redirect if
                  // access to index pattern is forbidden
                  // unable to fetch the index pattern for any other reason
                  let message;
                  if (error instanceof IndexPatternAuthorizationError) {
                    message =  error.message;
                  } else {
                    message = `Could not fetch index pattern. Cause: ${id} ${JSON.stringify(error)}`;
                  }
                  createNotifier({ location: 'Discover - resolve' }).warning(message);
                  kbnUrl.redirect('/discover');
                  return Promise.halt();
                });
            });
          });
      },
      // kibi: added createNotifier, Promise, kbnUrl
      savedSearch: function (courier, savedSearches, $route, createNotifier, Promise, kbnUrl) {
        return savedSearches.get($route.current.params.id)
          .catch(error => {
            // kibi: redirect if access to index pattern in saved search is forbidden
            if (error instanceof IndexPatternAuthorizationError) {
              createNotifier({ location: 'Discover - resolve' }).warning(error.message);
              kbnUrl.redirect('/discover');
              return Promise.halt();
            }

            // kibi: redirect if access to saved search is forbidden
            if (error instanceof SavedObjectAuthorizationError) {
              createNotifier({ location: 'Discover - resolve' }).warning(error.message);
              kbnUrl.redirect('/discover');
              return Promise.halt();
            }

            // kibi: redirect if access to saved search is forbidden
            if (error instanceof IndexPatternMissingIndices) {
              createNotifier({ location: 'Discover - resolve' }).warning(error.message);
              kbnUrl.redirect('/management/siren/datamodel/SAVED_SEARCH/' + $route.current.params.id);
              return Promise.halt();
            }

            return courier.redirectWhenMissing({
              'search': '/discover',
              'index-pattern': '/settings/objects/savedSearches/' + $route.current.params.id
            })(error);
          });
      },
      entities: function (ontologyModel, indexPatterns) {
        return ontologyModel.getEntityList()
          .then(entities => {
            // filter as in discover we should always deal only with searches
            const filtered = _.filter(entities, entity => {
              return entity.type === EntityType.SAVED_SEARCH;
            });

            return filtered;
          });
      }
    }
  });
// kibi: added

app.directive('discoverApp', function () {
  return {
    restrict: 'E',
    controllerAs: 'discoverApp',
    controller: discoverController
  };
});

function discoverController($scope, $rootScope, config, courier, $route, $window, createNotifier,
  AppState, timefilter, Promise, Private, kbnUrl,
  // kibi: added
  $injector, $timeout, dataModel
) {

  const Vis = Private(VisProvider);
  const docTitle = Private(DocTitleProvider);
  const brushEvent = Private(UtilsBrushEventProvider);
  const HitSortFn = Private(PluginsKibanaDiscoverHitSortFnProvider);
  const queryFilter = Private(FilterBarQueryFilterProvider);
  const filterManager = Private(FilterManagerProvider);
  // kibi: Added quick dashboard maker
  const quickDashboard = Private(QuickDashboardProvider);
  const guessFields = Private(GuessFieldsProvider);
  const SearchSource = Private(SearchSourceProvider);
  const SegmentedRequest = Private(SegmentedRequestProvider);
  $scope.EntityType = EntityType;
  // kibi: end

  const notify = createNotifier({
    location: 'Discover'
  });

  $scope.queryDocLinks = documentationLinks.query;
  $scope.intervalOptions = Private(AggTypesBucketsIntervalOptionsProvider);
  $scope.showInterval = false;

  $scope.intervalEnabled = function (interval) {
    return interval.val !== 'custom';
  };

  $scope.topNavMenu = [{
    key: 'new',
    description: 'New Search',
    run: function () { kbnUrl.change('/discover'); },
    testId: 'discoverNewButton',
  },{
    key: 'save',
    description: 'Save Search',
    template: require('plugins/kibana/discover/partials/save_search.html'),
    testId: 'discoverSaveButton',
  },
  // {
  //   key: 'open',
  //   description: 'Open Saved Search',
  //   template: require('plugins/kibana/discover/partials/load_search.html'),
  //   testId: 'discoverOpenButton',
  // },
  {
    key: 'share',
    description: 'Share Search',
    template: require('plugins/kibana/discover/partials/share_search.html'),
    testId: 'discoverShareButton',
  }];
  $scope.timefilter = timefilter;


  // the saved savedSearch
  const savedSearch = $route.current.locals.savedSearch;
  $scope.$on('$destroy', savedSearch.destroy);

  // kibi:
  $scope.entities = $route.current.locals.entities;
  $scope.entity = _.find($scope.entities, 'id', $route.current.params.id);

  $scope.changeSelectedEntity = function (entity) {
    if (entity) {
      kbnUrl.change('/discover/{{id}}', { id: entity.id });
    }
  };

  $scope.refreshEntities = function () {
    $scope.$broadcast('refresh.entities-sidebar');
  };
  // kibi: end

  // the actual courier.SearchSource
  $scope.searchSource = savedSearch.searchSource;
  $scope.indexPattern = resolveIndexPatternLoading();
  $scope.searchSource
    .set('index', $scope.indexPattern)
    .version(true);

  // kibi: create search source for discover doc table
  $scope.tableSearchSource = new SearchSource();
  $scope.tableSearchSource
    .set('index', $scope.indexPattern)
    .highlightAll(true)
    .version(true);
  // kibi: end

  if (savedSearch.id) {
    docTitle.change(savedSearch.title);
  }

  let stateMonitor;
  const $appStatus = $scope.appStatus = this.appStatus = {
    dirty: !savedSearch.id
  };
  const $state = $scope.state = new AppState(getStateDefaults());

  $scope.uiState = $state.makeStateful('uiState');

  function getStateDefaults() {
    return {
      query: $scope.searchSource.get('query') || getDefaultQuery(),
      sort: getSort.array(savedSearch.sort, $scope.indexPattern),
      columns: savedSearch.columns.length > 0 ? savedSearch.columns : config.get('defaultColumns').slice(),
      index: $scope.indexPattern.id,
      interval: 'auto',
      filters: _.cloneDeep($scope.searchSource.getOwn('filter'))
    };
  }

  $state.index = $scope.indexPattern.id;
  $state.sort = getSort.array($state.sort, $scope.indexPattern);

  $scope.$watchCollection('state.columns', function () {
    $state.save();
  });

  $scope.opts = {
    // number of records to fetch, then paginate through
    sampleSize: config.get('discover:sampleSize'),
    timefield: $scope.indexPattern.timeFieldName,
    savedSearch: savedSearch,
    indexPatternList: _.map($route.current.locals.ip.list, ip => {
      //kibi: we have to remove the reference to the _client as it is causing max stack calls
      // when choosing and index-pattern
      const clonned = _.cloneDeep(ip);
      delete clonned._client;
      // adding methods lost while clonning as they are on __proto__
      clonned.get = function (key) {
        return _.get(this.attributes, key);
      };
      clonned.set = function (key, value) {
        return _.set(this.attributes, key, value);
      };
      clonned.has = function (key) {
        return _.has(this.attributes, key);
      };
      // save and delete methods are not used used in discover
      // adding this exceptions to be able to quickly identify the cause
      // if something change in the future
      clonned.save = function () {
        throw new Error('save method not supported due to removed _client reference');
      };
      clonned.delete = function () {
        throw new Error('save method not supported due to removed _client reference');
      };
      return clonned;
      // kibi: end
    }),
    timefilter: $scope.timefilter
  };

  const init = _.once(function () {
    const showTotal = 5;
    $scope.failuresShown = showTotal;
    $scope.showAllFailures = function () {
      $scope.failuresShown = $scope.failures.length;
    };
    $scope.showLessFailures = function () {
      $scope.failuresShown = showTotal;
    };

    stateMonitor = stateMonitorFactory.create($state, getStateDefaults());
    stateMonitor.onChange((status) => {
      $appStatus.dirty = status.dirty || !savedSearch.id;
    });
    $scope.$on('$destroy', () => stateMonitor.destroy());

    $scope.updateDataSource()
      .then(function () {
        $scope.$listen(timefilter, 'fetch', function () {
          $scope.fetch();
        });

        $scope.$watchCollection('state.sort', function (sort) {
          if (!sort) return;

          // get the current sort from {key: val} to ["key", "val"];
          const currentSort = _.pairs($scope.searchSource.get('sort')).pop();

          // if the searchSource doesn't know, tell it so
          if (!angular.equals(sort, currentSort)) $scope.fetch();
        });

        // update data source when filters update
        $scope.$listen(queryFilter, 'update', function () {
          return $scope.updateDataSource().then(function () {
            $state.save();
          });
        });

        // update data source when hitting forward/back and the query changes
        $scope.$listen($state, 'fetch_with_changes', function (diff) {
          if (diff.indexOf('query') >= 0) $scope.fetch();
        });

        // fetch data when filters fire fetch event
        $scope.$listen(queryFilter, 'fetch', $scope.fetch);

        $scope.$watch('opts.timefield', function (timefield) {
          timefilter.enabled = !!timefield;
        });

        $scope.$watch('state.interval', function () {
          $scope.fetch();
        });

        $scope.$watch('vis.aggs', function () {
        // no timefield, no vis, nothing to update
        // kibi: added missing check about $scope.vis
          if (!$scope.opts.timefield || !$scope.vis) return;

          const buckets = $scope.vis.aggs.bySchemaGroup.buckets;

          if (buckets && buckets.length === 1) {
            $scope.bucketInterval = buckets[0].buckets.getInterval();
          }
        });

        $scope.$watchMulti([
          'rows',
          'fetchStatus'
        ], (function updateResultState() {
            let prev = {};
            const status = {
              LOADING: 'loading', // initial data load
              READY: 'ready', // results came back
              NO_RESULTS: 'none' // no results came back
            };

            function pick(rows, oldRows, fetchStatus) {
              // initial state, pretend we are loading
              if (rows == null && oldRows == null) return status.LOADING;

              const rowsEmpty = _.isEmpty(rows);
              // An undefined fetchStatus means the requests are still being
              // prepared to be sent. When all requests are completed,
              // fetchStatus is set to null, so it's important that we
              // specifically check for undefined to determine a loading status.
              const preparingForFetch = _.isUndefined(fetchStatus);
              if (preparingForFetch) return status.LOADING;
              else if (rowsEmpty && fetchStatus) return status.LOADING;
              else if (!rowsEmpty) return status.READY;
              else return status.NO_RESULTS;
            }

            return function () {
              const current = {
                rows: $scope.rows,
                fetchStatus: $scope.fetchStatus
              };

              $scope.resultState = pick(
                current.rows,
                prev.rows,
                current.fetchStatus,
                prev.fetchStatus
              );

              prev = current;
            };
          }()));

        $scope.searchSource.onError(function (err) {
        // kibi: don't notify the error
        // notify.error(err);
        }).catch(notify.fatal);

        function initForTime() {
          return setupVisualization().then($scope.updateTime);
        }

        return Promise.resolve($scope.opts.timefield && initForTime())
          .then(function () {
            init.complete = true;
            $state.replace();
            $scope.$emit('application.load');
          });
      });
  });

  $scope.$watch('opts.savedSearch.title', function (title) {
    if (title) {
      $scope.entity.label = title;
    }
  });

  $scope.opts.saveDataSource = function () {
    return $scope.updateDataSource()
      .then(function () {
        savedSearch.columns = $scope.state.columns;
        savedSearch.sort = $scope.state.sort;

        // kibi: added to set the label correclty
        $scope.entity.label = savedSearch.title;

        // no need to trigger the before save hooks on discover page as there is no relations nor index-pattern to save
        const triggerBeforeSave = false;
        const promise =
        savedSearch.copyOnSave ?
          dataModel.createSearch($scope.entity, savedSearch, $scope.entities) :
          dataModel.updateSearch($scope.entity, savedSearch, triggerBeforeSave);

        return promise
        // kibi: end
          .then(function (id) {
            stateMonitor.setInitialState($state.toJSON());
            $scope.kbnTopNav.close('save');

            if (id) {
              $scope.refreshEntities(); // kibi: refresh entities list

              if (savedSearch.id !== $route.current.params.id) {
                kbnUrl.change('/discover/{{id}}', { id: id });
              } else {
                // Update defaults so that "reload saved query" functions correctly
                $state.setDefaults(getStateDefaults());
                docTitle.change(savedSearch.lastSavedTitle);
              }
            }
          });
      })
      .catch((e) => {
      // kibi: do not notity if confirmation was rejected by user
        if (savedSearch.rejectedMessages.indexOf(e.message) === -1) {
          return notify.error(e);
        } else {
          return;
        };
      // kibi: end
      });
  };

  const initFunction = function (segmented) {
    function flushResponseData() {
      $scope.hits = 0;
      $scope.faliures = [];
      $scope.rows = [];
      $scope.fieldCounts = {};
      // kibi: clear the flag
      delete $scope.segmentedRequest;
      // kibi: end
    }

    if (!$scope.rows) flushResponseData();

    const sort = $state.sort;
    const timeField = $scope.indexPattern.timeFieldName;

    /**
     * Basically an emum.
     *
     * opts:
     *   "time" - sorted by the timefield
     *   "non-time" - explicitly sorted by a non-time field, NOT THE SAME AS `sortBy !== "time"`
     *   "implicit" - no sorting set, NOT THE SAME AS "non-time"
     *
     * @type {String}
     */
    const sortBy = (function () {
      if (!_.isArray(sort)) return 'implicit';
      else if (sort[0] === '_score') return 'implicit';
      else if (sort[0] === timeField) return 'time';
      else return 'non-time';
    }());

    let sortFn = null;
    if (sortBy !== 'implicit') {
      sortFn = new HitSortFn(sort[1]);
    }

    $scope.updateTime();
    if (sort[0] === '_score') segmented.setMaxSegments(1);
    segmented.setDirection(sortBy === 'time' ? (sort[1] || 'desc') : 'desc');
    segmented.setSortFn(sortFn);
    segmented.setSize($scope.opts.sampleSize);

    // triggered when the status updated
    segmented.on('status', function (status) {
      $scope.fetchStatus = status;
    });

    segmented.on('first', function () {
      flushResponseData();
    });

    segmented.on('segment', notify.timed('handle each segment', function (resp) {
      if (resp._shards && resp._shards.failed > 0) {
        $scope.failures = _.union($scope.failures, resp._shards.failures);
        $scope.failures = _.uniq($scope.failures, false, function (failure) {
          return failure.index + failure.shard + failure.reason;
        });
      }
    }));

    segmented.on('mergedSegment', function (merged) {
      $scope.mergedEsResp = merged;
      $scope.hits = merged.hits.total;

      const indexPattern = $scope.searchSource.get('index');

      // the merge rows, use a new array to help watchers
      $scope.rows = merged.hits.hits.slice();

      notify.event('flatten hit and count fields', function () {
        let counts = $scope.fieldCounts;

        // if we haven't counted yet, or need a fresh count because we are sorting, reset the counts
        if (!counts || sortFn) counts = $scope.fieldCounts = {};

        $scope.rows.forEach(function (hit) {
          // skip this work if we have already done it
          if (hit.$$_counted) return;

          // when we are sorting results, we need to redo the counts each time because the
          // "top 500" may change with each response, so don't mark this as counted
          if (!sortFn) hit.$$_counted = true;

          const fields = _.keys(indexPattern.flattenHit(hit));
          let n = fields.length;
          let field;
          while (field = fields[--n]) {
            if (counts[field]) counts[field] += 1;
            else counts[field] = 1;
          }
        });
      });
    });

    segmented.on('complete', function () {
      if ($scope.fetchStatus.hitCount === 0) {
        flushResponseData();
      }

      $scope.fetchStatus = null;
    });
  };

  $scope.opts.fetch = $scope.fetch = function () {
    // ignore requests to fetch before the app inits
    if (!init.complete) return;

    $scope.updateTime();

    $scope.updateDataSource()
      .then(setupVisualization)
      .then(function () {
        $state.save();
        // kibi: if there is no segmented request already created create new one
        if (!$scope.segmentedRequest) {
          new SegmentedRequest($scope.searchSource, Promise.defer(), initFunction);
        }
        // kibi: end
        return courier.fetch();
      })
      .catch(notify.error);
  };

  // kibi: create new segmented request
  $scope.segmentedRequest = new SegmentedRequest($scope.searchSource, Promise.defer(), initFunction);

  // kibi: use parseWithPrecision()
  $scope.updateTime = function () {
    $scope.timeRange = {
      from: parseWithPrecision(timefilter.time.from, false, $rootScope.sirenTimePrecision),
      to: parseWithPrecision(timefilter.time.to, true, $rootScope.sirenTimePrecision)
    };
  };

  $scope.newQuery = function () {
    kbnUrl.change('/discover');
  };

  $scope.updateDataSource = Promise.method(function updateDataSource() {
    $scope.searchSource
      .size($scope.opts.sampleSize)
      .sort(getSort($state.sort, $scope.indexPattern))
      .query(!$state.query ? null : $state.query)
      .set('filter', queryFilter.getFilters());

    // kibi: update discover table source source too
    $scope.tableSearchSource
      .size($scope.opts.sampleSize)
      .sort(getSort($state.sort, $scope.indexPattern))
      .query(!$state.query ? null : $state.query)
      .set('filter', queryFilter.getFilters());
    // kibi: end
  });

  $scope.setSortOrder = function setSortOrder(columnName, direction) {
    $scope.state.sort = [columnName, direction];
  };

  // TODO: On array fields, negating does not negate the combination, rather all terms
  // kibi: added options parameter to support more like this queries
  $scope.filterQuery = function (field, values, operation, options) {
    $scope.indexPattern.popularizeField(field, 1);
    filterManager.add(field, values, operation, $state.index, options);
  };

  $scope.addColumn = function addColumn(columnName) {
    $scope.indexPattern.popularizeField(columnName, 1)
      .then(() => {
        columnActions.addColumn($scope.state.columns, columnName);
      });
  };

  $scope.removeColumn = function removeColumn(columnName) {
    $scope.indexPattern.popularizeField(columnName, 1)
      .then(() => {
        columnActions.removeColumn($scope.state.columns, columnName);
      });
  };

  $scope.moveColumn = function moveColumn(columnName, newIndex) {
    columnActions.moveColumn($scope.state.columns, columnName, newIndex);
  };

  $scope.toTop = function () {
    $window.scrollTo(0, 0);
  };

  // kibi: Added quick dashboard maker
  $scope.makeQuickDashboard = function () {
    return $scope.updateDataSource()
      .then(function () {
        savedSearch.columns = $scope.state.columns;
        savedSearch.sort = $scope.state.sort;

        return quickDashboard.create({
          entity: $scope.entity,
          savedSearch,
          fieldNames: $scope.state.columns,
          query: $scope.state.query,
          filters: $scope.state.filters,
          timeFilter: $scope.timefilter
        });
      });
  };

  $scope.guessFields = function () {
    const index = $scope.indexPattern;

    // The timeField is excluded since it's already implicitly considered by discover.
    // Apart from that, 'bad' fields are already filtered away by the ranking procedure.
    const timeField = index.fields.byName[index.timeFieldName];
    const fields = _.without(index.fields, timeField);

    return guessFields(index, fields, { savedSearch })
      .then(guessedFields => {
        if (!guessedFields) { return; }

        $scope.state.columns = _.map(guessedFields, 'name');
      });
  };
  // kibi: end

  let loadingVis;
  function setupVisualization() {
    // If we're not setting anything up we need to return an empty promise
    // kibi: added condition to check if there are any fields
    if (!$scope.opts.timefield  || $scope.indexPattern.fields.length === 0) return Promise.resolve();
    if (loadingVis) return loadingVis;

    const visStateAggs = [
      {
        type: 'count',
        schema: 'metric'
      },
      {
        type: 'date_histogram',
        schema: 'segment',
        params: {
          field: $scope.opts.timefield,
          interval: $state.interval
        }
      }
    ];

    // we have a vis, just modify the aggs
    if ($scope.vis) {
      const visState = $scope.vis.getEnabledState();
      visState.aggs = visStateAggs;

      $scope.vis.setState(visState);
      return Promise.resolve($scope.vis);
    }

    $scope.vis = new Vis($scope.indexPattern, {
      title: savedSearch.title,
      type: 'histogram',
      params: {
        addLegend: false,
        addTimeMarker: true
      },
      listeners: {
        click: function (e) {
          notify.log(e);
          timefilter.time.from = moment(e.point.x);
          timefilter.time.to = moment(e.point.x + e.data.ordered.interval);
          timefilter.time.mode = 'absolute';
        },
        brush: brushEvent($scope.state)
      },
      aggs: visStateAggs
    });

    $scope.searchSource.aggs(function () {
      $scope.vis.requesting();
      return $scope.vis.aggs.toDsl();
    });

    // stash this promise so that other calls to setupVisualization will have to wait
    loadingVis = new Promise(function (resolve) {
      $scope.$on('ready:vis', function () {
        resolve($scope.vis);
      });
    })
      .finally(function () {
      // clear the loading flag
        loadingVis = null;
      });

    return loadingVis;
  }

  function resolveIndexPatternLoading() {
    const props = $route.current.locals.ip;
    const loaded = props.loaded;
    const stateVal = props.stateVal;
    const stateValFound = props.stateValFound;

    const own = $scope.searchSource.getOwn('index');

    if (own && !stateVal) return own;
    if (stateVal && !stateValFound) {
      const err = '"' + stateVal + '" is not a configured pattern ID. ';
      if (own) {
        notify.warning(err + ' Using the saved index pattern: "' + own.id + '"');
        return own;
      }

      notify.warning(err + ' Using the default index pattern: "' + loaded.id + '"');
    }
    return loaded;
  }

  init();
}
