'use strict';

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _fs_stubs = require('./__tests__/__mocks__/_fs_stubs');

var _metrics = require('./metrics');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

jest.mock('fs', () => ({
  readFile: jest.fn()
}));

jest.mock('os', () => ({
  freemem: jest.fn(),
  totalmem: jest.fn(),
  uptime: jest.fn()
}));

jest.mock('process', () => ({
  uptime: jest.fn()
}));

describe('Metrics', function () {
  _fs2.default.readFile.mockImplementation(_fs_stubs.readFileMock);

  const sampleConfig = {
    ops: {
      interval: 5000
    },
    server: {
      port: 5603
    }
  };
  const config = { get: path => _lodash2.default.get(sampleConfig, path) };

  let metrics;

  beforeEach(() => {
    const server = { log: _sinon2.default.mock() };

    metrics = new _metrics.Metrics(config, server);
  });

  afterEach(() => {
    (0, _fs_stubs.setMockFiles)();
  });

  describe('capture', () => {
    it('merges all metrics', async () => {
      (0, _fs_stubs.setMockFiles)();
      _sinon2.default.stub(metrics, 'captureEvent').returns({ 'a': [{ 'b': 2 }, { 'd': 4 }], process: { uptime_ms: 1980 } });
      _sinon2.default.stub(metrics, 'captureCGroupsIfAvailable').returns({ 'a': [{ 'c': 3 }, { 'e': 5 }] });
      _sinon2.default.stub(Date.prototype, 'toISOString').returns('2017-04-14T18:35:41.534Z');

      const capturedMetrics = await metrics.capture();
      expect(capturedMetrics).toMatchObject({
        last_updated: '2017-04-14T18:35:41.534Z',
        collection_interval_in_millis: 5000,
        a: [{ b: 2, c: 3 }, { d: 4, e: 5 }], process: { uptime_ms: 1980 }
      });
    });
  });

  describe('captureEvent', () => {
    it('parses the hapi event', () => {
      _sinon2.default.stub(_os2.default, 'uptime').returns(12000);
      _sinon2.default.stub(process, 'uptime').returns(5000);

      _os2.default.freemem.mockImplementation(() => 12);
      _os2.default.totalmem.mockImplementation(() => 24);

      const pidMock = jest.fn();
      pidMock.mockReturnValue(8675309);
      Object.defineProperty(process, 'pid', { get: pidMock }); //

      const hapiEvent = {
        'requests': { '5603': { 'total': 22, 'disconnects': 0, 'statusCodes': { '200': 22 } } },
        'responseTimes': { '5603': { 'avg': 1.8636363636363635, 'max': 4 } },
        'osload': [2.20751953125, 2.02294921875, 1.89794921875],
        'osmem': { 'total': 17179869184, 'free': 102318080 },
        'osup': 1008991,
        'psup': 7.168,
        'psmem': { 'rss': 193716224, 'heapTotal': 168194048, 'heapUsed': 130553400, 'external': 1779619 },
        'concurrent_connections': 0,
        'psdelay': 1.6091690063476562,
        'host': 'blahblah.local'
      };

      expect(metrics.captureEvent(hapiEvent)).toMatchObject({
        'concurrent_connections': 0,
        'os': {
          'load': {
            '15m': 1.89794921875,
            '1m': 2.20751953125,
            '5m': 2.02294921875
          },
          'memory': {
            'free_in_bytes': 12,
            'total_in_bytes': 24
          },
          'uptime_in_millis': 12000000
        },
        'process': {
          'memory': {
            'heap': {
              'total_in_bytes': 168194048,
              'used_in_bytes': 130553400
            },
            'resident_set_size_in_bytes': 193716224
          },
          'pid': 8675309
        },
        'requests': {
          'disconnects': 0,
          'total': 22
        },
        'response_times': {
          'avg_in_millis': 1.8636363636363635,
          'max_in_millis': 4
        }
      });
    });

    it('parses event with missing fields / NaN for responseTimes.avg', () => {
      const hapiEvent = {
        requests: {
          '5603': { total: 22, disconnects: 0, statusCodes: { '200': 22 } }
        },
        responseTimes: { '5603': { avg: NaN, max: 4 } },
        host: 'blahblah.local'
      };

      expect(metrics.captureEvent(hapiEvent)).toMatchObject({
        process: { memory: { heap: {} }, pid: 8675309, uptime_in_millis: 5000000 },
        os: {
          load: {},
          memory: { free_in_bytes: 12, total_in_bytes: 24 }
        },
        response_times: { max_in_millis: 4 },
        requests: { total: 22, disconnects: 0 }
      });
    });
  });

  describe('captureCGroups', () => {
    afterEach(() => {
      (0, _fs_stubs.setMockFiles)();
    });

    it('returns undefined if cgroups do not exist', async () => {
      (0, _fs_stubs.setMockFiles)();

      const stats = await metrics.captureCGroups();

      expect(stats).toBe(undefined);
    });

    it('returns cgroups', async () => {
      const fsStub = (0, _fs_stubs.cGroups)();
      (0, _fs_stubs.setMockFiles)(fsStub.files);

      const capturedMetrics = await metrics.captureCGroups();

      expect(capturedMetrics).toMatchObject({
        os: {
          cgroup: {
            cpuacct: {
              control_group: `/${fsStub.hierarchy}`,
              usage_nanos: 357753491408
            },
            cpu: {
              control_group: `/${fsStub.hierarchy}`,
              cfs_period_micros: 100000,
              cfs_quota_micros: 5000,
              stat: {
                number_of_elapsed_periods: 0,
                number_of_times_throttled: 10,
                time_throttled_nanos: 20
              }
            }
          }
        }
      });
    });
  });

  describe('captureCGroupsIfAvailable', () => {
    afterEach(() => {
      (0, _fs_stubs.setMockFiles)();
    });

    it('marks cgroups as unavailable and prevents subsequent calls', async () => {
      (0, _fs_stubs.setMockFiles)();
      _sinon2.default.spy(metrics, 'captureCGroups');

      expect(metrics.checkCGroupStats).toBe(true);

      await metrics.captureCGroupsIfAvailable();
      expect(metrics.checkCGroupStats).toBe(false);

      await metrics.captureCGroupsIfAvailable();
      _sinon2.default.assert.calledOnce(metrics.captureCGroups);
    });

    it('allows subsequent calls if cgroups are available', async () => {
      const fsStub = (0, _fs_stubs.cGroups)();
      (0, _fs_stubs.setMockFiles)(fsStub.files);
      _sinon2.default.spy(metrics, 'captureCGroups');

      expect(metrics.checkCGroupStats).toBe(true);

      await metrics.captureCGroupsIfAvailable();
      expect(metrics.checkCGroupStats).toBe(true);

      await metrics.captureCGroupsIfAvailable();
      _sinon2.default.assert.calledTwice(metrics.captureCGroups);
    });
  });
});
