import { uiModules } from 'ui/modules';
import chrome from 'ui/chrome';
import template from './siren_data_export.html';
import './siren_data_export.less';
import { SirenDataExportHelperProvider } from '../helper/siren_data_export_helper';
import { getAlternativeSortingField } from 'ui/kibi/components/courier/data_source/get_alternative_sorting_field';
import { find } from 'lodash';

function updateSelectorsInForm(template, currentId, timestamp) {
  return template = template.split(currentId).join(currentId + '-' + timestamp);
};

const module = uiModules.get('kibana');
module.directive('sirenDataExport', function (savedSearches, Private,
  createNotifier, indexPatterns, $timeout, jdbcDatasources) {
  return {
    restrict: 'A',
    scope: {
      savedSearchId: '=',
      useGlobalTime: '=?',
      useAppState: '=?',
      exportedFileName: '=',
      sorting: '=',
      siren: '=?'
    },
    link: function (scope, $element, attrs) {

      scope.getIframerUrl = () => {
        return chrome.getBasePath() + '/iframe_export';
      };

      if (scope.useGlobalTime === undefined) {
        scope.useGlobalTime = true;
      };

      if (scope.useAppState === undefined) {
        scope.useAppState = true;
      };

      function getSortingObject(index) {
        if (!scope.sorting || scope.sorting.length === 0) {
          return null;
        };

        const field = find(index.fields, field => field.name === scope.sorting[0]);
        const altSortingField = getAlternativeSortingField(index, field);
        const sortingObject = {};
        if (altSortingField !== null) {
          sortingObject[altSortingField.name] = scope.sorting[1];
        } else {
          sortingObject[scope.sorting[0]] = scope.sorting[1];
        };
        return sortingObject;
      };

      const timestamp = new Date().getTime();

      const notify = createNotifier({
        location: 'Data Export'
      });
      let interval;
      let userExportForm;
      let searchSource;
      let indexPattern;
      let expectedSize;
      let limitInputTimeout;
      // by default format selected as CSV
      let selectedCsv = true;

      $element.css('cursor', 'pointer');
      $element.on('click', () => {
        if (attrs.exportFormIsOpened) {
          return;
        }

        attrs.exportFormIsOpened = true;
        savedSearches.get(scope.savedSearchId)
          .then(search => {
            searchSource = JSON.parse(search.kibanaSavedObjectMeta.searchSourceJSON);
            return Promise.all([indexPatterns.get(searchSource.index), jdbcDatasources.listVirtualIndices()])
              .then(([selectedIndexPattern, virtualIndicesList]) => {
                const virtualIndicesIds = virtualIndicesList.map(index => index._id);
                if (virtualIndicesIds.indexOf(selectedIndexPattern.title) !== -1) {
                  notify.warning('Export for JDBC datasources is currently unsupported');
                  return;
                };
                if (!selectedIndexPattern.timeFieldName) {
                  scope.useGlobalTime = false;
                };
                const MARGIN_TOP_FOR_FORM = 10;
                const USER_EXPORT_FORM_WIDTH = 270;
                const USER_EXPORT_FORM_HEIGHT = 165;
                const CSV_ADDITIONAL_SETTINGS_HEIGHT = 30;
                const FIELD_SELECTOR_HEIGHT = 150;
                const TIME_SELECTOR_HEIGHT = 45;
                let ADDITIONAL_SETTINGS_HEIGHT = 85;
                if (scope.useGlobalTime === true) {
                  ADDITIONAL_SETTINGS_HEIGHT += TIME_SELECTOR_HEIGHT;
                };

                indexPattern = selectedIndexPattern;

                scope.sortingObject = getSortingObject(indexPattern);

                function updatePositionForUserExportForm(formObject, $element, positionString) {
                  if (positionString) {
                    const newPosition = JSON.parse(positionString);
                    formObject.style.top = newPosition.top - MARGIN_TOP_FOR_FORM - USER_EXPORT_FORM_HEIGHT + 'px';
                    formObject.style.left = newPosition.left + MARGIN_TOP_FOR_FORM + 'px';
                  } else {
                    const container = $element.closest('.vis-container');
                    formObject.style.top =
                      container.offset().top +
                      container.height() -
                      MARGIN_TOP_FOR_FORM -
                      USER_EXPORT_FORM_HEIGHT + 'px';
                    formObject.style.left = container.offset().left + MARGIN_TOP_FOR_FORM + 'px';
                  };
                  // ***** Will be used when we will use directive somewhere else instead of table
                  // const clientWidth = document.documentElement.clientWidth;
                  // const clientHeight = document.documentElement.scrollHeight;
                  // const topPositionForExportForm = offsetObject.top + MARGIN_TOP_FOR_FORM + $element.height();
                  // if (clientHeight - USER_EXPORT_FORM_HEIGHT - topPositionForExportForm < 0) {
                  //   formObject.style.top = offsetObject.top - USER_EXPORT_FORM_HEIGHT - MARGIN_TOP_FOR_FORM + 'px';
                  // } else {
                  //   formObject.style.top = topPositionForExportForm + 'px';
                  // };
                  // if (clientWidth - USER_EXPORT_FORM_WIDTH - offsetObject.left < 0) {
                  //   formObject.style.left = clientWidth - USER_EXPORT_FORM_WIDTH + 'px';
                  // } else {
                  //   formObject.style.left = offsetObject.left + 'px';
                  // }
                  // *****
                };
                // create unique ids and labels by timestamp to prevent conflicts between forms
                // if any other export form also was opened
                let templateForForm = template;
                templateForForm = updateSelectorsInForm(templateForForm, 'formatCSV', timestamp);
                templateForForm = updateSelectorsInForm(templateForForm, 'formatJSON', timestamp);
                templateForForm = updateSelectorsInForm(templateForForm, 'sirenDataExport', timestamp);
                templateForForm = updateSelectorsInForm(templateForForm, 'formattedYes', timestamp);
                templateForForm = updateSelectorsInForm(templateForForm, 'formattedNo', timestamp);
                templateForForm = updateSelectorsInForm(templateForForm, 'limited', timestamp);
                userExportForm = document.createElement('div');
                userExportForm.style.position = 'absolute';
                userExportForm.style.zIndex = 999;
                userExportForm.style.width = USER_EXPORT_FORM_WIDTH + 'px';
                userExportForm.style.height = USER_EXPORT_FORM_HEIGHT + 'px';
                userExportForm.className = 'kuiModal';
                userExportForm.innerHTML = templateForForm;

                const SirenDataExportHelperClass = Private(SirenDataExportHelperProvider);
                const sirenDataExportHelper = new SirenDataExportHelperClass({
                  form: userExportForm,
                  source: searchSource,
                  indexPattern,
                  stamp: timestamp,
                  useAppState: scope.useAppState,
                  useGlobalTime: scope.useGlobalTime,
                  attrs,
                  exportedFileName: scope.exportedFileName,
                  siren: scope.siren
                });

                const indexFieldNames = indexPattern.fields.map(field => field.name);
                indexPattern.metaFields.forEach(metaField => {
                  indexFieldNames.splice(indexFieldNames.indexOf(metaField), 1);
                });
                const escapeField = fieldName => fieldName.replace(/</g, '&lt;').replace(/>/g, '&gt;');
                const fieldSelectorHtmlTemplate = indexFieldNames
                  .map(field => `<input type="checkbox" name="${field}" value="${field}" checked> ${escapeField(field)}`)
                  .join('<br/>');
                const fieldsContainer = userExportForm.querySelector('div.fields-container div.fields');
                const fieldsPreview = userExportForm.querySelector('div.fields-container div.fields-preview');
                fieldsContainer.style.height = FIELD_SELECTOR_HEIGHT + 'px';
                const innerFieldsContainer = fieldsContainer.querySelector('div.fields-in');
                innerFieldsContainer.innerHTML = fieldSelectorHtmlTemplate;

                Array.prototype.forEach.call(innerFieldsContainer.querySelectorAll('input'), (field) => {
                  field.onclick = () => {
                    refreshFormState.call(null, null);
                    const selectedFields = sirenDataExportHelper.getSelectedFieldsArray();
                    const pluralFields = selectedFields.length === 1 ? ' field' : ' fields';
                    fieldsPreview.querySelector('div.fields-label').innerHTML = selectedFields.length > 0 ?
                      `${selectedFields.length} ${pluralFields} exported` : 'All fields exported';
                  };
                });

                const toggleFields = (checked) => {
                  Array.prototype.forEach.call(innerFieldsContainer.querySelectorAll('input'), (field) => {
                    field.checked = checked;
                  });
                  fieldsPreview.querySelector('div.fields-label').innerHTML = 'All fields exported';
                  const userFormElements = document.forms['sirenDataExport-' + timestamp].elements;
                  const format = userFormElements.format.value;
                  sirenDataExportHelper.updateEstimatedSize({ format });
                };

                const buttonToSelectAllFields = fieldsContainer.querySelector('button.fields-select-all');
                const buttonToUnselectAllFields = fieldsContainer.querySelector('button.fields-deselect-all');

                buttonToSelectAllFields.onclick = event => {
                  event.preventDefault();
                  toggleFields(true);
                  buttonToSelectAllFields.style.display = 'none';
                  buttonToUnselectAllFields.style.display = 'block';
                };
                buttonToUnselectAllFields.onclick = (event) => {
                  event.preventDefault();
                  toggleFields(false);
                  buttonToSelectAllFields.style.display = 'block';
                  buttonToUnselectAllFields.style.display = 'none';
                };

                const fieldsPreviewButton = fieldsPreview.querySelector('button.select-fields');
                fieldsPreviewButton.onclick = event => {
                  event.preventDefault();
                  const FIELDS_CONTAINER_LEFT_POSITION = 95;
                  const fieldsPreviewButtonIcon = fieldsPreviewButton.querySelector('i');
                  if (fieldsContainer.getAttribute('selected')) {
                    fieldsContainer.removeAttribute('selected');
                    fieldsContainer.style.display = 'none';
                    fieldsContainer.style.right = '';
                    fieldsContainer.style.left = '';
                    fieldsPreviewButtonIcon.setAttribute('class', 'kuiIcon fa-chevron-circle-up');
                    return;
                  };
                  fieldsContainer.setAttribute('selected', 'true');
                  fieldsContainer.style.display = 'block';
                  const clientWidth = document.documentElement.clientWidth;
                  const fieldsContainerLeftPosition = fieldsContainer.getBoundingClientRect().left;
                  if (fieldsContainerLeftPosition + fieldsContainer.offsetWidth > clientWidth) {
                    fieldsContainer.style.right = '0px';
                  } else {
                    fieldsContainer.style.left = FIELDS_CONTAINER_LEFT_POSITION + 'px';
                  };
                  fieldsPreviewButtonIcon.setAttribute('class', 'kuiIcon fa-chevron-circle-down');
                };

                updatePositionForUserExportForm(userExportForm, $element, null);

                const cancelButton = userExportForm.querySelector('button[name="cancelButton"]');
                cancelButton.onclick = () => {
                  if (limitInputTimeout) {
                    $timeout.cancel(limitInputTimeout);
                  };
                  sirenDataExportHelper.destroyForm(interval);
                };

                const exportButton = userExportForm.querySelector('button[name="exportButton"]');
                exportButton.onclick = () => {
                  const sortingObject = scope.sortingObject ? JSON.stringify(scope.sortingObject) : null;
                  sirenDataExportHelper.submitForm(sortingObject);
                };

                const limitField = userExportForm.querySelector('input[name="limit"]');
                limitField.oninput = () => {
                  if (limitInputTimeout) {
                    $timeout.cancel(limitInputTimeout);
                  };
                  const incorrectSizeContainer = userExportForm.querySelector('span.incorrect-size');
                  if (limitField.value <= 0 && limitField.value !== '') {
                    incorrectSizeContainer.style.display = 'inline';
                    exportButton.disabled = true;
                  } else {
                    incorrectSizeContainer.style.display = 'none';
                    exportButton.disabled = false;
                    limitInputTimeout = $timeout(() => refreshFormState.call(null, null), 500);
                  };
                  expectedSize = limitField.value;
                  const maxSizeContainer = userExportForm.querySelector('span.max-size');
                  if (expectedSize > sirenDataExportHelper.getIndexSize()) {
                    maxSizeContainer.style.display = 'inline';
                  } else {
                    maxSizeContainer.style.display = 'none';
                  };
                };

                const additionalSettingsButton = userExportForm.querySelector('a.additional-selector-link');
                additionalSettingsButton.onclick = openAdditionalSelector;

                const jsonButton = userExportForm.querySelector(`label[for="formatJSON-${timestamp}"]`);
                jsonButton.onclick = refreshFormState.bind(null, 'json');

                const csvButton = userExportForm.querySelector(`label[for="formatCSV-${timestamp}"]`);
                csvButton.onclick = refreshFormState.bind(null, 'csv');

                const iframe = userExportForm.querySelector('iframe');
                iframe.src = scope.getIframerUrl();
                iframe.onload = function () {
                  const content = iframe.contentWindow.document || iframe.contentWindow.document;
                  const text = content.body.innerHTML;
                  if (text.includes('statusCode":400')) {
                    notify.error(text.replace(/<[^>]*>/g, ''));
                    sirenDataExportHelper.destroyForm();
                  };
                };

                function refreshFormState(formatType) {
                  const userFormElements = document.forms[`sirenDataExport-${timestamp}`].elements;
                  const format = formatType ? formatType : userFormElements.format.value;
                  selectedCsv = format === 'csv';
                  const size = userFormElements.limit.value;
                  if (size && size < 1) {
                    return;
                  };
                  sirenDataExportHelper.updateEstimatedSize({ format, size });
                  updateStateOfAdditionalSettings(format);
                };

                function hideAdditionalSettings() {
                  const additionalSettings = userExportForm.querySelectorAll('.additional-export-setting');
                  Array.prototype.forEach.call(additionalSettings, setting => {
                    setting.style.display = 'none';
                    setting.removeAttribute('selected');
                  });
                  userExportForm.style.height = USER_EXPORT_FORM_HEIGHT + 'px';
                  additionalSettingsButton.removeAttribute('selected');
                };

                function updateStateOfAdditionalSettings(formatType) {
                  if (!additionalSettingsButton.getAttribute('selected')) {
                    return;
                  };
                  const additionalSettings = userExportForm.querySelectorAll('.additional-export-setting');
                  Array.prototype.forEach.call(additionalSettings, setting => {
                    if (!setting.className.includes('for-csv') && !setting.className.includes('for-time')) {
                      setting.style.display = 'block';
                      setting.setAttribute('selected', 'true');
                    } else if (scope.useGlobalTime === true && setting.className.includes('for-time')) {
                      setting.style.display = 'block';
                    };
                  });
                  const additionalSettingsForCsv = userExportForm.querySelectorAll('.for-csv');
                  if (formatType === 'csv') {
                    Array.prototype.forEach.call(additionalSettingsForCsv, setting => {
                      setting.style.display = 'block';
                      setting.setAttribute('selected', 'true');
                    });
                    selectedCsv = true;
                    userExportForm.style.height =
                    USER_EXPORT_FORM_HEIGHT +
                    CSV_ADDITIONAL_SETTINGS_HEIGHT +
                    ADDITIONAL_SETTINGS_HEIGHT + 'px';
                  } else {
                    Array.prototype.forEach.call(additionalSettingsForCsv, setting => {
                      setting.style.display = 'none';
                      setting.removeAttribute('selected');
                    });
                    selectedCsv = false;
                    userExportForm.style.height = USER_EXPORT_FORM_HEIGHT + ADDITIONAL_SETTINGS_HEIGHT + 'px';
                  };
                };

                function enoughSpaceToOpenAdditional(form) {
                  const spaceBelow = document.documentElement.scrollHeight - form.offsetTop - USER_EXPORT_FORM_HEIGHT;
                  const requiredSpace = selectedCsv ?
                    CSV_ADDITIONAL_SETTINGS_HEIGHT + ADDITIONAL_SETTINGS_HEIGHT :
                    ADDITIONAL_SETTINGS_HEIGHT;
                  return spaceBelow >= requiredSpace;
                };

                function openAdditionalSelector() {
                  if (additionalSettingsButton.getAttribute('selected')) {
                    hideAdditionalSettings();
                    return;
                  };
                  additionalSettingsButton.setAttribute('selected', 'true');
                  const userFormElements = document.forms['sirenDataExport-' + timestamp].elements;
                  const format = userFormElements.format.value;
                  // ***** Related position for modal is hidden for now
                  // if (!enoughSpaceToOpenAdditional(userExportForm)) {
                  // userExportForm.scrollIntoView();
                  // const positionBefore = $element.offset().top - USER_EXPORT_FORM_HEIGHT - MARGIN_TOP_FOR_FORM;
                  // if (selectedCsv) {
                  //   userExportForm.style.top = positionBefore - CSV_ADDITIONAL_SETTINGS_HEIGHT - ADDITIONAL_SETTINGS_HEIGHT + 'px';
                  // } else {
                  //   userExportForm.style.top = positionBefore - ADDITIONAL_SETTINGS_HEIGHT + 'px';
                  // };
                  // };
                  // *****
                  updateStateOfAdditionalSettings(format);
                };

                // update position of form if position of element was changed
                scope.$watch(() => {
                  const parent = $element.closest('.vis-container');
                  const parentPosition = parent.offset();
                  return JSON.stringify({
                    top: parentPosition.top + parent.height(),
                    left: parentPosition.left
                  });
                }, positionString => {
                  updatePositionForUserExportForm(userExportForm, $element, positionString);
                });

                sirenDataExportHelper.updateEstimatedSize({ callWithTotalSizeUpdating: true });

                document.body.appendChild(userExportForm);

                scope.$watch('sorting', () => {
                  scope.sortingObject = getSortingObject(indexPattern);
                });

                scope.$on('$destroy', () => {
                  $element.off('click');
                  sirenDataExportHelper.destroyForm();
                });
              });
          }).catch(e => notify.error(e));
      });

    }
  };
});
