'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FederateVersionParser = exports.Version = undefined;

var _errors = require('./errors');

const FEDERATE_VERSION_SEPARATOR = '-';
// Matches anything that starts like '6.5.4-10.3.0' and may or may not have '-' separated custom label,
// eg: 6.5.4-10.3.0-SNAPSHOT
const VALID_FEDERATE_VERSION_REGEX = /^(\d+\.)(\d+\.)(\d+)-(\d+\.)(\d+\.)(\d+)(-.*?)?$/;
// Matches anything in form of 'majorVersion.minorVersion.patch' (All numbers), eg: '5.6.14'
const VALID_VERSION_REGEX = /^(\d+\.)(\d+\.)(\d+)$/;

class Version {

  static isValid(version) {
    return VALID_VERSION_REGEX.test(version);
  }

  constructor(version) {
    if (Version.isValid(version)) {
      this._rawVersion = version;
      this._splitVersions = version.split('.').map(Number);
    } else {
      throw new _errors.InvalidVersionError(`Invalid version found: ${version}`);
    }
  }

  get rawVersion() {
    return this._rawVersion;
  }

  get majorVersion() {
    return this._splitVersions[0];
  }

  get minorVersion() {
    return this._splitVersions[1];
  }

  get patchVersion() {
    return this._splitVersions[2];
  }

  /**
   * Tests if the passed version has a higher Major than this version
   * @param  {Version}  otherVersion
   * @return {Boolean}  True if the passed version has a higher major version
   */
  compareMajorVersion(otherVersion) {
    return this.majorVersion < otherVersion.majorVersion;
  }

  /**
   * Compares this object with the passed version
   * @param  {Version}  otherVersion to be matched against
   * @return {Number}    -1: if the passed version is lesser than this, 0: is equal, 1: is greater than this version
   */
  compareWith(otherVersion) {
    for (let i = 0; i < 3; i++) {
      const diff = otherVersion._splitVersions[i] - this._splitVersions[i];
      if (diff) {
        return diff > 0 ? 1 : -1;
      }
    }
    return 0;
  }

  toString() {
    return this._rawVersion;
  }
}

class FederateVersionParser {

  static isValid(federateVersion) {
    return VALID_FEDERATE_VERSION_REGEX.test(federateVersion);
  }

  static retrieveElasticsearchAndFederateVersions(federateVersion) {
    if (FederateVersionParser.isValid(federateVersion)) {
      const versions = federateVersion.split(FEDERATE_VERSION_SEPARATOR);
      return {
        ES: versions[0],
        Federate: versions[1]
      };
    } else {
      throw new _errors.InvalidVersionError(`Invalid Federate version found: ${federateVersion}`);
    }
  }

  constructor(federateVersion) {
    this._rawVersion = federateVersion;
    const versions = FederateVersionParser.retrieveElasticsearchAndFederateVersions(federateVersion);
    this._internalFederateVersion = new Version(versions.Federate);
    this._elasticsearchVersion = new Version(versions.ES);
  }

  get rawVersion() {
    return this._rawVersion;
  }

  get internalFederateVersion() {
    return this._internalFederateVersion;
  }

  get elasticsearchVersion() {
    return this._elasticsearchVersion;
  }

  /**
   * Checks if the passed version has a higher major version
   * @param  {FederateVersionParser}  otherVersion
   * @return {Boolean}    True if the passed version has a higher major version
   */
  compareMajorVersion(otherVersion) {
    const esCheck = this._elasticsearchVersion.compareMajorVersion(otherVersion._elasticsearchVersion);
    if (!esCheck) {
      const federateCheck = this._internalFederateVersion.compareMajorVersion(otherVersion._internalFederateVersion);
      return federateCheck;
    }
    return esCheck;
  }

  /**
   * Compares this object with the passed version
   * @param  {FederateVersionParser}  otherVersion to be matched against
   * @return {Number}    -1: if the passed version is lesser than this, 0: is equal, 1: is greater than this version
   */
  compareWith(otherVersion) {
    const compareES = this._elasticsearchVersion.compareWith(otherVersion._elasticsearchVersion);
    if (compareES >= 0) {
      const compareFederate = this._internalFederateVersion.compareWith(otherVersion._internalFederateVersion);
      if (compareFederate >= 0) {
        const isEsGreater = compareES === 0 ? 0 : 1;
        return compareFederate === 0 ? isEsGreater : 1;
      }
    }
    return -1;
  }

  toString() {
    return this._rawVersion;
  }
};

exports.Version = Version;
exports.FederateVersionParser = FederateVersionParser;
