'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _requirefrom = require('requirefrom');

var _requirefrom2 = _interopRequireDefault(_requirefrom);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 12.
 *
 * Looks for:
 *
 * - the kibi:default_dashboard_title in investigate.yml
 *
 * Then:
 *
 * - put kibi:defaultDashboardId value to the advanced settings
 */
class Migration12 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._config = configuration.config;
    this._index = configuration.config.get('kibana.index');
    this._defaultDashboardTitleYml = configuration.config.get('investigate_core.default_dashboard_title');
    this._type = 'config';
    this._query = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'kibi'
            }
          }]
        }
      }
    };
  }
  static get description() {
    return 'Migrate investigate_core:default_dashboard_title property to advanced settings';
  }

  async _fetchDashboards() {
    if (!this._dashboards) {
      this._dashboards = await this.scrollSearch(this._index, 'dashboard');
    }
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let count = 0;
    if (!this._defaultDashboardTitleYml) {
      return count;
    }

    await this._fetchDashboards();
    const dashboardWithTitleFromYmlFound = _lodash2.default.find(this._dashboards, d => d._source.title === this._defaultDashboardTitleYml);

    if (!dashboardWithTitleFromYmlFound) {
      this._logger.warning('[' + this._defaultDashboardTitleYml + '] is set as investigate_core.default_dashboard_title' + ' in investigate.yml but dashboard cannot be found.');
      return count;
    }

    const objects = await this.scrollSearch(this._index, this._type, this._query);
    _lodash2.default.each(objects, object => {
      if (!this._doesDashboardExist(object._source['kibi:defaultDashboardId'])) {
        count++;
      }
    });
    return count;
  }

  _doesDashboardExist(dashboardId) {
    if (!dashboardId) {
      return false;
    }

    const found = _lodash2.default.find(this._dashboards, d => d._id === dashboardId);
    return Boolean(found);
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let upgraded = 0;
    if (!this._defaultDashboardTitleYml) {
      return upgraded;
    }

    let body = '';
    this._logger.info(`Updating investigate_core.default_dashboard_title from config`);

    await this._fetchDashboards();

    let defaultDashboardId;
    const dashboardWithTitleFromYmlFound = _lodash2.default.find(this._dashboards, d => d._source.title === this._defaultDashboardTitleYml);
    if (dashboardWithTitleFromYmlFound) {
      defaultDashboardId = dashboardWithTitleFromYmlFound._id;
    } else {
      this._logger.info(this._defaultDashboardTitleYml + ` dashboard cannot be found.`);
      return upgraded;
    }

    const objects = await this.scrollSearch(this._index, this._type, this._query);
    for (const obj of objects) {
      // check if kibi:defaultDashboardId contains a valid dashboard id
      if (!this._doesDashboardExist(obj._source['kibi:defaultDashboardId'])) {
        body += JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({
          doc: {
            'kibi:defaultDashboardId': defaultDashboardId
          }
        }) + '\n';
        upgraded++;
      }
    }

    if (upgraded > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return upgraded;
  }
}
exports.default = Migration12;
module.exports = exports['default'];
