'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _requirefrom = require('requirefrom');

var _requirefrom2 = _interopRequireDefault(_requirefrom);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 13.
 *
 * Looks for:
 *
 * - the "sourceFiltering" property in index patterns
 *
 * Then:
 *
 * - migrate "sourceFiltering" property to new source filter syntax
 */
class Migration13 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._config = configuration.config;
    this._index = configuration.config.get('kibana.index');
    this._type = 'index-pattern';
    this._query = {
      'query': {
        'exists': { 'field': 'sourceFiltering' }
      }
    };
  }

  static get description() {
    return 'Migrate deprecated "SourceFiltering" property in Index Pattern to "SourceFilters" property';
  }

  async _fetchIndexPatterns() {
    this._indexPatterns = await this.scrollSearch(this._index, this._type, this._query);
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    await this._fetchIndexPatterns();
    if (this._indexPatterns) {
      return this._indexPatterns.length;
    }
    return 0;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let upgraded = 0;
    await this._fetchIndexPatterns();
    if (this._indexPatterns.length === 0) {
      return upgraded;
    }

    let body = '';
    this._logger.info(`Updating old source filter syntax to new source filter syntax`);

    await this._fetchIndexPatterns();

    _lodash2.default.each(this._indexPatterns, indexPattern => {
      if (indexPattern._source.sourceFiltering) {
        const sourceFilteringObject = JSON.parse(indexPattern._source.sourceFiltering);
        const graphBrowserFilter = sourceFilteringObject.kibi_graph_browser;
        const allExclude = sourceFilteringObject.all.exclude;
        const allInclude = sourceFilteringObject.all.include;

        if (graphBrowserFilter) {
          const message = '[ Graph Browser ] property in source filter is deprecated, removing from index pattern';
          this._logger.warning(message);
        }
        if (allInclude) {
          const message = '[ Include ] property in source filter is deprecated, removing from index pattern';
          this._logger.warning(message);
        }

        if (allExclude) {
          const newFilter = [];
          if (_lodash2.default.isArray(allExclude)) {
            _lodash2.default.each(allExclude, function (filter) {
              newFilter.push({
                value: filter
              });
            });
          } else if (_lodash2.default.isString(allExclude)) {
            newFilter.push({
              value: allExclude
            });
          }
          const newFilterString = JSON.stringify(newFilter);
          body += JSON.stringify({
            update: {
              _index: indexPattern._index,
              _type: indexPattern._type,
              _id: indexPattern._id
            }
          }) + '\n' + JSON.stringify({
            doc: {
              'sourceFilters': newFilterString
            }
          }) + '\n';
        }

        body += JSON.stringify({
          update: {
            _index: indexPattern._index,
            _type: indexPattern._type,
            _id: indexPattern._id
          }
        }) + '\n' + JSON.stringify({
          script: 'ctx._source.remove(\'sourceFiltering\')'
        }) + '\n';
        upgraded++;
      }
    });

    if (upgraded > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return upgraded;
  }
}
exports.default = Migration13;
module.exports = exports['default'];
