'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 21.
 *
 * Looks for:
 *
 * - the siren:relations advanced setting inside siren (singleton) config
 *
 * Then:
 *
 * - removes unsupported type field from relations.
 */
class Migration21 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._type = 'config';
    this._server = configuration.server;
    this._query = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'siren'
            }
          }, {
            exists: {
              field: 'siren:relations'
            }
          }]
        }
      }
    };

    this._supportedOptions = new Set();
    this._supportedOptions.add('BROADCAST_JOIN');
    this._supportedOptions.add('HASH_JOIN');
    this._supportedOptions.add('MERGE_JOIN');
  }

  static get description() {
    return 'Remove type field in relations if with an unsupported value';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }
    if (objects.length !== 1) {
      this._logger.error('There should be only one config object');
      return 0;
    }
    if (!objects[0]._source['siren:relations']) {
      return 0;
    }

    const relations = JSON.parse(objects[0]._source['siren:relations']);
    if (this._isUpgradeable(relations)) {
      return 1;
    }

    return 0;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }
    if (objects.length !== 1) {
      this._logger.error('There should be only one config object');
      return 0;
    }
    const obj = objects[0];
    if (!obj._source['siren:relations']) {
      return 0;
    }

    const relations = JSON.parse(obj._source['siren:relations']);
    this._upgradeSirenRelations(relations);

    this._logger.info(`Updating siren:relations from config with _id=${obj._id}`);

    let body = '';
    body += JSON.stringify({
      update: {
        _index: obj._index,
        _type: obj._type,
        _id: obj._id
      }
    }) + '\n' + JSON.stringify({ doc: {
        'siren:relations': JSON.stringify(relations)
      } }) + '\n';

    if (body !== '') {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }

    return 1;
  }

  /**
   * Checks if the siren:relations is upgradeable.
   */
  _isUpgradeable(relations) {
    let check = false;
    if (relations && relations.relationsIndices && relations.relationsIndices.length) {
      _lodash2.default.each(relations.relationsIndices, rel => {
        if (rel.type && !this._supportedOptions.has(rel.type)) {
          check = true;
        }
      });
    }
    return check;
  }

  /**
   * Upgrades the siren:relations
   *
   * @param {Object} relations the siren:relations object to update.
   */
  _upgradeSirenRelations(relations) {
    _lodash2.default.each(relations.relationsIndices, rel => {
      if (rel.type && !this._supportedOptions.has(rel.type)) {
        delete rel.type;
      }
    });
  }

}
exports.default = Migration21;
module.exports = exports['default'];
