'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 24.
 *
 * Looks for:
 *
 * - the sourceFilters object within the kibana.index index-pattern mapping
 *
 * Then:
 *
 * - if not found, adds the sourceFilters mapping object
 */
class Migration24 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._type = 'index-pattern';
    this._server = configuration.server;
    this._defaultSourceFiltersMapping = {
      properties: {
        sourceFilters: {
          type: 'text'
        }
      }
    };
  }

  static get description() {
    return 'Add the sourceFilters mapping type to the mappings if it doesn\'t exist';
  }

  /**
   * Checks if index-pattern mapping contains a property called "sourceFilters".
   */
  _isUpgradeable(mappings) {
    const indexPatternMappingsProperties = mappings['index-pattern'].properties;
    return indexPatternMappingsProperties.sourceFilters ? 0 : 1;
  }

  /**
   * Exposes the default sourceFilters mapping object
   */
  getDefaultSourceFiltersMapping() {
    return this._defaultSourceFiltersMapping;
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    try {
      const indexPatternMapping = await this._client.indices.getMapping({
        index: this._index,
        type: this._type,
        ignoreUnavailable: true,
        allowNoIndices: true
      });
      return this._isUpgradeable(indexPatternMapping[this._index].mappings);
    } catch (e) {
      if (e.status === 404) {
        // no mappings mean that there is no object of type index-pattern in the index
        return 0;
      }
      throw e;
    }
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let indexPatternMapping;
    try {
      indexPatternMapping = await this._client.indices.getMapping({
        index: this._index,
        type: this._type
      });
    } catch (e) {
      if (e.status === 404) {
        // no mappings mean that there is no object of type index-pattern in the index
        return 0;
      }
      throw e;
    }

    const mapping = indexPatternMapping[this._index].mappings;
    if (this._isUpgradeable(mapping) !== 0) {
      mapping['index-pattern'].properties.sourceFilters = this.getDefaultSourceFiltersMapping();

      await this._client.indices.putMapping({
        index: this._index,
        type: this._type,
        body: mapping
      });

      return 1;
    } else {
      return 0;
    }
  }
}
exports.default = Migration24;
module.exports = exports['default'];
